#include "mbed.h"
#include "radio.h"

#include "mbed-client-cli/ns_cmdline.h"
#include "mbed-trace/mbed_trace.h"
#define TRACE_GROUP   "mast"

#if defined(SX127x_H) || defined(SX126x_H)
    #define BW_KHZ              500
    #define SPREADING_FACTOR    11
    #define CF_HZ               910800000
    #if defined(SX126x_H)
        #define TX_DBM              (Radio::chipType == CHIP_TYPE_SX1262 ? 20 : 14) 
    #else
        #define TX_DBM              17
    #endif
#elif defined(SX128x_H)
    #define BW_KHZ              200
    #define SPREADING_FACTOR    11
    #define CF_HZ               2487000000

    #define TX_DBM              5
#endif
 
Timer t;
//InterruptIn UserButtonPB2(PB_2);
DigitalIn b[8] = {PB_2,PB_5,PB_8,PB_9,PB_12,PB_13,PB_14,PB_15};
Thread thread;
// Create a queue that can hold a maximum of 32 events
EventQueue queue(32 * EVENTS_EVENT_SIZE);


volatile bool tx_done;
uint8_t sendCommandFlag = 0;

static uint16_t crc_ccitt( uint8_t *buffer, uint16_t length )
{
    tr_debug(__func__);
    const uint16_t polynom = 0x1021;    // The CRC calculation follows CCITT
    uint16_t crc = 0x0000;              // CRC initial value

    if( buffer == NULL ) return 0;

    for( uint16_t i = 0; i < length; ++i )
    {
        crc ^= ( uint16_t ) buffer[i] << 8;
        for( uint16_t j = 0; j < 8; ++j )
        {
            crc = ( crc & 0x8000 ) ? ( crc << 1 ) ^ polynom : ( crc << 1 );
        }
    }
    return crc;
}

void transmit(unsigned target, uint8_t cmd)
{
    tr_debug(__func__);
    unsigned t_diff;
    uint16_t crc;

    Radio::radio.tx_buf[0] = cmd;
    t_diff = target - t.read_us();
    Radio::radio.tx_buf[1] = t_diff >> 24;
    Radio::radio.tx_buf[2] = t_diff >> 16;
    Radio::radio.tx_buf[3] = t_diff >> 8;
    Radio::radio.tx_buf[4] = t_diff & 0xff;
    crc = crc_ccitt(Radio::radio.tx_buf, 5);
    Radio::radio.tx_buf[5] = crc >> 8;
    Radio::radio.tx_buf[6] = crc & 0xff;
    Radio::Send(7, 0, 0, 0);

    for (tx_done=false;!tx_done;) 
    {
        Radio::service();
    }
    tr_debug("t_diff:%u crc:%04x\r\n", t_diff, crc);
}

#define TARGET_LATENCY      2000000
static void send_alarm(uint8_t cmd)
{
    tr_debug(__func__);
    int i;
    unsigned target = t.read_us() + TARGET_LATENCY;
    tr_debug("send_alarm() %u\n", target);

    for (i = 0; i < 5; i++)
    {
        transmit(target, cmd);
        wait(0.1);
    }
}

void txDoneCB()
{
    tx_done = true;
}

void rxDoneCB(uint8_t size, float Rssi, float Snr)
{
}

const RadioEvents_t rev =
{
    /* Dio0_top_half */     NULL,
    /* TxDone_topHalf */    NULL,
    /* TxDone_botHalf */    txDoneCB,
    /* TxTimeout  */        NULL,
    /* RxDone  */           rxDoneCB,
    /* RxTimeout  */        NULL,
    /* RxError  */          NULL,
    /* FhssChangeChannel  */NULL,
    /* CadDone  */          NULL
};


static Mutex serialOutMutex;
static void serial_out_mutex_wait()
{
    serialOutMutex.lock();
}

static void serial_out_mutex_release()
{
    osStatus s = serialOutMutex.unlock();
    MBED_ASSERT(s == osOK);
}

static int cmd_transmitByte(int argc, char *argv[]) 
{
    tr_debug("%s args: %d",__FUNCTION__,argc);
    if(argc != 2)
    {
        tr_debug("Invalid argument count");
        return CMDLINE_RETCODE_INVALID_PARAMETERS;
    }
    uint8_t byte = atoi(argv[1]);
    tr_debug("sending byte: %d",byte);
    send_alarm(byte);
    return CMDLINE_RETCODE_SUCCESS;
}

static void manual_transmission_queue_callback(void)
{
   uint8_t byte = 0;
   uint8_t value = 0;
   tr_debug("Parallel Port Data Input");
   for (int i = 0; i< 8; i++)
   {
        value = 0;
        value = b[i].read();
        //tr_debug("bit %d, value: %d",i,value);
        value = value << i;
        //tr_debug("value shifted: %d",value);
        byte |= value;
        //tr_debug("byte: %d",byte);
   }
    tr_debug("sending byte: %d",byte);
    send_alarm(byte);
}

static void manual_transmission(void)
{

    queue.call(manual_transmission_queue_callback);
}


   
int main()
{


    mbed_trace_mutex_wait_function_set( serial_out_mutex_wait );
    mbed_trace_mutex_release_function_set( serial_out_mutex_release );
    mbed_trace_init();
    tr_debug(__FUNCTION__);
    tr_debug("Program Entry");

    thread.start(callback(&queue, &EventQueue::dispatch_forever));
    //UserButtonPB2.rise(&manual_transmission);

    wait(0.05);
    t.start();

    Radio::Init(&rev);
    Radio::Standby();
    Radio::LoRaModemConfig(BW_KHZ, SPREADING_FACTOR, 1);
    Radio::LoRaPacketConfig(8, false, true, false);  // preambleLen, fixLen, crcOn, invIQ
    Radio::SetChannel(CF_HZ);
    Radio::set_tx_dbm(TX_DBM);

    tr_debug("Initializing Command Line");
    cmd_init(0);
    cmd_mutex_wait_func( serial_out_mutex_wait );
    cmd_mutex_release_func( serial_out_mutex_release );
    cmd_add("transmit-byte", cmd_transmitByte,"Transmit Byte","Transmit a decimal byte over LoRa radio");


    while(true)
    {

        int c = getchar();
        if (c != EOF)
        {
            cmd_char_input(c);
        }
    }
}
