#include "Module.h"

Serial pl(USBTX, USBRX);
DigitalOut led4(LED4);

Module::Module()
{
    pl.baud(9600);
    
    //objects
    this->daq = new DAQ();
    this->uart = new Uart(this);
    
    //local variables
    this->ID[0] = 0;
    this->ID[1] = 1;
    
    this->IP[0] = 0xff;
    this->IP[1] = 0xff;
    this->IP[2] = 0xff;
    this->IP[3] = 0xff;
         
    this->version[0] = 0;
    this->version[1] = 0;
    
    this->enableData= false;
    this->packageSize = 100;
    this->sampleNumber = 0;
    this->sampleRate = 250; //default waarde - let op, digitale filters moeten herberekend worden bij het veranderen van de sampleRate
    this->dataPacket = NULL;
    
    this->wifiEnabled = false;
    this->creatDataPacket();
}
Module::~Module(void)
{
    delete uart;
    delete daq;
}

//main loop
void Module::mainLoop(void)
{
    this->sendAlivePacket();
}

void Module::sendAlivePacket()
{
    uint8_t *reserved = new uint8_t[2];
    reserved[0] = 0x00;
    reserved[1] = 0x00;
    
    Packet *alive = new Packet(this->ID, PACKET_ALIVE, reserved);
    this->sendPacket(alive); 
}

void Module::meting_thread()
{
    if(enableData && !this->daq->ChannelsDisabled())
    {
        led4 = 1;
        Measurement *m = this->daq->measure();
        if(this->dataPacket->addMeasurement(m))
        {
            this->sampleNumber++;
        }
        else
        {
            //printf("false \n");
            this->sendPacket(this->dataPacket);
            this->creatDataPacket();
        
            this->dataPacket->addMeasurement(m); //hier niet meer controleren of het lukt?
            this->sampleNumber++;
        }
        delete m;
    }
    else
    {
        led4 = 0;
    }
}

//PACKET WORKER
void Module::packetWorker(Packet *p)
{
    switch(p->getCMD())
    {
        //S2C packets
        case PACKET_ACK : this->sendPacket(p); break;
        case PACKET_NACK : this->sendPacket(p); break;
        case PACKET_ERROR : this->sendPacket(p); break;
        case PACKET_ALIVE : this->sendPacket(p); break;
        case PACKET_DATA : this->sendPacket(p); break;
        case PACKET_GET_IP_REPLY : this->sendPacket(p); break;
        
        //C2S packets
        case PACKET_INIT :
            this->sendPacket(p);
            this->HandleInit(p);
            break;
        case PACKET_ENABLE_DATA :
            this->sendPacket(p);
            this->HandleEnableData(p);
            break;
        case PACKET_CHANGE_IP :
            this->sendPacket(p);
            this->HandleChangeIP(p);
            break;
        case PACKET_REMOVE_IP :
            this->sendPacket(p);
            this->removeIP();
            break;
        case PACKET_GET_IP :
            this->sendPacket(p);
            this->IpReply();
            break;
        case PACKET_VERSION :
            this->sendPacket(p);
            this->VersionReply();
            break;
        //case PACKET_VERSION_REPLY : --> zelf versturen!
        //    break;
        case PACKET_SET_PACKAGE_SIZE :
            this->sendPacket(p);
            HandleSetPackageSize(p);
            break;
        case PACKET_GET_SETTINGS :
            this->sendPacket(p);
            this->SettingsReply();
            break;
        //case PACKET_GET_SETTINGS_REPLY : --> moet verstuurd worden!
        //    break;
        case PACKET_SET_ID :
            this->sendPacket(p);
            uint8_t *cmdData = p->getCMD_DATA();
            this->ID[0] = cmdData[0];
            this->ID[1] = cmdData[1];
            delete cmdData;
            this->Acknowledge();
            break;
        case PACKET_TEST_MODE :
            this->sendPacket(p);
            this->uart->send(p); //uitsturen via beide interfaces
            //this->wifi->send(p);
            break;
        case PACKET_LED_TEST_MODE :
            this->sendPacket(p);
            uint8_t *ledData = p->getCMD_DATA();
            uint8_t ledInfo = ledData[0];
            this->LedTest(ledInfo);
            delete ledData;
            break;
        default:
            /// packet dat ik binnen krijg !!!!!
            pl.printf("erorrrrrrr");
            break;
    } 
}

void Module::sendPacket(Packet *p)
{
    if(wifiEnabled)
    {
        //this->wifi->send(p);    
    }
    else
    {
        this->uart->send(p);    
    }    
}
void Module::creatDataPacket()
{
    //printf("MAAK PACKET !! \n");
    uint8_t *reserved = new uint8_t[2];
    reserved[0] = 0x00;
    reserved[1] = 0x00;
    int activeChannels = 0; //Het aantal actieve kanalen
    int nActiveChannels = 0; //Voor te versturen in het datapakket (zie verder)
    
    for(int i = 0; i < 16; i++)
    {
        if(this->daq->ChannelEnabled(i) == true)
        {
            nActiveChannels++;   
            activeChannels |= 1 << (15 - i);
        }    
    }
    this->dataPacket = new Packet(this->ID,this->sampleNumber,this->packageSize,this->sampleRate,activeChannels,nActiveChannels,reserved);  
    delete [] reserved;
}
void Module::HandleInit(Packet *p)
{
    uint8_t cmdSize = p->getCMD_SIZE();
    /* 
    //eventueel controle nog uitvoeren..
    if(cmdSize != 33){
        Packet *error = new Packet();
        this->sendPacket(error);
        delete packet;
        return;    
    }*/
    uint8_t *cmdData = p->getCMD_DATA();
    uint8_t sampleRate = cmdData[0];
    
    uint8_t channelGainByte;
    uint8_t channelEnabledByte;
    
    this->sampleRate = sampleRate;
    
    for(int i = 0; i < 16; i++){
        channelGainByte = cmdData[1 + i * 2];
        channelEnabledByte = cmdData[1 + i * 2 + 1];
        switch(channelGainByte){
        default: //0x00
            this->daq->setChannelGain(i, 1);
            break;
        case 0x01:
            this->daq->setChannelGain(i, 10);
            break;
        case 0x02:
            this->daq->setChannelGain(i, 100);
            break;
        case 0x03:
            this->daq->setChannelGain(i, 1000);
            break;    
        }
        
        if(channelEnabledByte == 0x01){ //enabled
            this->daq->enableChannel(i);
        }
        else{ //disabled
            this->daq->disableChannel(i);
        }
    }
    
    delete cmdData;   
}

void Module::HandleEnableData(Packet *p){
    uint8_t *data = p->getCMD_DATA();
    if(data[0] == 0x01)
    {
        this->EnableData();    
    }
    else
    {
        this->DisableData();
    }    
}

void Module::HandleSetPackageSize(Packet *p){
    uint8_t *data = p->getCMD_DATA();
    this->setPackageSize(data[0]); //data[0] = packageSize!
    delete data;    
}

void Module::HandleChangeIP(Packet *p){
    int cmdSize = p->getCMD_SIZE();
                
    if(cmdSize != 4){
        Packet *errorPacket = new Packet(this->ID);
        this->sendPacket(errorPacket);
        delete errorPacket; 
        return;
    }
    
    uint8_t* data = p->getCMD_DATA();
    this->setIP(data);
    delete data;    
}

void Module::IpReply(){
    uint8_t cmdDataLengte = 4;
    uint8_t *cmdData = new uint8_t[cmdDataLengte];
    uint8_t *reserved = new uint8_t[2];
    
    cmdData[0] = this->IP[0];
    cmdData[1] = this->IP[1];
    cmdData[2] = this->IP[2];
    cmdData[3] = this->IP[3];
    
    reserved[0] = 0x00;
    reserved[1] = 0x00;
    
    Packet *ipInfoPacket = new Packet(this->ID, PACKET_GET_IP_REPLY, cmdData, cmdDataLengte, reserved);
    this->sendPacket(ipInfoPacket);
    
    delete ipInfoPacket;    
    delete reserved;
    delete cmdData;
}

void Module::VersionReply(){
    uint8_t cmdDataLengte = 2;
    uint8_t *cmdData = new uint8_t[cmdDataLengte];
    uint8_t *reserved = new uint8_t[2];
    
    cmdData[0] = this->version[0];
    cmdData[1] = this->version[1];
    
    reserved[0] = 0x00;
    reserved[1] = 0x00;
    
    Packet *versionPacket = new Packet(this->ID, PACKET_VERSION_REPLY, cmdData, cmdDataLengte, reserved);
    this->sendPacket(versionPacket);
    
    delete reserved;
    delete versionPacket;
    delete cmdData;    
}

void Module::SettingsReply(){
    uint8_t cmdDataLengte = 41;
    uint8_t *cmdData = new uint8_t[cmdDataLengte];
    uint8_t *reserved = new uint8_t[2];
    
    cmdData[0] = this->IP[0];
    cmdData[1] = this->IP[1];
    cmdData[2] = this->IP[2];
    cmdData[3] = this->IP[3];
    
    if(enableData){
        cmdData[4] = 0x01;        
    }
    else{
        cmdData[4] = 0x00;
    }
    
    cmdData[5] = this->version[0];
    cmdData[6] = this->version[1];
    
    for(int i = 0; i < 16; i++){
        cmdData[7 + i * 2] = this->daq->getChannelGain(i);
        
        if(this->daq->ChannelEnabled(i) == true){
            cmdData[7 + i * 2 + 1] = 0x01;       
        }
        else{
            cmdData[7 + i * 2 + 1] = 0x00;    
        }
    }    
    
    cmdData[39] = this->sampleRate;
    cmdData[40] = this->packageSize;
    
    reserved[0] = 0x00;
    reserved[1] = 0x00;
    
    Packet *settingsPacket = new Packet(this->ID, PACKET_GET_SETTINGS_REPLY, cmdData, cmdDataLengte, reserved);
    this->sendPacket(settingsPacket);
    
    delete settingsPacket;   
    delete cmdData;
    delete reserved;
}

void Module::LedTest(uint8_t ledInfo){
    for(int i = 4; i < 8; i++){
        if(((ledInfo >> i) & 1) == 1){ //LED moet aan!
            switch(i){
            case 4:
                //LED0 aanzetten
                
                break;
            case 5:
                //LED1 aanzetten
                
                break;
            case 6:
                //LED2 aanzetten
                
                break;
            case 7:
                //LED3 aanzetten
                
                break;
            default:
                break;
            }
        }  
    }   
}

void Module::Acknowledge(uint8_t RESERVED[2])
{
    Packet *ack = new Packet(this->ID, PACKET_ACK, RESERVED);
    this->sendPacket(ack);
    delete ack;
    //delete reserved; //moet weggedaan worden waar deze array aangemaakt wordt!
}

void Module::Acknowledge()
{
    uint8_t *reserved = new uint8_t[2];
    reserved[0] = 0x00;
    reserved[1] = 0x00;
    
    Packet *ack = new Packet(this->ID, PACKET_ACK, reserved);
    this->sendPacket(ack);
    
    delete ack;
    delete reserved;
}


//GETTERS
Uart* Module::getUart(void)
{
    return(this->uart);
}
uint8_t* Module::getID(void)
{
    return(this->ID);
}
uint8_t* Module::getIP(void)
{
    return(this->IP);
}
uint8_t* Module::getVersion(void)
{
    return(this->version);
}
bool Module::dataEnabled(void)
{
    return(this->enableData);
}
uint8_t Module::getPackageSize(void)
{
    return(this->packageSize);
}
unsigned int Module::getSampleNumber(void)
{
    return(this->sampleNumber);
}
uint8_t Module::getSampleRate(void)
{
    return(this->sampleRate);
}
    
//SETTERS
void Module::setID(uint8_t ID[2])
{
    this->ID[0] = ID[0];
    this->ID[1] = ID[1];
}
void Module::setIP(uint8_t IP[4])
{
    this->IP[0] = IP[0];
    this->IP[1] = IP[1];
    this->IP[2] = IP[2];
    this->IP[3] = IP[3];
}
void Module::removeIP(){
    this->IP[0] = 0xff;
    this->IP[1] = 0xff;
    this->IP[2] = 0xff;
    this->IP[3] = 0xff;    
}
void Module::setVersion(uint8_t version[2])
{
    this->version[0] = version[0];
    this->version[1] = version[1];
}
void Module::EnableData(void)
{
    this->enableData = true;
}
void Module::DisableData(void)
{
    this->enableData = false;
}
void Module::setPackageSize(uint8_t packageSize)
{
    this->packageSize = packageSize;
}
void Module::setSampleNumber(unsigned int sampleNumber)
{
    this->sampleNumber = sampleNumber;
}
void Module::setSampleRate(uint8_t sampleRate)
{
    this->sampleRate = sampleRate;
}
