/*
  ______                              _
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2016 Semtech

Description: PingPong, PER and Ranging demo implementation.

Maintainer: Gregory Cristian & Gilbert Menth
*/

#include <cstdlib>

#include "mbed.h"
#include <math.h>
#include "radio.h"
#include "sx1280-hal.h"
#include "Eeprom.h"
#include "DemoApplication.h"
#include "FreqLUT.h"
#include "Menu.h"

//#include "RangingCorrection.h"

#define max(a,b) \
   ({ __typeof__ (a) _a = (a); \
       __typeof__ (b) _b = (b); \
     _a > _b ? _a : _b; })
   
double t0 =       -0.016432807883697;                         // X0
double t1 =       0.323147003165358;                          // X1
double t2 =       0.014922061351196;                          // X1^2
double t3 =       0.000137832006285;                          // X1^3
double t4 =       0.536873856625399;                          // X2
double t5 =       0.040890089178579;                          // X2^2
double t6 =       -0.001074801048732;                         // X2^3
double t7 =       0.000009240142234;                          // X2^4


double p[8] = { 0,
                -4.1e-9,
                1.03e-7,
                1.971e-5,
                -0.00107,
                0.018757,
                0.869171,
                3.072450 };

/*!
 * \brief Defines the local payload buffer size
 */
#define BUFFER_SIZE                     255

/*!
 * \brief Defines the size of the token defining message type in the payload
 *        cf. above.
 */
#define PINGPONG_SIZE                   4
#define PER_SIZE                        3

/*!
 * \brief Define time used in PingPong demo to synch with cycle
 * RX_TX_INTER_PACKET_DELAY is the free time between each cycle (time reserve)
 */
#define RX_TX_INTER_PACKET_DELAY        150  // ms
#define RX_TX_TRANSITION_WAIT           15   // ms

/*!
 * \brief Size of ticks (used for Tx and Rx timeout)
 */
#define RX_TIMEOUT_TICK_SIZE            RADIO_TICK_SIZE_1000_US

#define RNG_TIMER_MS                    384 // ms
#define RNG_COM_TIMEOUT                 100 // ms

/*!
 * \brief Ranging raw factors
 *                                  SF5     SF6     SF7     SF8     SF9     SF10
 */
const uint16_t RNG_CALIB_0400[] = { 10299,  10271,  10244,  10242,  10230,  10246  };
const uint16_t RNG_CALIB_0800[] = { 11486,  11474,  11453,  11426,  11417,  11401  };
const uint16_t RNG_CALIB_1600[] = { 13308,  13493,  13528,  13515,  13430,  13376  };
const double   RNG_FGRAD_0400[] = { -0.148, -0.214, -0.419, -0.853, -1.686, -3.423 };
const double   RNG_FGRAD_0800[] = { -0.041, -0.811, -0.218, -0.429, -0.853, -1.737 };
const double   RNG_FGRAD_1600[] = { 0.103,  -0.041, -0.101, -0.211, -0.424, -0.87  };

/*!
 * \brief Define the possible message type for the Ping-Pong and PER apps
 */
//const uint8_t PingMsg[] = "PING";
//const uint8_t PongMsg[] = "PONG";
//const uint8_t PerMsg[]  = "PER";

/*!
 * \brief Buffer and its size
 */
uint8_t BufferSize = BUFFER_SIZE;
uint8_t Buffer[BUFFER_SIZE];

static uint8_t CurrentChannel;
static uint16_t MeasuredChannels;
int RngResultIndex;
double RawRngResults[DEMO_RNG_CHANNELS_COUNT_MAX];
//double RawRngResults[255];
double RssiRng[DEMO_RNG_CHANNELS_COUNT_MAX];
//double RssiRng[255];

//
//#define cCircularBufferWidth    32      // LA:  Dimensione del Buffer Circolare 
//#define cCircularBufferWidth    8      // LA:  Dimensione del Buffer Circolare
//#define cMaxActivePartners      4       // LA:  Numero Massimo di partners gestibili    (Saranno circa 50, nella versione definitiva)

//
const int ciAnsweringTimeout_s =    6;      // LA:  Secondi di Inattività per cancellare un Partner dalla lista dei Rispondenti
const int ciActivityTimeout_s = 16;         // LA:  Secondi di Inattività per cancellare un Partner dalla lista dei Richiedenti
const int ciBlackListTime_s =   4;          // LA:  Secondi di Oscuramento del partner dopo un ranging

ActivePartner_t ActivePartnersList [ciMaxActivePartners];           // LA:  Numero Massimo di Partners "sostenibili"
uint8_t ui8APIndex;                                                 // LA:  Indice Generico di Partner (Uso Globale)
uint8_t ui8APEmptyItemIndex;                                        // LA:  Indice della Prima Cella Libera x Aggiungere una Entry (Uso Globale)

AnsweringPartner_t AnsweringPartnersList [ciMaxActivePartners];     // LA:  Numero Massimo di Partners Rispondenti "sostenibili"
uint8_t ui8AnsweringPIndex;                                         // LA:  Indice Generico di Partner Rispondente (Uso Globale)
uint8_t ui8AnsweringPEmptyItemIndex;                                // LA:  Indice della Prima Cella Libera x Aggiungere una Entry di Partner Rispondente (Uso Globale)

// LA:  Buffer Circolare (Solo il Dispositivo Stesso)
//
double  dAverageDistanceCircularBuffer [ciCircularBufferWidth];     // LA:  Media Calcolata x la Distanza di qst Partner
uint8_t ui8CBIndex = 0;                                             // LA:  Indice Nel Buffer Circolare
uint8_t ui8CB1stCountsIndex = 0;                                    // LA:  Indice Parziale dei Conteggi

uint8_t CRC8(const uint8_t *data, int length);

//
//  ============================================================================
//  ============================================================================


Ticker  tTimeInTx;              // LA:  Aggiunta, Tempo concesso alla Trasmissione
bool    bTimeInTx = false;      // LA:  Aggiunta, True quando siamo in tempo concesso alla Tx, False a tempo Spirato.
Ticker  tTimeInRx;              // LA:  Aggiunta, Tempo concesso alla ricezione dopo una Richiesta di Ranging evasa.
bool    bTimeInRx = false;      // LA:  Aggiunta, True quando siamo in tempo concesso alla Rx, False a tempo Spirato.
//
void    pRefreshRxTimeout (void);   // LA:  Rinfresca il TimeOut a seguito di un'evento Pollato di Ricezione
void    pTimeInTxDone   (void);     // LA:  Allo scatenarsi del Ticker fa terminare il tempo concesso alla Trasmissione
void    pRefreshTxTimeout (void);   // LA:  Rinfresca il TimeOut a seguito di un'evento Pollato di Trasmissione
void    pTimeInRxDone   (void);     // LA:  Allo scatenarsi del Ticker fa terminare il tempo concesso alla Ricezione

/*!
 * \brief Function to be executed on Radio Tx Done event
 */
void OnTxDone( void );

/*!
 * \brief Function to be executed on Radio Rx Done event
 */
void OnRxDone( void );

/*!
 * \brief Function executed on Radio Tx Timeout event
 */
void OnTxTimeout( void );

/*!
 * \brief Function executed on Radio Rx Timeout event
 */
void OnRxTimeout( void );

/*!
 * \brief Function executed on Radio Rx Error event
 */
void OnRxError( IrqErrorCode_t );

/*!
 * \brief Function executed on Radio Rx Error event
 */
void OnRangingDone( IrqRangingCode_t );

/*!
 * \brief All the callbacks are stored in a structure
 */
RadioCallbacks_t Callbacks =
{
    &OnTxDone,        // txDone
    &OnRxDone,        // rxDone
    NULL,             // syncWordDone
    NULL,             // headerDone
    &OnTxTimeout,     // txTimeout
    &OnRxTimeout,     // rxTimeout
    &OnRxError,       // rxError
    &OnRangingDone,   // rangingDone
    NULL,             // cadDone
};

/*!
 * \brief Define IO and callbacks for radio
 * mosi, miso, sclk, nss, busy, dio1, dio2, dio3, rst, callbacks
 */
SX1280Hal Radio( D11, D12, D13, D7, D3, D5, NC, NC, A0, &Callbacks );

/*!
 * \brief Control the Antenna Diversity switch
 */
DigitalOut ANT_SW( A3 );

/*!
 * \brief Tx LED toggling on transmition success
 */
DigitalOut TX_LED( A4 );

/*!
 * \brief Rx LED toggling on reception success
 */
DigitalOut RX_LED( A5 );

/*!
 * \brief Mask of IRQs
 */
uint16_t IrqMask = 0x0000;

/*!
 * \brief Locals parameters and status for radio API
 * NEED TO BE OPTIMIZED, COPY OF STUCTURE ALREADY EXISTING
 */
PacketParams_t PacketParams;
PacketStatus_t PacketStatus;
ModulationParams_t ModulationParams;

/*!
 * \brief Flag to indicate if the demo is already running
 */
static bool DemoRunning = false;

/*!
 * \brief Flag holding the current internal state of the demo application
 */
static uint8_t DemoInternalState = APP_IDLE;

/*!
 * \brief Ticker for master to synch Tx frames. Flags for PER and PingPong demo
 * for Synch TX in cycle.
 */
Ticker SendNextPacket;
static bool SendNext = false;

/*!
 * \brief Ticker for slave to synch Tx frames. Flags for PER and PingPong demo
 * for Synch RX in cycle.
 */
Ticker ReceiveNextPacket;
static bool ReceiveNext = false;

/*!
 * \brief Hold last Rx packet number to compute PER in PER and PingPong demo
 */
//static uint32_t PacketRxSequence = 0;
//static uint32_t PacketRxSequencePrev = 0;

void SetAntennaSwitch( void );
void LedBlink( void );
void InitializeDemoParameters( uint8_t modulation );
//uint16_t GetTimeOnAir( uint8_t modulation );
void SendNextPacketEvent( void );
void ReceiveNextPacketEvent( void );
uint8_t CheckDistance( void );

// ************************        Utils            ****************************
// *                                                                           *
// *                                                                           *
// *                                                                           *
// *****************************************************************************

void InitDemoApplication( void )
{
//    RX_LED = 1;
//    TX_LED = 1;

    SetAntennaSwitch( );

    wait_ms( 500 ); // wait for on board DC/DC start-up time

    Radio.Init( );

    // Can also be set in LDO mode but consume more power
    Radio.SetRegulatorMode( ( RadioRegulatorModes_t )Eeprom.EepromData.DemoSettings.RadioPowerMode );
    Radio.SetStandby( STDBY_RC );

    memset( &Buffer, 0x00, BufferSize );

//    RX_LED = 0;
//    TX_LED = 0;

//    PacketRxSequence = 0;
//    PacketRxSequencePrev = 0;
    Eeprom.EepromData.DemoSettings.CntPacketTx    = 0;
    Eeprom.EepromData.DemoSettings.CntPacketRxOK  = 0;
    Eeprom.EepromData.DemoSettings.CntPacketRxKO  = 0;
    Eeprom.EepromData.DemoSettings.RxTimeOutCount = 0;
}

// LA:  ========================================================================
// LA:  ========================================================================
// LA:  ========================================================================
// LA:  ========================================================================

uint8_t RunDemoApplicationRangingMaster (void) {
static uint32_t DeviceRangingInquirerAddress;
static uint32_t DeviceRangingAnswererAddress;
static bool     DeviceAnswered;

//    bool    refreshDisplay =    false;

    // LA:  Inizializza l'HW Radio se la Demo parte ora
    //      ===========================================
    //
    //  Questa parte viene eseguita SOLO una volta, ad ogni avvio della DEMO
    //
    if  (!DemoRunning) {
        DemoRunning = true;
        DeviceAnswered = false;

        ANT_SW = 1;

        Eeprom.EepromData.DemoSettings.CntPacketTx = 0;
        Eeprom.EepromData.DemoSettings.RngFei      = 0.0;
        Eeprom.EepromData.DemoSettings.RngStatus   = RNG_INIT;
        InitializeDemoParameters( Eeprom.EepromData.DemoSettings.ModulationType );

        // LA:  Inizilaizza i Parametri come "MASTER"
        //      =====================================
        //
        Eeprom.EepromData.DemoSettings.TimeOnAir = RX_TX_INTER_PACKET_DELAY;

        Radio.SetDioIrqParams   (IRQ_RX_DONE| IRQ_TX_DONE| IRQ_RX_TX_TIMEOUT| IRQ_RANGING_MASTER_RESULT_VALID| IRQ_RANGING_MASTER_TIMEOUT,      // irqMask
                                IRQ_RX_DONE| IRQ_TX_DONE| IRQ_RX_TX_TIMEOUT| IRQ_RANGING_MASTER_RESULT_VALID| IRQ_RANGING_MASTER_TIMEOUT,       // dio1Mask
                                IRQ_RADIO_NONE,                                                                                                 // dio2Mask
                                IRQ_RADIO_NONE);                                                                                                // dio3Mask

        DemoInternalState = APP_RANGING_CONFIG;                 // LA:  Termina la Configurazione dentro al "Main Case"
        pRefreshTxTimeout ();                                   // LA:  Avvio il Timeout Esterno d'Ascolto
    }

    // LA:  Attenzione, la gestione del chip è "PollingMode"
    //      ================================================
    //
    //  Per elaborare le Interruzioni occorre richiamare, ciclicamente, "Radio.ProcessIrqs( );"
    //
    Radio.ProcessIrqs( );

    // LA:  Caso "MASTER"
    //      =============
    //
    switch  (DemoInternalState) {
        case    APP_RANGING_CONFIG:

            Eeprom.EepromData.DemoSettings.RngStatus =  RNG_INIT;   // LA:  Stato = Inizializzazione Ranging in Corso
                                                                    //
                                                                    //  =================================================================================
                                                                    //  =================================================================================
                                                                    //  Questo identifica il Primo Pacchetto che verrà ricevuto dopo il primo SendPayload
                                                                    //  =================================================================================
                                                                    //  =================================================================================
            Eeprom.EepromData.DemoSettings.CntPacketTx++;

            // LA:  Estrazione della parametrizzazione dalla Flash
            //
            ModulationParams.PacketType =   PACKET_TYPE_LORA;       // LA:  In Ranging la Modulazione è solo  LORA
            ModulationParams.Params.LoRa.SpreadingFactor =  Eeprom.EepromData.ModulationParams.Params.LoRa.SpreadingFactor;
            //memcpy(&(ModulationParams.Params.LoRa.SpreadingFactor), Eeprom.Buffer+ MOD_RNG_SPREADF_EEPROM_ADDR, 1);
            ModulationParams.Params.LoRa.Bandwidth =    Eeprom.EepromData.ModulationParams.Params.LoRa.Bandwidth;
            //memcpy(&(ModulationParams.Params.LoRa.Bandwidth), Eeprom.Buffer+ MOD_RNG_BW_EEPROM_ADDR, 1);
            ModulationParams.Params.LoRa.CodingRate =   Eeprom.EepromData.ModulationParams.Params.LoRa.CodingRate;
            //memcpy(&(ModulationParams.Params.LoRa.CodingRate), Eeprom.Buffer+ MOD_RNG_CODERATE_EEPROM_ADDR, 1);
            //
            PacketParams.PacketType =   PACKET_TYPE_LORA;           // LA:  In Ranging la Modulazione è solo  LORA
            PacketParams.Params.LoRa.PreambleLength =   Eeprom.EepromData.PacketParams.Params.LoRa.PreambleLength;
            //memcpy(&(PacketParams.Params.LoRa.PreambleLength), Eeprom.Buffer+ PAK_RNG_PREAMBLE_LEN_EEPROM_ADDR, 1);
            PacketParams.Params.LoRa.HeaderType =   Eeprom.EepromData.PacketParams.Params.LoRa.HeaderType;
            //memcpy(&(PacketParams.Params.LoRa.HeaderType), Eeprom.Buffer+ PAK_RNG_HEADERTYPE_EEPROM_ADDR, 1);
            PacketParams.Params.LoRa.PayloadLength =    9;                      // LA:  In Ranging il Payload Atteso per Default è di 7 bytes, +1 x Lunghezza, +1 x CheckSum
            PacketParams.Params.LoRa.Crc =  Eeprom.EepromData.PacketParams.Params.LoRa.Crc;
            //memcpy(&(PacketParams.Params.LoRa.Crc), Eeprom.Buffer+ PAK_RNG_CRC_MODE_EEPROM_ADDR, 1);
            PacketParams.Params.LoRa.InvertIQ = Eeprom.EepromData.PacketParams.Params.LoRa.InvertIQ;
            //memcpy(&(PacketParams.Params.LoRa.InvertIQ), Eeprom.Buffer+ PAK_RNG_IQ_INV_EEPROM_ADDR, 1);

            Radio.SetPacketType(ModulationParams.PacketType );                  // LA:  Imposta il tipo di Pacchetto
            Radio.SetModulationParams(&ModulationParams );                      // LA:  Imposta i Parametri di Modulazione
            Radio.SetPacketParams(&PacketParams );                              // LA:  Imposta i Parametri relativi al Pacchetto
            Radio.SetRfFrequency(Eeprom.EepromData.DemoSettings.Frequency );    // LA:  Imposta la Frequenza di Lavoro

            Eeprom.EepromData.DemoSettings.CntPacketRxOK = 0;           // LA:  Azzera il Contatore dei Pacchetti Ricevuti
            Eeprom.EepromData.DemoSettings.CntPacketRxOKSlave = 0;      // 
            
            MeasuredChannels =  0;
//            CurrentChannel    = 0;
            CurrentChannel =    (std::rand()/ ((RAND_MAX + 1u)/ 40));   // LA: Randomized, 0..39

            // LA:  Preparazione del Buffer col PayLoad
            //      ===================================
            //
            //  0 <=    Indirizzo HW di questo Dispositivo, x---
            //  1 <=    Indirizzo HW di questo Dispositivo, -x--
            //  2 <=    Indirizzo HW di questo Dispositivo, --x-
            //  3 <=    Indirizzo HW di questo Dispositivo, ---x
            //
            //  4 <=    Canale in Uso       (Random 0..39)
            //  5 <=    Modalità d'Antenna  (0, 1 o Entrambe)
            //  6 <=    Numero di Pacchetti
            //
            //  7 <=    Lunghezza del Buffer Trasmesso (9)
            //  8 <=    CRC8 del Buffer[0..7]
            //
            Buffer[0] = ( Eeprom.EepromData.DemoSettings.RngAddress >> 24 ) & 0xFF;
            Buffer[1] = ( Eeprom.EepromData.DemoSettings.RngAddress >> 16 ) & 0xFF;
            Buffer[2] = ( Eeprom.EepromData.DemoSettings.RngAddress >>  8 ) & 0xFF;
            Buffer[3] = ( Eeprom.EepromData.DemoSettings.RngAddress & 0xFF );
            Buffer[4] = CurrentChannel;                                     // set the first channel to use
            Buffer[5] = Eeprom.EepromData.DemoSettings.RngAntenna;          // set the antenna strategy
            Buffer[6] = Eeprom.EepromData.DemoSettings.RngRequestCount;     // set the number of hops
            //
            Buffer[7] = PacketParams.Params.LoRa.PayloadLength;             // LA:  Numero di Bytes Totali
            Buffer[8] = CRC8    (Buffer,
                                (PacketParams.Params.LoRa.PayloadLength- 1));

            // LA:  Invio del Payload:
            //
            Radio.SendPayload   (Buffer,
                                PacketParams.Params.LoRa.PayloadLength,
                                (TickTime_t){ RX_TIMEOUT_TICK_SIZE, RNG_COM_TIMEOUT });

            // LA:  Inizializzazione Terminata
            //      ==========================
            //
            //  Lo Stato passa ad "APP_IDLE" in attesa di un interrupt di ricezione o del TimeOut
            //
            DemoInternalState = APP_IDLE;
            break;

        // LA:  Evento di Interruzione mutuato dal Basso Livello,
        // LA:  Evento di Fine Trasmissione. ====================
        //      =================================================
        //
        case APP_TX:
            if( Eeprom.EepromData.DemoSettings.RngStatus == RNG_INIT ) {

                // LA:  Trasmissione terminata del Pacchetto di Ranging (Configurazione)
                //      ================================================================
                //
                //  Imposta il "Polling" di Ricezione
                //
                Radio.SetRx ((TickTime_t) {RX_TIMEOUT_TICK_SIZE, RNG_COM_TIMEOUT});
                DemoInternalState = APP_IDLE;
            }
            else {

                // LA:  Qualcosa è andato storto
                //      ========================
                //
                //  Questo caso è piuttosto improbabile,
                //  ad ogni modo, Chiudo ed Esco.
                //
//                refreshDisplay =    true;
                bTimeInTx = false;              // LA:  Termina la Sessione di Trasmissione
            }
            break;

        // LA:  Evento di Interruzione mutuato dal Basso Livello,
        // LA:  Evento di Fine Ricezione. =======================
        //      =================================================
        //
        case APP_RX:
            if  (Eeprom.EepromData.DemoSettings.RngStatus == RNG_INIT) {

                Radio.GetPayload    (Buffer,
                                    &BufferSize,
                                    BUFFER_SIZE);
                                    
                //  00 =>   Indirizzo HW del Dispositivo che ha richiesto il Ranging (deve essere questo Dispositivo), x---
                //  01 =>   Indirizzo HW del Dispositivo che ha richiesto il Ranging (deve essere questo Dispositivo), -x--
                //  02 =>   Indirizzo HW del Dispositivo che ha richiesto il Ranging (deve essere questo Dispositivo), --x-
                //  03 =>   Indirizzo HW del Dispositivo che ha richiesto il Ranging (deve essere questo Dispositivo), ---x
                //
                //  04 <=   Frequency Error Indicator x il Ranging (RngFei), x---
                //  05 <=   Frequency Error Indicator x il Ranging (RngFei), -x--
                //  06 <=   Frequency Error Indicator x il Ranging (RngFei), --x-
                //  07 <=   Frequency Error Indicator x il Ranging (RngFei), ---x
                //  08 <=   RssiValue
                //
                //  09 <=   Indirizzo HW del Dispositivo che ha risposto al Ranging, x---
                //  10 <=   Indirizzo HW del Dispositivo che ha risposto al Ranging, -x--
                //  11 <=   Indirizzo HW del Dispositivo che ha risposto al Ranging, --x-
                //  12 <=   Indirizzo HW del Dispositivo che ha risposto al Ranging, ---x
                //
                //  13 <=   Lunghezza del Buffer Trasmesso (15)
                //  14 <=   CRC8 del Buffer[0..13]

                // LA:  PayLoad di Inizializzazione Ricevuto:
                //       =====================================
                //
                //  Il contenuto è consistente (Lunghezza/Crc)?
                //
                if  ((Buffer[13] == 15) &&
                    (Buffer[14] == CRC8 (Buffer,
                                        (15- 1)))) {

                    DeviceRangingInquirerAddress =   (uint32_t (Buffer[0]))<< 24;
                    DeviceRangingInquirerAddress +=  (uint32_t (Buffer[1]))<< 16;
                    DeviceRangingInquirerAddress +=  (uint32_t (Buffer[2]))<< 8;
                    DeviceRangingInquirerAddress +=  (uint32_t (Buffer[3]));

                    //  Il contenuto è consistente (Lunghezza/Crc),
                    //
                    //  La richiesta veniva da me?
                    //
                    if  (DeviceRangingInquirerAddress == Eeprom.EepromData.DemoSettings.RngAddress) {
                        
                        //  Il contenuto è consistente (Lunghezza/Crc),
                        //  La richiesta veniva da me,
                        //
                        // LA:  Posso estrarre l'Indirizzo del Partner che ha risposto al Ranging:
                        //      ==================================================================
                        //
                        DeviceRangingAnswererAddress =   (uint32_t (Buffer[9]))<< 24;
                        DeviceRangingAnswererAddress +=  (uint32_t (Buffer[10]))<< 16;
                        DeviceRangingAnswererAddress +=  (uint32_t (Buffer[11]))<< 8;
                        DeviceRangingAnswererAddress +=  (uint32_t (Buffer[12]));

                        //  Verifico che l'indirizzo del Partner che ha risposto al Ranging (DeviceRangingAnswererAddress) sia presente in Tabella.
                        //  Se nn è presente lo aggiungo e morta lì.
                        //  Se è presente, aggiorno il suo Timeout.
                        //  Scansionando la tabella verifico anche se ci sono posti vuoti, riportandone l'ultimo indice trovato in "ui8AnsweringPEmptyItemIndex"
                        //  (Attenzione, se i posti vuoti sono esauriti, "ui8AnsweringPEmptyItemIndex" ritorna "ciMaxActivePartners").
                        //
                        ui8AnsweringPEmptyItemIndex = ciMaxActivePartners;
                        for (ui8AnsweringPIndex = 0; ui8AnsweringPIndex < ciMaxActivePartners; ui8AnsweringPIndex++) {

                            if  (AnsweringPartnersList[ui8AnsweringPIndex].ui32Address == 0)                            ui8AnsweringPEmptyItemIndex = ui8AnsweringPIndex;
                            if  (AnsweringPartnersList[ui8AnsweringPIndex].ui32Address == DeviceRangingAnswererAddress) break;
                        }
                        if  (ui8AnsweringPIndex == ciMaxActivePartners) {                                       // LA:  Non Trovato, se possibile creo nuovo indice su "ui8AnsweringPEmptyItemIndex"
                            if  (ui8AnsweringPEmptyItemIndex < ciMaxActivePartners) {                               // LA:  Slot libero in "ui8AnsweringPEmptyItemIndex"
                        
                                // LA:  Aggiunta di un Nuovo Partner Rispondente
                                //      ========================================
                                //
                                ui8AnsweringPIndex =    ui8AnsweringPEmptyItemIndex;                                        // LA:  Punta all'Indice Vuoto (In Creazione)
                                AnsweringPartnersList[ui8AnsweringPIndex].ui32Address =     DeviceRangingAnswererAddress;   //

                                //  Svuoto il Buffer Circolare del Nuovo "Partner Rispondente"
                                //  Reinizializzo i Puntatori a qst specifico BC
                                //
                                for (AnsweringPartnersList[ui8AnsweringPIndex].ui8CBIndex = 0; AnsweringPartnersList[ui8AnsweringPIndex].ui8CBIndex < ciCircularBufferWidth; AnsweringPartnersList[ui8AnsweringPIndex].ui8CBIndex++) {
                                    AnsweringPartnersList[ui8AnsweringPIndex].dAverageDistanceCircularBuffer[AnsweringPartnersList[ui8AnsweringPIndex].ui8CBIndex] = 0;
                                }
                                AnsweringPartnersList[ui8AnsweringPIndex].ui8CBIndex = 0;                           // LA:  Indice Nel Buffer Circolare
                                AnsweringPartnersList[ui8AnsweringPIndex].ui8CB1stCountsIndex = 0;                  // LA:  Indice Parziale dei Conteggi
                                AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance =   18.5;   // LA:  Out of Bounds
                            }
                        }
                        //
                        // LA:  In Uscita da Qui, se "ui8AnsweringPIndex" < "ciMaxActivePartners" posso memorizzare il Ranging, appena completata la ricezione degli HOP
                        // LA:  ========================================================================================================================================
                        // LA:  ========================================================================================================================================

                        if  (ui8AnsweringPIndex < ciMaxActivePartners) {                                        // LA:  Trovato, indice in "ui8APIndex"
                            AnsweringPartnersList[ui8AnsweringPIndex].ui8ActivityTimeOut = ciAnsweringTimeout_s;    // LA:  Rinfresco (Comunque, ORA) il TimeOut
                            DeviceAnswered = true;                                                                  //      Posso elaborare la Risposta, ui8AnsweringPIndex è Valido
    
                            // LA:  Entry Presente o Creata, Posso memorizzare il Ranging (appena ce l'ho ...)
                            //      ===========================================================================
                            //
                            //  Procedo col Parsing degli HOP(s)
                            //
                            Eeprom.EepromData.DemoSettings.RxTimeOutCount = 0;          //
                            Eeprom.EepromData.DemoSettings.RngStatus =  RNG_PROCESS;    // LA:  Stato Attuale ->    Elaborazione del I° Payload
                            //
                            Eeprom.EepromData.DemoSettings.RngFei = (double)    (((int32_t)Buffer[4]<< 24 )| 
                                                                                ((int32_t)Buffer[5]<< 16 )| 
                                                                                ((int32_t)Buffer[6]<< 8 )| 
                                                                                Buffer[7]);
                            //
                            Eeprom.EepromData.DemoSettings.RssiValue = Buffer[8];       // for ranging post-traitment (since V3 only)
        
                            ModulationParams.PacketType = PACKET_TYPE_RANGING;
                            ModulationParams.Params.LoRa.SpreadingFactor =  (RadioLoRaSpreadingFactors_t) Eeprom.EepromData.DemoSettings.ModulationParam1;
                            //memcpy(&(ModulationParams.Params.LoRa.SpreadingFactor), Eeprom.Buffer + MOD_RNG_SPREADF_EEPROM_ADDR, 1);
                            ModulationParams.Params.LoRa.Bandwidth =    (RadioLoRaBandwidths_t) Eeprom.EepromData.DemoSettings.ModulationParam2;
                            //memcpy(&(ModulationParams.Params.LoRa.Bandwidth), Eeprom.Buffer + MOD_RNG_BW_EEPROM_ADDR, 1);
                            ModulationParams.Params.LoRa.CodingRate =   (RadioLoRaCodingRates_t) Eeprom.EepromData.DemoSettings.ModulationParam3;
                            //memcpy(&(ModulationParams.Params.LoRa.CodingRate), Eeprom.Buffer + MOD_RNG_CODERATE_EEPROM_ADDR, 1);

                            PacketParams.PacketType     = PACKET_TYPE_RANGING;
                            PacketParams.Params.LoRa.PreambleLength =   (uint8_t) Eeprom.EepromData.DemoSettings.PacketParam1;
                            //memcpy(&(PacketParams.Params.LoRa.PreambleLength), Eeprom.Buffer + PAK_RNG_PREAMBLE_LEN_EEPROM_ADDR, 1);
                            PacketParams.Params.LoRa.HeaderType =   (RadioLoRaPacketLengthsModes_t) Eeprom.EepromData.DemoSettings.PacketParam2;
                            //memcpy(&(PacketParams.Params.LoRa.HeaderType), Eeprom.Buffer + PAK_RNG_HEADERTYPE_EEPROM_ADDR, 1);
                            PacketParams.Params.LoRa.PayloadLength =    10;
                            PacketParams.Params.LoRa.Crc =  (RadioLoRaCrcModes_t) Eeprom.EepromData.DemoSettings.PacketParam4;
                            //memcpy(&(PacketParams.Params.LoRa.Crc), Eeprom.Buffer + PAK_RNG_CRC_MODE_EEPROM_ADDR, 1);
                            PacketParams.Params.LoRa.InvertIQ = (RadioLoRaIQModes_t) Eeprom.EepromData.DemoSettings.PacketParam5;
                            //memcpy(&(PacketParams.Params.LoRa.InvertIQ), Eeprom.Buffer + PAK_RNG_IQ_INV_EEPROM_ADDR, 1);

                            // LA:  Configura la Radio x il sucessivo treno di pacchetti
                            //
                            Radio.SetPacketType(ModulationParams.PacketType);
                            Radio.SetModulationParams(&ModulationParams);
                            Radio.SetPacketParams(&PacketParams);
                            Radio.SetRangingRequestAddress(Eeprom.EepromData.DemoSettings.RngAddress);              // LA:  Attenzione Qui !!
                            Radio.SetRangingCalibration(Eeprom.EepromData.DemoSettings.RngCalib);
        //                    Radio.SetTxParams(Eeprom.EepromData.DemoSettings.TxPower, RADIO_RAMP_20_US);
                            Radio.SetTxParams(Eeprom.EepromData.DemoSettings.TxPower, RADIO_RAMP_02_US);
        
                            MeasuredChannels =  0;
                            RngResultIndex =    0;
        
                            // LA:  Crea il Ticker di invio dei pacchetti
                            //
                            SendNextPacket.attach_us    (&SendNextPacketEvent,
                                                        Eeprom.EepromData.DemoSettings.RngReqDelay* 1000);
        
                            // LA:  Il prossimo Stato è "Ranging in Corso"
                            //
                            DemoInternalState = APP_RNG;
                        }
                        else {
                        
                            // LA:  La Lista dei Partners è Piena.
                            //      ==============================
                            //
                            //  Mollo tutto, Chiudo ed Esco.
                            //
                            bTimeInTx = false;              // LA:  Termina la Sessione di Trasmissione
                        }
                    }
                    else {
                    
                        // LA:  Non ho richiesto io questo Ranging.
                        //      ===================================
                        //
                        //  Mollo tutto, Chiudo ed Esco.
                        //
                        bTimeInTx = false;              // LA:  Termina la Sessione di Trasmissione
                    }
                }            
                // LA:  Il contenuto del PayLoad è Inconsistente.
                //      =========================================
                //
                //  Mollo tutto, Chiudo ed Esco.
                //
                else {
                    bTimeInTx = false;              // LA:  Termina la Sessione di Trasmissione
                }
            }
            else {

                // LA:  La Ricezione NON CONSISTE nel Payload di Inizializzazione
                //      =========================================================
                //
                //  Questo caso è piuttosto improbabile,
                //  ad ogni modo, Chiudo ed Esco.
                //
                bTimeInTx = false;              // LA:  Termina la Sessione di Trasmissione
            }
            break;

        case APP_RX_TIMEOUT:
            Eeprom.EepromData.DemoSettings.RngStatus = RNG_TIMEOUT;
//            DemoInternalState = APP_RANGING_CONFIG;
            Eeprom.EepromData.DemoSettings.HoldDemo = true;

//            refreshDisplay =    true;
            bTimeInTx = false;              // LA:  Termina la Sessione di Trasmissione
            break;

        case APP_RX_ERROR:
//            DemoInternalState = APP_RANGING_CONFIG;
//            refreshDisplay =    true;
//            bTimeInTx = false;              // LA:  Termina la Sessione di Trasmissione

            Eeprom.EepromData.DemoSettings.HoldDemo = true;
            DemoInternalState = APP_IDLE;
            break;

        case APP_RANGING_DONE:

            // LA:  L'handshake è andato a buon fine, Ricevo gli HOP(s) richiesti
            //      =============================================================
            //
            RawRngResults[RngResultIndex] = Radio.GetRangingResult  (RANGING_RESULT_RAW);

            // LA:  Abilito/Disabilito la Correzione di I° Livello
            //
//                RawRngResults[RngResultIndex] += Sx1280RangingCorrection::GetRangingCorrectionPerSfBwGain   (ModulationParams.Params.LoRa.SpreadingFactor,
//                                                                                                            ModulationParams.Params.LoRa.Bandwidth,
//                                                                                                            Radio.GetRangingPowerDeltaThresholdIndicator( ));
            RngResultIndex++;
            Eeprom.EepromData.DemoSettings.CntPacketRxOK++;
            DemoInternalState = APP_RNG;
            break;

        case APP_RANGING_TIMEOUT:

            // LA:  L'handshake è andato a buon fine, Ricevo gli HOP(s) richiesti, ma ho avuto uno o più Timeout
            //      ============================================================================================
            //
            //  Lo gestisco ugualmente in "APP_RNG", ma senza registrarne il valore
            //
            DemoInternalState = APP_RNG;
            break;

        case APP_RNG:

            // LA:  Elaborazione del Pacchetto di Ranging
            //      =====================================
            //
            if  (SendNext == true) {

                //LA:   Il Ticker ha scatenato almeno un evento
                //
                SendNext = false;
                MeasuredChannels++;
                
                // LA:  Ci sono ancora Ricezioni da fare?
                //
                if  (MeasuredChannels <= Eeprom.EepromData.DemoSettings.RngRequestCount) {
                    Radio.SetRfFrequency    (Channels[CurrentChannel]);

                    // LA:  Applica la prevista Strategia d'Antenna
                    // LA:  Cicla i canali in autonomia     
                    //
                    switch  (Eeprom.EepromData.DemoSettings.RngAntenna) {

                        case DEMO_RNG_ANT_1:
                            Eeprom.EepromData.DemoSettings.AntennaSwitch = 0;
                            CurrentChannel++;
                            if  (CurrentChannel >= CHANNELS)    CurrentChannel -= CHANNELS;
                            break;

                        case DEMO_RNG_ANT_0:
                            Eeprom.EepromData.DemoSettings.AntennaSwitch = 1;
                            CurrentChannel++;
                            if  (CurrentChannel >= CHANNELS)    CurrentChannel -= CHANNELS;
                            break;

                        case DEMO_RNG_ANT_BOTH:
                            if  (ANT_SW == 1)   Eeprom.EepromData.DemoSettings.AntennaSwitch = 1;
                            else
                            {
                                Eeprom.EepromData.DemoSettings.AntennaSwitch = 0;
                                CurrentChannel++;
                                if  (CurrentChannel >= CHANNELS)    CurrentChannel -= CHANNELS;
                            }
                            break;
                    }
                    SetAntennaSwitch( );
                    DemoInternalState = APP_IDLE;
                    Radio.SetTx( ( TickTime_t ){ RADIO_TICK_SIZE_1000_US, 0xFFFF } );
                }
                else {

                    // LA:  Le ricezioni son state tutte eseguite:
                    //      ======================================
                    //
                    double  median = 0;
                    double  dOriginalMedian;
                
                    uint16_t i;
                    uint8_t ui8CBIterator;
                
                    if  ((DeviceAnswered) &&
                    //  (Eeprom.EepromData.DemoSettings.CntPacketRxOK >= (Eeprom.EepromData.DemoSettings.RngRequestCount/ 2))) {
                        (Eeprom.EepromData.DemoSettings.CntPacketRxOK > 0)) {
                        
                        //      ===========================================================
                        //      ===========================================================
                        // LA:  Un Partner Ha risposto alla richiesta di Ranging,
                        // LA:  Il numero di Pacchetti ricevuti è sufficiente al "Ranging".
                        //      ===========================================================
                        //      ===========================================================
                        //
                        Eeprom.EepromData.DemoSettings.RngStatus = RNG_VALID;

                        // LA:  Estraggo la Media degli HOP(s)
                        //
                        for (i = 0; i < Eeprom.EepromData.DemoSettings.CntPacketRxOK; i++) {
                            median +=   RawRngResults[i];
                        }
                        median /= Eeprom.EepromData.DemoSettings.CntPacketRxOK;
                        dOriginalMedian =   median;
                
                        // LA:  Compilazione del Buffer Circolare
                        //      =================================
                        //
                        //  Aggiungo il Valore appena rilevato.
                        //  Include la parte NEGATIVA del rilevamento.
                        //
                        //  Attenzione: Il BC Compilato è specifico per il Partner che ha risposto alla Richiesta di Ranging
                        //              Il Puntatore all'Effettivo BC è "ancora" contenuto in "ui8AnsweringPIndex".
                        //
                        if  (AnsweringPartnersList[ui8AnsweringPIndex].ui8CBIndex >= ciCircularBufferWidth) AnsweringPartnersList[ui8AnsweringPIndex].ui8CBIndex = 0;
                        AnsweringPartnersList[ui8AnsweringPIndex].dAverageDistanceCircularBuffer[AnsweringPartnersList[ui8AnsweringPIndex].ui8CBIndex++] = dOriginalMedian;

                        // LA:  Calcolo della Media Circolare
                        //      =============================
                        //
                        if  (AnsweringPartnersList[ui8AnsweringPIndex].ui8CB1stCountsIndex < ciCircularBufferWidth) AnsweringPartnersList[ui8AnsweringPIndex].ui8CB1stCountsIndex++;
                    
                        AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance =   AnsweringPartnersList[ui8AnsweringPIndex].dAverageDistanceCircularBuffer[0];
                        for (ui8CBIterator = 1; ui8CBIterator < AnsweringPartnersList[ui8AnsweringPIndex].ui8CB1stCountsIndex; ui8CBIterator++) {
                            AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance +=  AnsweringPartnersList[ui8AnsweringPIndex].dAverageDistanceCircularBuffer[ui8CBIterator];
                        }
                        AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance /=  AnsweringPartnersList[ui8AnsweringPIndex].ui8CB1stCountsIndex;
                
                        // LA:  Correzione di III° Livello    (< 18.5m)
                        // LA:  Da questo momento in poi il valore è per forza Positivo
                        //      =======================================================
                        //
                    //  if  (AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance <= 18.5) {
                        if  (AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance < 18.5) {
                            AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance =   exp ((AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance+ 2.4917)/ 7.2262);
                        }
                        else {
                            AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance =   18.5;
                        }

                        // LA:  Assumo i METRI come udm definitiva
                        //      ==================================
                        //
                        Eeprom.EepromData.DemoSettings.RngDistance = AnsweringPartnersList[ui8AnsweringPIndex].dRngRawCircularBufferDistance;
                        Eeprom.EepromData.DemoSettings.RngRawDistance = dOriginalMedian;                            // LA:  Aggiunta la Grezza
                    }
                    else {
                
                        // LA:  Numero di Pacchetti INSUFFICIENTE oppure
                        //      Indirizzamento NON CORRETTO.
                        //
                        Eeprom.EepromData.DemoSettings.RngStatus = RNG_PER_ERROR;
                    }
                    Eeprom.EepromData.DemoSettings.CntPacketRxOKSlave = Eeprom.EepromData.DemoSettings.CntPacketRxOK;
                    SendNextPacket.detach( );
                    Eeprom.EepromData.DemoSettings.HoldDemo = true;
                    SendNext = false;

                    bTimeInTx = false;              // LA:  Termina la Sessione di Trasmissione
                }
            }
            break;

        case APP_TX_TIMEOUT:
            DemoInternalState = APP_RANGING_CONFIG;
            Eeprom.EepromData.DemoSettings.HoldDemo = true;

//            refreshDisplay =    true;
//            bTimeInTx = false;              // LA:  Termina la Sessione di Trasmissione
            break;

        case APP_IDLE: // do nothing
            break;

        default:
            DemoInternalState = APP_RANGING_CONFIG;
            Eeprom.EepromData.DemoSettings.HoldDemo = true;
            break;
    }
//    return refreshDisplay;
    return DeviceAnswered;
}

// LA:  ========================================================================
// LA:  ========================================================================
// LA:  ========================================================================
// LA:  ========================================================================

// LA:  RangingSlave
//      ============
//
//  RangingSlave è il cuore del sistema MasterSlave 1-N.
//  Se un nuovo partner richiede un ranging, la richiesta viene soddisfatta e il suo indirizzo aggiunto alla lista.
//  dopo il ranging (Se terminato con successo) il partner viene BlackListed per un prefissato numero di secondi.
//  Per tutto qst tempo nessuna ulteriore richiesta di ranging verrà evasa a quel partner.
//

uint8_t RunDemoApplicationRangingSlave  (void) {
static uint32_t DeviceRangingAddress;
static bool     DeviceAddressed;    

    // LA:  Inizializza l'HW Radio se la Demo parte ora
    //      ===========================================
    //
    //  Questa parte viene eseguita SOLO una volta, ad ogni avvio della DEMO
    //
    if  (!DemoRunning) {
        DemoRunning = true;
        DeviceAddressed =   false;

        ANT_SW = 1;

        Eeprom.EepromData.DemoSettings.CntPacketTx = 0;
        Eeprom.EepromData.DemoSettings.RngFei      = 0.0;
        Eeprom.EepromData.DemoSettings.RngStatus   = RNG_INIT;
        InitializeDemoParameters    (Eeprom.EepromData.DemoSettings.ModulationType);

        // LA:  Inizializza i Parametri come "SLAVE"
        //      ====================================
        //
//        Radio.SetDioIrqParams   (irqMask, dio1Mask, dio2Mask, dio3Mask);
        Radio.SetDioIrqParams   (IRQ_RADIO_ALL,                                                                                                 // irqMask
                                IRQ_RADIO_ALL,                                                                                                  // dio1Mask
                                IRQ_RADIO_NONE,                                                                                                 // dio2Mask
                                IRQ_RADIO_NONE );                                                                                               // dio3Mask
        //
        DemoInternalState = APP_RANGING_CONFIG;                 // LA:  Termina la Configurazione dentro al "Main Case"
        pRefreshRxTimeout ();                                   // LA:  Avvio il Timeout Esterno d'Ascolto
    }

    // LA:  Attenzione, la gestione del chip è "PollingMode"
    //      ================================================
    //
    //  Per elaborare le Interruzioni occorre richiamare, ciclicamente, "Radio.ProcessIrqs( );"
    //
    Radio.ProcessIrqs( );

    // LA:  Caso "SLAVE"
    //      =============
    //
    switch  (DemoInternalState) {
        case APP_RANGING_CONFIG:
            if  (!DeviceAddressed) {
                Eeprom.EepromData.DemoSettings.RngStatus = RNG_INIT;

                ModulationParams.PacketType = PACKET_TYPE_LORA;
                ModulationParams.Params.LoRa.SpreadingFactor =  Eeprom.EepromData.ModulationParams.Params.LoRa.SpreadingFactor;
                //memcpy( &( ModulationParams.Params.LoRa.SpreadingFactor ), Eeprom.Buffer + MOD_RNG_SPREADF_EEPROM_ADDR,      1 );
                ModulationParams.Params.LoRa.Bandwidth =    Eeprom.EepromData.ModulationParams.Params.LoRa.Bandwidth;
                //memcpy( &( ModulationParams.Params.LoRa.Bandwidth ),       Eeprom.Buffer + MOD_RNG_BW_EEPROM_ADDR,           1 );
                ModulationParams.Params.LoRa.CodingRate =   Eeprom.EepromData.ModulationParams.Params.LoRa.CodingRate;
                //memcpy( &( ModulationParams.Params.LoRa.CodingRate ),      Eeprom.Buffer + MOD_RNG_CODERATE_EEPROM_ADDR,     1 );

                PacketParams.PacketType     = PACKET_TYPE_LORA;
                PacketParams.Params.LoRa.PreambleLength =   Eeprom.EepromData.PacketParams.Params.LoRa.PreambleLength;
                //memcpy( &( PacketParams.Params.LoRa.PreambleLength ),      Eeprom.Buffer + PAK_RNG_PREAMBLE_LEN_EEPROM_ADDR, 1 );
                PacketParams.Params.LoRa.HeaderType =   Eeprom.EepromData.PacketParams.Params.LoRa.HeaderType;
                //memcpy( &( PacketParams.Params.LoRa.HeaderType ),          Eeprom.Buffer + PAK_RNG_HEADERTYPE_EEPROM_ADDR,   1 );
                PacketParams.Params.LoRa.PayloadLength = 15;
                PacketParams.Params.LoRa.Crc =  Eeprom.EepromData.PacketParams.Params.LoRa.Crc;
                //memcpy( &( PacketParams.Params.LoRa.Crc ),                 Eeprom.Buffer + PAK_RNG_CRC_MODE_EEPROM_ADDR,     1 );
                PacketParams.Params.LoRa.InvertIQ = Eeprom.EepromData.PacketParams.Params.LoRa.InvertIQ;
                //memcpy( &( PacketParams.Params.LoRa.InvertIQ ),            Eeprom.Buffer + PAK_RNG_IQ_INV_EEPROM_ADDR,       1 );

                Radio.SetPacketType( ModulationParams.PacketType );
                Radio.SetModulationParams( &ModulationParams );
                Radio.SetPacketParams( &PacketParams );
                Radio.SetRfFrequency( Eeprom.EepromData.DemoSettings.Frequency );   // LA:  La frequenza x l'handshaking è quella di base

                // use listen mode here instead of rx continuous
                Radio.SetRx ((TickTime_t) { RADIO_TICK_SIZE_1000_US, 0xFFFF });     //
    
                DemoInternalState = APP_IDLE;
                //
                // LA:  Prepara il pacchetto e si mette in ascolto.
                //      ===========================================
                //
                //  L'Idle verrà interrotto da un "evento" di ricezione.
                //  Se la Ricezione confermerà il pacchetto di inizializzazione si potrà procedere.
                //  Se il Dispositivo è già indirizzato ("DeviceAddressed") la richiesta verrà scartata (ranging in corso)
            }
            break;

        case APP_RX_TIMEOUT:

            // LA:  E' terminato il tempo concesso (Internamente) alla ricezione.
            // LA:  Se sono ancora in "RNG_INIT", ritento
            // LA:  Altrimenti, vado in "APP_IDLE" permettendo al resto del thread di terminare

            if  ((!DeviceAddressed) && (Eeprom.EepromData.DemoSettings.RngStatus == RNG_INIT))  DemoInternalState = APP_RANGING_CONFIG;
            else                                                                                DemoInternalState = APP_IDLE;
            break;

        case APP_RX_ERROR:

            // LA:  E' avvenuto un errore di Ricezione.
            // LA:  Se stavo attendendo la risposta ad un "RNG_INIT" ritento,
            // LA:  Altrimenti, vado in "APP_IDLE" permettendo al resto del thread di terminare

            if  ((!DeviceAddressed) && (Eeprom.EepromData.DemoSettings.RngStatus == RNG_INIT))  DemoInternalState = APP_RANGING_CONFIG;
            else                                                                                DemoInternalState = APP_IDLE;
            break;

        case APP_RX:
            if  (!DeviceAddressed) {
            
                // LA:  Ricezione di un pacchetto completata
                //      ====================================
                //
                //  00 =>   Indirizzo del Dispositivo che ha richiesto il Ranging, x---
                //  01 =>   Indirizzo del Dispositivo che ha richiesto il Ranging, -x--
                //  02 =>   Indirizzo del Dispositivo che ha richiesto il Ranging, --x-
                //  03 =>   Indirizzo del Dispositivo che ha richiesto il Ranging, ---x
                //
                //  04 =>    Canale in Uso       (Parte da Zero)
                //  05 =>    Modalità d'Antenna  (0, 1 o Entrambe)
                //  06 =>    Numero di Pacchetti
                //
                //  07 =>    Lunghezza del Buffer Trasmesso (9)
                //  08 =>    CRC8 del Buffer[0..7]
                //
                //  Se il pacchetto ricevuto corrisponde ad un pacchetto di inizializzazione x il ranging e
                //  Se il dispositivo è sulla lista, posso iniziare il ranging.
                //
                //  In caso contrario attendo un pacchetto corretto o il timeout.
                //
                if  (Eeprom.EepromData.DemoSettings.RngStatus == RNG_INIT ) {
                    Radio.GetPayload    (Buffer,
                                        &BufferSize,
                                        BUFFER_SIZE);
                                        
                    if  ((Buffer[7] == 9) &&
                        (Buffer[8] == CRC8  (Buffer,
                                            (9- 1)))) {

                        DeviceRangingAddress =   (uint32_t (Buffer[0]))<< 24;
                        DeviceRangingAddress +=  (uint32_t (Buffer[1]))<< 16;
                        DeviceRangingAddress +=  (uint32_t (Buffer[2]))<< 8;
                        DeviceRangingAddress +=  (uint32_t (Buffer[3]));
                        //
                        Eeprom.EepromData.DemoSettings.ExtractRngAddress =  DeviceRangingAddress;
    
                        // LA:  Elaboro il Filtro di risposta:
                        //      ==============================
                        //
                        //  Verifico che l'indirizzo "Eeprom.EepromData.DemoSettings.ExtractRngAddress" sia presente in Tabella.
                        //  Se nn è presente lo aggiungo e morta lì.
                        //  Se è presente, aggiorno il suo Timeout e verifico che non sia BlackListed.
                        //  Se non è BlackListed Rispondo, Altrimenti lo Ignoro bellamente.
                        //  Scansionando la tabella verifico anche se ci sono posti vuoti, riportandone l'ultimo indice trovato in "ui8APEmptyItemIndex"
                        //  (Attenzione, se i posti vuoti sono esauriti, "ui8APEmptyItemIndex" ritorna "ciMaxActivePartners").
                        //
                        ui8APEmptyItemIndex = ciMaxActivePartners;
                        for (ui8APIndex = 0; ui8APIndex < ciMaxActivePartners; ui8APIndex++) {
                            if  (ActivePartnersList[ui8APIndex].ui32Address == 0)                                                   ui8APEmptyItemIndex = ui8APIndex;
                            if  (ActivePartnersList[ui8APIndex].ui32Address == Eeprom.EepromData.DemoSettings.ExtractRngAddress)    break;
                        }
                        if  (ui8APIndex == ciMaxActivePartners) {                       // LA:  Non Trovato, se possibile creo nuovo indice su "ui8APEmptyItemIndex"
                            if  (ui8APEmptyItemIndex < ciMaxActivePartners) {               // LA:  Slot libero in "ui8APEmptyItemIndex"
    
                                // LA:  Aggiunta di un Nuovo Partner
                                //      ============================
                                //
                                ui8APIndex =    ui8APEmptyItemIndex;        // LA:  Punta all'Indice Vuoto (In Creazione)
    
                                ActivePartnersList[ui8APIndex].ui32Address =    Eeprom.EepromData.DemoSettings.ExtractRngAddress;
                                ActivePartnersList[ui8APIndex].ui8BlackListTime =   0;          // LA:  E' il primo giro, Posso Rispondere
                            }
                        }
                        //
                        // LA:  In Uscita da Qui, se "ui8APIndex" < "ciMaxActivePartners" posso eseguire il Ranging
                        // LA:  ===================================================================================
                        // LA:  ===================================================================================

                        if  (ui8APIndex < ciMaxActivePartners) {                                        // LA:  Trovato, indice in "ui8APIndex"
                            ActivePartnersList[ui8APIndex].ui8ActivityTimeOut = ciActivityTimeout_s;    // LA:  Rinfresco (Comunque) il TimeOut
    
                            // LA:  Il Tempo in BlackList è Scaduto, posso Rispondere alla Richiesta di Ranging
                            //      ===========================================================================
                            //
                            if  (ActivePartnersList[ui8APIndex].ui8BlackListTime == 0) {                    // LA:  Tempo in BlackList Scaduto, posso Rispondere
                                ActivePartnersList[ui8APIndex].ui8BlackListTime =   ciBlackListTime_s;      // LA:  BlackList (Re)Starts Now
    
                                // LA:  Il Buffer 0..4 contiene l'indirizzo del richiedente
                                //      ===================================================
                                //
                                //  Lo rispedisco uguale, accettando il sincronismo ...
                                //
                                DeviceAddressed =   true;
                                Radio.GetPacketStatus   (&PacketStatus);
        
                                Eeprom.EepromData.DemoSettings.RngFei = Radio.GetFrequencyError ();
                                Eeprom.EepromData.DemoSettings.RssiValue =  PacketStatus.LoRa.RssiPkt;
                                Eeprom.EepromData.DemoSettings.CntPacketTx++;
                                //
                                CurrentChannel =    Buffer[4];
                                Eeprom.EepromData.DemoSettings.RngAntenna = Buffer[5];
                                Eeprom.EepromData.DemoSettings.RngRequestCount =    Buffer[6];     // LA:  Fin qui è "Buffer Ricevuto"
                                wait_us (10);
            
                                Buffer[4] = (((int32_t) Eeprom.EepromData.DemoSettings.RngFei)>> 24)& 0xFF ;
                                Buffer[5] = (((int32_t) Eeprom.EepromData.DemoSettings.RngFei)>> 16)& 0xFF ;
                                Buffer[6] = (((int32_t) Eeprom.EepromData.DemoSettings.RngFei)>> 8)& 0xFF ;
                                Buffer[7] = (((int32_t) Eeprom.EepromData.DemoSettings.RngFei)& 0xFF);
                                //
                                Buffer[8] = Eeprom.EepromData.DemoSettings.RssiValue;

                                Buffer[9] = ( Eeprom.EepromData.DemoSettings.RngAddress >> 24 ) & 0xFF;
                                Buffer[10] = ( Eeprom.EepromData.DemoSettings.RngAddress >> 16 ) & 0xFF;
                                Buffer[11] = ( Eeprom.EepromData.DemoSettings.RngAddress >>  8 ) & 0xFF;
                                Buffer[12] = ( Eeprom.EepromData.DemoSettings.RngAddress & 0xFF );

                                // LA:  Completa la risposta:
                                //      =====================
                                //
                                //  00 =>   Indirizzo del Dispositivo che ha richiesto il Ranging, x---
                                //  01 =>   Indirizzo del Dispositivo che ha richiesto il Ranging, -x--
                                //  02 =>   Indirizzo del Dispositivo che ha richiesto il Ranging, --x-
                                //  03 =>   Indirizzo del Dispositivo che ha richiesto il Ranging, ---x
                                //
                                //  04 <=   Frequency Error Indicator x il Ranging (RngFei), x---
                                //  05 <=   Frequency Error Indicator x il Ranging (RngFei), -x--
                                //  06 <=   Frequency Error Indicator x il Ranging (RngFei), --x-
                                //  07 <=   Frequency Error Indicator x il Ranging (RngFei), ---x
                                //  08 <=   RssiValue
                                //
                                //  09 <=   Indirizzo HW di questo Dispositivo, x---
                                //  10 <=   Indirizzo HW di questo Dispositivo, -x--
                                //  11 <=   Indirizzo HW di questo Dispositivo, --x-
                                //  12 <=   Indirizzo HW di questo Dispositivo, ---x
                                //
                                //  13 <=   Lunghezza del Buffer Trasmesso (15)
                                //  14 <=   CRC8 del Buffer[0..13]
                                //
                                Buffer[13] = PacketParams.Params.LoRa.PayloadLength;                // LA:  Numero di Bytes Totali
                                Buffer[14] = CRC8   (Buffer,                                        //      All but CRC8
                                                    (PacketParams.Params.LoRa.PayloadLength- 1));   //      ============
            
                                Radio.SendPayload   (Buffer,
                                                    PacketParams.Params.LoRa.PayloadLength,
                                                    (TickTime_t){ RADIO_TICK_SIZE_1000_US, RNG_COM_TIMEOUT } );
                                //
                                // LA:  Invia la risposta di Configurazione (Operazione di Ranging) al dispositivo chiamante.
                                //      =====================================================================================
                                //
                                //  Appena terminata la trasmissione, l'IDLE verrà interrotto da un evento di trasmissione (Completata)
                                //
                                DemoInternalState = APP_IDLE;
                            }
                            else {
                                
                                // LA:  Il Dispositivo è ancora in BlackList
                                //      ====================================
                                //
                                DemoInternalState = APP_RANGING_CONFIG;         // LA:  Mi rimetto in Attesa di un pacchetto di Configurazione
                            }
                        }
                        else {
                            
                            // LA:  Il Dispositivo Chiamante è nuovo ma non c'è posto x Aggiungerlo
                            //      ===============================================================
                            //
                            DemoInternalState = APP_RANGING_CONFIG;         // LA:  Mi rimetto in Attesa di un pacchetto di Configurazione
                        }
                    }
                    else {

                        // LA:  Il Payload (Ricevuto) è incoerente
                        //      ==================================
                        //
                        DemoInternalState = APP_RANGING_CONFIG;         // LA:  Mi rimetto in Attesa di un pacchetto di Configurazione
                    }
                }
                else {

                    // LA:  Ricevuto un pacchetto inatteso
                    //      ==============================
                    //
                    DemoInternalState = APP_IDLE;                   // LA:  Ignoro la Ricezione e torno in Idle
                //  DemoInternalState = APP_RANGING_CONFIG;         // LA:  Mi rimetto in Attesa di un pacchetto di Configurazione
                }
            }
            else {

                // LA:  Il Dispositivo era già stato indirizzato
                //      ========================================
                //
                DemoInternalState = APP_IDLE;                   // LA:  Ignoro la Ricezione e torno in Idle
            }
            break;

        case APP_TX:
            if  (DeviceAddressed) {

                // LA:  Trasmissione di un pacchetto completata
                //      =======================================
                //
                //  Caso 1, Ho risposto alla configurazione di Ranging
                //  Caso 2, L'evento consiste in un HOP terminato
                //
                if  (Eeprom.EepromData.DemoSettings.RngStatus == RNG_INIT) {
    
                    // LA:  Pacchetto di configurazione trasmesso al chiamante
                    //      ==================================================
                    //
                    Eeprom.EepromData.DemoSettings.RngStatus = RNG_PROCESS;     // LA:  Inizializzazione Teminata, entering "RNG_PROCESS"
    
                    // LA:  Imposto il treno di Pacchetti (HOP) impostati x il ranging
                    //      ==========================================================
                    //
                    ModulationParams.PacketType =   PACKET_TYPE_RANGING;
                    ModulationParams.Params.LoRa.SpreadingFactor =  (RadioLoRaSpreadingFactors_t) Eeprom.EepromData.DemoSettings.ModulationParam1;
                    //memcpy  (&(ModulationParams.Params.LoRa.SpreadingFactor), Eeprom.Buffer+ MOD_RNG_SPREADF_EEPROM_ADDR, 1);
                    ModulationParams.Params.LoRa.Bandwidth =    (RadioLoRaBandwidths_t) Eeprom.EepromData.DemoSettings.ModulationParam2;
                    //memcpy  (&(ModulationParams.Params.LoRa.Bandwidth), Eeprom.Buffer + MOD_RNG_BW_EEPROM_ADDR, 1);
                    ModulationParams.Params.LoRa.CodingRate =   (RadioLoRaCodingRates_t) Eeprom.EepromData.DemoSettings.ModulationParam3;
                    //memcpy  (&(ModulationParams.Params.LoRa.CodingRate), Eeprom.Buffer + MOD_RNG_CODERATE_EEPROM_ADDR, 1);
                    //
                    PacketParams.PacketType =   PACKET_TYPE_RANGING;
                    PacketParams.Params.LoRa.PreambleLength =   (uint8_t) Eeprom.EepromData.DemoSettings.PacketParam1;
                    //memcpy  (&(PacketParams.Params.LoRa.PreambleLength), Eeprom.Buffer + PAK_RNG_PREAMBLE_LEN_EEPROM_ADDR, 1);
                    PacketParams.Params.LoRa.HeaderType =   (RadioLoRaPacketLengthsModes_t) Eeprom.EepromData.DemoSettings.PacketParam2;
                    //memcpy  (&(PacketParams.Params.LoRa.HeaderType), Eeprom.Buffer + PAK_RNG_HEADERTYPE_EEPROM_ADDR, 1);
                    PacketParams.Params.LoRa.PayloadLength =    10;
                    PacketParams.Params.LoRa.Crc =  (RadioLoRaCrcModes_t) Eeprom.EepromData.DemoSettings.PacketParam4;
                    //memcpy  (&(PacketParams.Params.LoRa.Crc), Eeprom.Buffer + PAK_RNG_CRC_MODE_EEPROM_ADDR, 1);
                    PacketParams.Params.LoRa.InvertIQ = (RadioLoRaIQModes_t) Eeprom.EepromData.DemoSettings.PacketParam5;
                    //memcpy  (&(PacketParams.Params.LoRa.InvertIQ), Eeprom.Buffer + PAK_RNG_IQ_INV_EEPROM_ADDR, 1);
    
                    Radio.SetPacketType( ModulationParams.PacketType );
                    Radio.SetModulationParams( &ModulationParams );
                    Radio.SetPacketParams( &PacketParams );
    
    //                Radio.SetDeviceRangingAddress( Eeprom.EepromData.DemoSettings.RngAddress );           // LA:  Attenzione qui ...
    //                Radio.SetDeviceRangingAddress(DeviceRangingAddress);                                  // LA:  Attenzione qui ...
                    Radio.SetDeviceRangingAddress(Eeprom.EepromData.DemoSettings.ExtractRngAddress);        // LA:  Attenzione qui ...
    
                    Radio.SetRangingCalibration( Eeprom.EepromData.DemoSettings.RngCalib );
    //                Radio.SetTxParams( Eeprom.EepromData.DemoSettings.TxPower, RADIO_RAMP_20_US );
                    Radio.SetTxParams( Eeprom.EepromData.DemoSettings.TxPower, RADIO_RAMP_02_US );

                    Eeprom.EepromData.DemoSettings.CntPacketRxOK = 0;
                    MeasuredChannels = 0;
                    Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave = 0;
    
                    // LA:  Il Ranging è iniziato.
                    //      ======================
                    //
                    //  Creo un ticker con callback a "SendNextPacketEvent" e temporizzazione nota, per gestire il Timeout
                    //  Quando viene scatenato, avviene che:
                    //      SendNext =  true;
                    //      Se lo stato interno è (Ancora) RNG_PROCESS, incrementa il Pacchetto KO (Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave++)
                    //
                    SendNextPacket.detach ();
                    SendNextPacket.attach_us    (&SendNextPacketEvent,
                                                Eeprom.EepromData.DemoSettings.RngReqDelay* 1000);
    
                    // LA:  Se va tutto bene
                    //      ================
                    //
                    //  Se la comunicazione procede come previsto, arriveranno (Evento "APP_RANGING_DONE") una serie
                    //  di pacchetti (HOP) di ranging. Mi metto quindi in Ascolto.
    
                    DemoInternalState = APP_RNG;
                }
                
                // LA:  Trasmissione Qualunque Terminata, Torno in Idle
                //      ===============================================
                //
                else    DemoInternalState = APP_IDLE;
            }
            break;

        case APP_TX_TIMEOUT:

            // LA:  Timeout di Trasmissione
            //      =======================
            //
            //  Queto caso è pressochè impossibile
            //
            if  (Eeprom.EepromData.DemoSettings.RngStatus == RNG_INIT)  DemoInternalState = APP_RANGING_CONFIG;
            else                                                        DemoInternalState = APP_IDLE;
            break;

        case APP_RANGING_DONE:

            // LA:  E' arrivato un pacchetto di Ranging
            //      ===================================
            //
            //  Se "DeviceAddressed", il pacchetto è x me e lo elaboro,
            //  Altrimenti, lo ignoro
            //
            if  (DeviceAddressed) {
                Eeprom.EepromData.DemoSettings.CntPacketRxOK++;
                DemoInternalState = APP_RNG;
            }
            else    DemoInternalState = APP_IDLE;
            break;

        case APP_RANGING_TIMEOUT:

            // LA:  Un pacchetto di Ranging ha sforato
            //      ==================================
            //
            //  Se "DeviceAddressed", il pacchetto era x me e lo elaboro,
            //  Altrimenti, lo ignoro
            //
            if  (DeviceAddressed) {
                Eeprom.EepromData.DemoSettings.CntPacketRxOK++;
                DemoInternalState = APP_RNG;
            }
            else    DemoInternalState = APP_IDLE;
            break;

        case APP_RNG:
            if  (DeviceAddressed) {
                if  (SendNext == true) {
                    SendNext = false;
    
                    // LA:  Elabora ogni singolo HOP fino all'ultimo (RngRequestCount)
                    //      ==========================================================
                    //
                    MeasuredChannels++;
                    if  (MeasuredChannels <= Eeprom.EepromData.DemoSettings.RngRequestCount) {
                        Radio.SetRfFrequency    (Channels[CurrentChannel]);
    
                        // LA:  Cambia Autonomamente il Canale (40 Canali) ciclandoli
                        // LA:  Applica la Strategia d'antenna  =====================
                        //      =====================================================
                        //
                        switch  (Eeprom.EepromData.DemoSettings.RngAntenna) {
                            case DEMO_RNG_ANT_1:
                                Eeprom.EepromData.DemoSettings.AntennaSwitch =  0;
                                CurrentChannel++;
                                if  (CurrentChannel >= CHANNELS)    CurrentChannel -=   CHANNELS;
                                break;
    
                            case DEMO_RNG_ANT_0:
                                Eeprom.EepromData.DemoSettings.AntennaSwitch =  1;
                                CurrentChannel++;
                                if  (CurrentChannel >= CHANNELS)    CurrentChannel -=   CHANNELS;
                                break;
    
                            case DEMO_RNG_ANT_BOTH:
                                if  (ANT_SW == 1)   Eeprom.EepromData.DemoSettings.AntennaSwitch = 1;
                                else {
                                    Eeprom.EepromData.DemoSettings.AntennaSwitch = 0;
                                    CurrentChannel++;
                                    if  (CurrentChannel >= CHANNELS)    CurrentChannel -=   CHANNELS;
                                }
                                break;
                        }
                        SetAntennaSwitch( );
    
                        // LA:  Mi rimetto in Ascolto
                        //      =====================
                        //
                        DemoInternalState = APP_IDLE;
                        Radio.SetRx ((TickTime_t) { RADIO_TICK_SIZE_1000_US, Eeprom.EepromData.DemoSettings.RngReqDelay });
                    }
                    else {
    
                        // LA:  Tutti gli HOP son stati ricevuti (Buoni o TimedOut)
                        //      ===================================================
                        //
                        //  Spengo la radio
                        //  Richiedo un refresh al display (Slave Only)
                        //  Arresto il Ticker
                        //  Termino il tempo dedicato alla Trasmissione
                        //
                        Radio.SetStandby(STDBY_RC);
                        SendNextPacket.detach( );
                        //
                        DeviceAddressed =   false;                  // LA:  Reinizializzo ...

                        // LA:  Finale Normale
                        //      ==============
                        //
//                        Eeprom.EepromData.DemoSettings.RngStatus = RNG_VALID;
//                        bTimeInRx = false;                          // LA:  Ricezione Terminata
                        //
                        //      ==============================================================
                        //      ==============================================================
                        // LA:  In qualunque stato lo metta, esce fino allo Slaving Sucessivo.
                        //      ==============================================================
                        //      ==============================================================

                        // LA:  Finale Alternativo:
                        //      ===================
                        //
                        //  Ripete L'Inizializzazione.
                        //  Si rimette in Ricezione.
                        //
                        Eeprom.EepromData.DemoSettings.CntPacketTx = 0;
                        Eeprom.EepromData.DemoSettings.RngFei      = 0.0;
//                        Eeprom.EepromData.DemoSettings.RngStatus   = RNG_INIT;
                        Eeprom.EepromData.DemoSettings.RngStatus = RNG_VALID;
                        InitializeDemoParameters    (Eeprom.EepromData.DemoSettings.ModulationType);

                        // LA:  Inizializza i Parametri come "SLAVE"
                        //      ====================================
                        //
                //        Radio.SetDioIrqParams   (irqMask, dio1Mask, dio2Mask, dio3Mask);
                        Radio.SetDioIrqParams   (IRQ_RADIO_ALL,         // irqMask
                                                IRQ_RADIO_ALL,          // dio1Mask
                                                IRQ_RADIO_NONE,         // dio2Mask
                                                IRQ_RADIO_NONE );       // dio3Mask
                        //
                        DemoInternalState = APP_RANGING_CONFIG;         // LA:  Termina la Configurazione dentro al "Main Case"
                        //
                        //      ==================================================
                        //      ==================================================
                        // LA:  Sfrutta TUTTO il tempo in Rx concesso dal TimeOut.
                        //      ==================================================
                        //      ==================================================
                    }
                }
            }
            else    DemoInternalState = APP_IDLE;
            break;

        case APP_IDLE: // do nothing
            if  (Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave > DEMO_RNG_CHANNELS_COUNT_MAX) {
//            if  (Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave > 255) {

                // LA:  Ho perso tutti i Pacchetti
                //      ==========================
                //
                //  Spengo la radio
                //  Richiedo un refresh al display (Slave Only)
                //  Arresto il Ticker
                //  Termino il tempo dedicato alla Trasmissione
                //
                Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave = 0;

                Radio.SetStandby(STDBY_RC);
                SendNextPacket.detach( );

                DeviceAddressed =   false;                  // LA:  Reinizializzo ...
                bTimeInRx = false;                          // LA:  Ricezione Terminata
                //
                //      ==============================================================
                //      ==============================================================
                // LA:  In qualunque stato lo metta, esce fino allo Slaving Sucessivo.
                //      ==============================================================
                //      ==============================================================
            }
            break;

        default:

            // LA:  Se passa da qui qualcosa è andato a ...
            //      =======================================
            //
            //  Spengo la radio
            //  Richiedo un refresh al display (Slave Only)
            //  Arresto il Ticker
            //  Termino il tempo dedicato alla Trasmissione
            //

            Radio.SetStandby(STDBY_RC);
            SendNextPacket.detach( );

            DeviceAddressed =   false;                  // LA:  Reinizializzo ...
            bTimeInRx = false;                          // LA:  Ricezione Terminata
            //
            //      ==============================================================
            //      ==============================================================
            // LA:  In qualunque stato lo metta, esce fino allo Slaving Sucessivo.
            //      ==============================================================
            //      ==============================================================
            break;
    }
    return DeviceAddressed;
}

// LA:  ========================================================================
// LA:  ========================================================================
// LA:  ========================================================================
// LA:  ========================================================================

void StopDemoApplication( void )
{
    if( DemoRunning == true )
    {
        __disable_irq( );    // Disable Interrupts

        if( Radio.GetOpMode( ) == MODE_SLEEP )
        {
            Radio.Wakeup( );
            InitializeDemoParameters( Eeprom.EepromData.DemoSettings.ModulationType );
        }
//        RX_LED = 0;
//        TX_LED = 0;
        DemoRunning = false;
        SendNext = false;
//        ReceiveNext = false;
//        PacketRxSequence = 0;
//        PacketRxSequencePrev = 0;
        Eeprom.EepromData.DemoSettings.CntPacketTx    = 0;
        Eeprom.EepromData.DemoSettings.CntPacketRxOK  = 0;
        Eeprom.EepromData.DemoSettings.CntPacketRxKO  = 0;
        Eeprom.EepromData.DemoSettings.RxTimeOutCount = 0;

        Radio.SetAutoFs( false );
        DemoInternalState = APP_IDLE;
        Radio.SetStandby( STDBY_RC );
        Radio.ClearIrqStatus( IRQ_RADIO_ALL );
        SendNextPacket.detach( );
        ReceiveNextPacket.detach( );
        
        __enable_irq( );     // Enable Interrupts
    }
}

void InitializeDemoParameters( uint8_t modulation )
{
    Radio.SetStandby( STDBY_RC );
    Radio.SetRegulatorMode( ( RadioRegulatorModes_t )Eeprom.EepromData.DemoSettings.RadioPowerMode );

    if( modulation == PACKET_TYPE_LORA )
    {
        ModulationParams.PacketType = PACKET_TYPE_LORA;
        PacketParams.PacketType     = PACKET_TYPE_LORA;

        ModulationParams.Params.LoRa.SpreadingFactor = ( RadioLoRaSpreadingFactors_t )  Eeprom.EepromData.DemoSettings.ModulationParam1;
        ModulationParams.Params.LoRa.Bandwidth       = ( RadioLoRaBandwidths_t )        Eeprom.EepromData.DemoSettings.ModulationParam2;
        ModulationParams.Params.LoRa.CodingRate      = ( RadioLoRaCodingRates_t )       Eeprom.EepromData.DemoSettings.ModulationParam3;
        PacketParams.Params.LoRa.PreambleLength      =                                  Eeprom.EepromData.DemoSettings.PacketParam1;
        PacketParams.Params.LoRa.HeaderType          = ( RadioLoRaPacketLengthsModes_t )Eeprom.EepromData.DemoSettings.PacketParam2;
        PacketParams.Params.LoRa.PayloadLength       =                                  Eeprom.EepromData.DemoSettings.PacketParam3;
        PacketParams.Params.LoRa.Crc                 = ( RadioLoRaCrcModes_t )          Eeprom.EepromData.DemoSettings.PacketParam4;
        PacketParams.Params.LoRa.InvertIQ            = ( RadioLoRaIQModes_t )           Eeprom.EepromData.DemoSettings.PacketParam5;

        Eeprom.EepromData.DemoSettings.PayloadLength = PacketParams.Params.LoRa.PayloadLength;
        Radio.SetLNAGainSetting(LNA_LOW_POWER_MODE);
    }
    else if( modulation == PACKET_TYPE_FLRC )
    {
        ModulationParams.PacketType = PACKET_TYPE_FLRC;
        PacketParams.PacketType     = PACKET_TYPE_FLRC;

        ModulationParams.Params.Flrc.BitrateBandwidth  = ( RadioFlrcBitrates_t )       Eeprom.EepromData.DemoSettings.ModulationParam1;
        ModulationParams.Params.Flrc.CodingRate        = ( RadioFlrcCodingRates_t )    Eeprom.EepromData.DemoSettings.ModulationParam2;
        ModulationParams.Params.Flrc.ModulationShaping = ( RadioModShapings_t )        Eeprom.EepromData.DemoSettings.ModulationParam3;
        PacketParams.Params.Flrc.PreambleLength        = ( RadioPreambleLengths_t )    Eeprom.EepromData.DemoSettings.PacketParam1;
        PacketParams.Params.Flrc.SyncWordLength        = ( RadioFlrcSyncWordLengths_t )Eeprom.EepromData.DemoSettings.PacketParam2;
        PacketParams.Params.Flrc.SyncWordMatch         = ( RadioSyncWordRxMatchs_t )   Eeprom.EepromData.DemoSettings.PacketParam3;
        PacketParams.Params.Flrc.HeaderType            = ( RadioPacketLengthModes_t )  Eeprom.EepromData.DemoSettings.PacketParam4;
        PacketParams.Params.Flrc.PayloadLength         =                               Eeprom.EepromData.DemoSettings.PacketParam5;
        PacketParams.Params.Flrc.CrcLength             = ( RadioCrcTypes_t )           Eeprom.EepromData.DemoSettings.PacketParam6;
        PacketParams.Params.Flrc.Whitening             = ( RadioWhiteningModes_t )     Eeprom.EepromData.DemoSettings.PacketParam7;

        Eeprom.EepromData.DemoSettings.PayloadLength = PacketParams.Params.Flrc.PayloadLength;
        Radio.SetLNAGainSetting(LNA_HIGH_SENSITIVITY_MODE);
    }
    else if( modulation == PACKET_TYPE_GFSK )
    {
        ModulationParams.PacketType = PACKET_TYPE_GFSK;
        PacketParams.PacketType     = PACKET_TYPE_GFSK;

        ModulationParams.Params.Gfsk.BitrateBandwidth  = ( RadioGfskBleBitrates_t )  Eeprom.EepromData.DemoSettings.ModulationParam1;
        ModulationParams.Params.Gfsk.ModulationIndex   = ( RadioGfskBleModIndexes_t )Eeprom.EepromData.DemoSettings.ModulationParam2;
        ModulationParams.Params.Gfsk.ModulationShaping = ( RadioModShapings_t )      Eeprom.EepromData.DemoSettings.ModulationParam3;
        PacketParams.Params.Gfsk.PreambleLength        = ( RadioPreambleLengths_t )  Eeprom.EepromData.DemoSettings.PacketParam1;
        PacketParams.Params.Gfsk.SyncWordLength        = ( RadioSyncWordLengths_t )  Eeprom.EepromData.DemoSettings.PacketParam2;
        PacketParams.Params.Gfsk.SyncWordMatch         = ( RadioSyncWordRxMatchs_t ) Eeprom.EepromData.DemoSettings.PacketParam3;
        PacketParams.Params.Gfsk.HeaderType            = ( RadioPacketLengthModes_t )Eeprom.EepromData.DemoSettings.PacketParam4;
        PacketParams.Params.Gfsk.PayloadLength         =                             Eeprom.EepromData.DemoSettings.PacketParam5;
        PacketParams.Params.Gfsk.CrcLength             = ( RadioCrcTypes_t )         Eeprom.EepromData.DemoSettings.PacketParam6;
        PacketParams.Params.Gfsk.Whitening             = ( RadioWhiteningModes_t )   Eeprom.EepromData.DemoSettings.PacketParam7;

        Eeprom.EepromData.DemoSettings.PayloadLength = PacketParams.Params.Gfsk.PayloadLength;
        Radio.SetLNAGainSetting(LNA_HIGH_SENSITIVITY_MODE);
    }

    if( modulation == PACKET_TYPE_RANGING )
    {
        Radio.SetBufferBaseAddresses( 0x00, 0x00 );
//        Radio.SetTxParams( Eeprom.EepromData.DemoSettings.TxPower, RADIO_RAMP_20_US );
        Radio.SetTxParams( Eeprom.EepromData.DemoSettings.TxPower, RADIO_RAMP_02_US );
        ModulationParams.Params.LoRa.SpreadingFactor =  (RadioLoRaSpreadingFactors_t) Eeprom.EepromData.DemoSettings.ModulationParam1;
        //memcpy( &( ModulationParams.Params.LoRa.SpreadingFactor ), Eeprom.Buffer + MOD_RNG_SPREADF_EEPROM_ADDR, 1 );
        ModulationParams.Params.LoRa.Bandwidth =    (RadioLoRaBandwidths_t) Eeprom.EepromData.DemoSettings.ModulationParam2;
        //memcpy( &( ModulationParams.Params.LoRa.Bandwidth ),       Eeprom.Buffer + MOD_RNG_BW_EEPROM_ADDR,      1 );

        switch( ModulationParams.Params.LoRa.Bandwidth )
        {
            case LORA_BW_0400:
                Eeprom.EepromData.DemoSettings.RngCalib     = RNG_CALIB_0400[ ( ModulationParams.Params.LoRa.SpreadingFactor >> 4 ) - 5 ];
                Eeprom.EepromData.DemoSettings.RngFeiFactor = ( double )RNG_FGRAD_0400[ ( ModulationParams.Params.LoRa.SpreadingFactor >> 4 ) - 5 ];
                Eeprom.EepromData.DemoSettings.RngReqDelay  = RNG_TIMER_MS >> ( 0 + 10 - ( ModulationParams.Params.LoRa.SpreadingFactor >> 4 ) );
                break;

            case LORA_BW_0800:
                Eeprom.EepromData.DemoSettings.RngCalib     = RNG_CALIB_0800[ ( ModulationParams.Params.LoRa.SpreadingFactor >> 4 ) - 5 ];
                Eeprom.EepromData.DemoSettings.RngFeiFactor = ( double )RNG_FGRAD_0800[ ( ModulationParams.Params.LoRa.SpreadingFactor >> 4 ) - 5 ];
                Eeprom.EepromData.DemoSettings.RngReqDelay  = RNG_TIMER_MS >> ( 1 + 10 - ( ModulationParams.Params.LoRa.SpreadingFactor >> 4 ) );
                break;

            case LORA_BW_1600:
                Eeprom.EepromData.DemoSettings.RngCalib     = RNG_CALIB_1600[ ( ModulationParams.Params.LoRa.SpreadingFactor >> 4 ) - 5 ];
                Eeprom.EepromData.DemoSettings.RngFeiFactor = ( double )RNG_FGRAD_1600[ ( ModulationParams.Params.LoRa.SpreadingFactor >> 4 ) - 5 ];
                Eeprom.EepromData.DemoSettings.RngReqDelay  = RNG_TIMER_MS >> ( 2 + 10 - ( ModulationParams.Params.LoRa.SpreadingFactor >> 4 ) );
                break;
        }
        Radio.SetPollingMode( );
        Radio.SetLNAGainSetting(LNA_LOW_POWER_MODE);
    }
    else
    {
        Radio.SetStandby( STDBY_RC );
        Radio.SetPacketType( ModulationParams.PacketType );
        Radio.SetRfFrequency( Eeprom.EepromData.DemoSettings.Frequency );
        Radio.SetBufferBaseAddresses( 0x00, 0x00 );
        Radio.SetModulationParams( &ModulationParams );
        Radio.SetPacketParams( &PacketParams );

        uint8_t A1[] = { 0xDD, 0xA0, 0x96, 0x69, 0xDD };

        // only used in GFSK, FLRC (4 bytes max) and BLE mode
        Radio.SetSyncWord( 1, A1 );
//        Radio.SetSyncWord( 1, ( uint8_t[] ){ 0xDD, 0xA0, 0x96, 0x69, 0xDD } );

        // only used in GFSK, FLRC
        uint8_t crcSeedLocal[2] = {0x45, 0x67};
        Radio.SetCrcSeed( crcSeedLocal );

        Radio.SetCrcPolynomial( 0x0123 );
//        Radio.SetTxParams( Eeprom.EepromData.DemoSettings.TxPower, RADIO_RAMP_20_US );
        Radio.SetTxParams( Eeprom.EepromData.DemoSettings.TxPower, RADIO_RAMP_02_US );
        Radio.SetPollingMode( );
    }
}

/*!
 * \brief Callback of ticker PerSendNextPacket
 */
void SendNextPacketEvent    (void) {
    SendNext =  true;
    if  (Eeprom.EepromData.DemoSettings.RngStatus == RNG_PROCESS)   Eeprom.EepromData.DemoSettings.CntPacketRxKOSlave++;
}

/*!
 * \brief Callback of ticker ReceiveNextPacket
 */
void ReceiveNextPacketEvent (void) {
    ReceiveNext =   true;
}

uint8_t CheckDistance( void )
{
//    double  displayRange = 0.0;
    double  median = 0;
    double  dOriginalMedian;

//    uint16_t j = 0;
    uint16_t i;
    uint8_t ui8CBIterator;

//    if  (Eeprom.EepromData.DemoSettings.CntPacketRxOK >= Eeprom.EepromData.DemoSettings.RngRequestCount) {
    if  (Eeprom.EepromData.DemoSettings.CntPacketRxOK > 0) {
        Eeprom.EepromData.DemoSettings.RngStatus = RNG_VALID;

        for (i = 0; i < Eeprom.EepromData.DemoSettings.CntPacketRxOK; i++) {
            median +=   RawRngResults[i];
        }
        median /= Eeprom.EepromData.DemoSettings.CntPacketRxOK;
        dOriginalMedian =   median;

        // LA:  Compilazione del Buffer Circolare
        //      =================================
        //
        //  Aggiungo il Valore appena rilevato.
        //  Include la parte NEGATIVA del rilevamento.
        //
        if  (ui8CBIndex >= ciCircularBufferWidth)    ui8CBIndex = 0;
        dAverageDistanceCircularBuffer [ui8CBIndex++] = dOriginalMedian;

        // LA:  Calcolo della Media Circolare
        //      =============================
        //
        if  (ui8CB1stCountsIndex < ciCircularBufferWidth)    ui8CB1stCountsIndex++;
    
        Eeprom.EepromData.DemoSettings.RngRawCircularBufferDistance =   dAverageDistanceCircularBuffer[0];
        for (ui8CBIterator = 1; ui8CBIterator < ui8CB1stCountsIndex; ui8CBIterator++) {
            Eeprom.EepromData.DemoSettings.RngRawCircularBufferDistance +=   dAverageDistanceCircularBuffer[ui8CBIterator];
        }
        Eeprom.EepromData.DemoSettings.RngRawCircularBufferDistance /=   ui8CB1stCountsIndex;

        // LA:  Correzione di II° Livello    (< 100m)
        //      =====================================
        //
//        if  (median < 100) {
//            median =    Sx1280RangingCorrection::ComputeRangingCorrectionPolynome   (ModulationParams.Params.LoRa.SpreadingFactor,
//                                                                                    ModulationParams.Params.LoRa.Bandwidth,
//                                                                                    median);
//        }

        // LA:  Correzione di III° Livello    (< 18.5m)
        // LA:  Da questo momento in poi il valore è per forza Positivo
        //      =======================================================
        //
        if  (Eeprom.EepromData.DemoSettings.RngRawCircularBufferDistance <= 18.5) {
            Eeprom.EepromData.DemoSettings.RngRawCircularBufferDistance = exp ((Eeprom.EepromData.DemoSettings.RngRawCircularBufferDistance+ 2.4917)/ 7.2262);
        }

        // LA:  Adeguamento della misura all'untità di visualizzazione
        //      ======================================================
        //
        switch  (Eeprom.EepromData.DemoSettings.RngUnit) {
            case DEMO_RNG_UNIT_SEL_M:
                Eeprom.EepromData.DemoSettings.RngDistance = Eeprom.EepromData.DemoSettings.RngRawCircularBufferDistance;
                Eeprom.EepromData.DemoSettings.RngRawDistance = dOriginalMedian;                            // LA:  Aggiunta la Grezza
                break;

            case DEMO_RNG_UNIT_SEL_YD:
                Eeprom.EepromData.DemoSettings.RngDistance = Eeprom.EepromData.DemoSettings.RngRawCircularBufferDistance * DEMO_RNG_UNIT_CONV_YD;
                Eeprom.EepromData.DemoSettings.RngRawDistance = dOriginalMedian * DEMO_RNG_UNIT_CONV_YD;    // LA:  Aggiunta la Grezza
                break;

            case DEMO_RNG_UNIT_SEL_MI:
                Eeprom.EepromData.DemoSettings.RngDistance = Eeprom.EepromData.DemoSettings.RngRawCircularBufferDistance * DEMO_RNG_UNIT_CONV_MI;
                Eeprom.EepromData.DemoSettings.RngRawDistance = dOriginalMedian * DEMO_RNG_UNIT_CONV_MI;    // LA:  Aggiunta la Grezza
                break;
        }
    }
    else {

        // LA:  Numero di Pacchetti INSUFFICIENTE
        //      =================================
        //
        Eeprom.EepromData.DemoSettings.RngStatus = RNG_PER_ERROR;
    }
    return  Eeprom.EepromData.DemoSettings.CntPacketRxOK;
}

void LedBlink   (void) {
    if  ((TX_LED == 0)&& (RX_LED == 0))     TX_LED = 1;
    else if ((TX_LED == 1)&& (RX_LED == 0)) RX_LED = 1;
    else if ((TX_LED == 1)&& (RX_LED == 1)) TX_LED = 0;
    else                                    RX_LED = 0;
}

void SetAntennaSwitch   (void) {
    if  (Eeprom.EepromData.DemoSettings.AntennaSwitch == 0) ANT_SW =    1;
    else                                                    ANT_SW =    0;
}

// ************************     Radio Callbacks     ****************************
// *                                                                           *
// * These functions are called through function pointer by the Radio low      *
// * level drivers                                                             *
// *                                                                           *
// *****************************************************************************

void OnTxDone   (void) {
    DemoInternalState = APP_TX;
}

void OnRxDone   (void) {
    DemoInternalState = APP_RX;
}

void OnTxTimeout    (void) {
    DemoInternalState = APP_TX_TIMEOUT;
}

void OnRxTimeout    (void) {
    DemoInternalState = APP_RX_TIMEOUT;
}

void OnRxError  (IrqErrorCode_t errorCode ) {
    DemoInternalState = APP_RX_ERROR;
}

void OnRangingDone  (IrqRangingCode_t val) {
    if  (val == IRQ_RANGING_MASTER_VALID_CODE || val == IRQ_RANGING_SLAVE_VALID_CODE)       DemoInternalState = APP_RANGING_DONE;
    else if (val == IRQ_RANGING_MASTER_ERROR_CODE || val == IRQ_RANGING_SLAVE_ERROR_CODE)   DemoInternalState = APP_RANGING_TIMEOUT;
    else                                                                                    DemoInternalState = APP_RANGING_TIMEOUT;
}

void OnCadDone( bool channelActivityDetected )
{
}

uint8_t CRC8(const uint8_t *data,int length) {
   uint8_t crc = 0x00;
   uint8_t extract;
   uint8_t sum;

   for(int i=0;i<length;i++)   {
      extract = *data;
      for (uint8_t tempI = 8; tempI; tempI--)       {
         sum = (crc ^ extract) & 0x01;
         crc >>= 1;
         if (sum)
            crc ^= 0x8C;
         extract >>= 1;
      }
      data++;
   }
   return crc;
}

// LA:  'Extern' LED Drives
//      ===================
//
//  I LED sono indirizzati fisicamente qui.
//  Il modo più rapido per usarli è questo ...
//
void    LedRED  (uint8_t Status) {
    if  (Status != 0)   TX_LED =    1;
    else                TX_LED =    0;
}

void    LedGREEN   (uint8_t Status) {
    if  (Status != 0)   RX_LED =    1;
    else                RX_LED =    0;
}

// LA:  Refresh's Rx Timeout
//      ====================
//
void    pRefreshRxTimeout (void) {

    tTimeInRx.detach ();
    bTimeInRx = true;
//    tTimeInRx.attach_us (&pTimeInRxDone, 1000000+ (100000* (std::rand()/ ((RAND_MAX + 1u)/10))));      // LA:  1000ms+ (0 .. 900ms, step 100ms)
    tTimeInRx.attach_us (&pTimeInRxDone, 1000000+ (std::rand()/ ((RAND_MAX + 1u)/ 500000)) );         // LA:  1000ms+ Random(0 .. 500ms)
}

// LA:  Allo scatenarsi del Ticker fa terminare il tempo concesso alla ricezione
//      ========================================================================
//
void    pTimeInRxDone   (void) {

    bTimeInRx = false;
    tTimeInRx.detach ();
}

// LA:  Refresh's Tx Timeout
//      ====================
//
void    pRefreshTxTimeout (void) {

    tTimeInTx.detach ();
    bTimeInTx = true;
//    tTimeInTx.attach_us (&pTimeInTxDone, 0250000+ (100000* (std::rand()/ ((RAND_MAX + 1u)/10))));     // LA:  0250ms+ (0 .. 900ms, step 100ms)
//    tTimeInTx.attach_us (&pTimeInTxDone, 0250000+ (100000* (std::rand()/ ((RAND_MAX + 1u)/10))));     // LA:  0250ms+ (0 .. 900ms, step 100ms)
    tTimeInTx.attach_us (&pTimeInTxDone, 0250000+ (std::rand()/ ((RAND_MAX + 1u)/ 500000)) );         // LA:  0250ms+ Random(0 .. 500ms)
}

// LA:  Allo scatenarsi del Ticker fa terminare il tempo concesso alla Trasmissione
//      ===========================================================================
//
void    pTimeInTxDone   (void) {

    bTimeInTx = false;
    tTimeInTx.detach ();
}

// LA:  Scansore Temporale TimeOut(s)
//      =============================
//
//  Esegue Ogni Secondo
//
void    p1sWentPast (void) {
    uint8_t ui81sIndex;

    for (ui81sIndex = 0; ui81sIndex < ciMaxActivePartners; ui81sIndex++) {
        if  (ActivePartnersList[ui81sIndex].ui32Address > 0) {
            if  (ActivePartnersList[ui81sIndex].ui8ActivityTimeOut == 0) {

                // LA:  Cancella il Partner x Inattività
                //      ================================
                //
                ActivePartnersList[ui81sIndex].ui32Address =    0;          // LA:  Ora lo Slot è libero
                ActivePartnersList[ui81sIndex].ui8BlackListTime =   0;      // LA:  No BlackList
            }
            else                                                        ActivePartnersList[ui81sIndex].ui8ActivityTimeOut--;    // LA:  1s went past Inactive
            if  (ActivePartnersList[ui81sIndex].ui8BlackListTime > 0)   ActivePartnersList[ui81sIndex].ui8BlackListTime--;      // LA:  1s went past BlackListed
        }
        if  (AnsweringPartnersList[ui81sIndex].ui32Address > 0) {
            if  (AnsweringPartnersList[ui81sIndex].ui8ActivityTimeOut == 0) {

                // LA:  Cancella il Partner x Inattività
                //      ================================
                //
                AnsweringPartnersList[ui81sIndex].ui32Address =    0;          // LA:  Ora lo Slot è libero
            }
            else    AnsweringPartnersList[ui81sIndex].ui8ActivityTimeOut--;    // LA:  1s went past Inactive
        }
    }
}

//extern bool    PageRefreshMaster;   // LA:  Aggiorna LCD, in RadioSLEEP, solo se serve
//extern bool    PageRefreshSlave;    // LA:  Aggiorna LCD, in RadioSLEEP, solo se serve

// LA:  Scansore Temporale LCD
//      ======================
//
//  Esegue Ogni tot
//
void    pLCDRefreshTimeWentPast (void) {
    MenuSetMasterPage   (true);                                     // LA:  xxHz Display Refresh
}