/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "BLE.h"
#include "ble/services/BatteryService.h"
#include "DeviceInformationService.h"
#include <string> 

BLE        ble;
AnalogIn ain(p2); 
DigitalOut led1(LED1);

const static char     DEVICE_NAME[]        = "Battery";
static const uint16_t uuid16_list[]        = {GattService::UUID_BATTERY_SERVICE};
uint8_t pass[6] = {'1', '1', '1', '1', '1', '1'};
uint8_t  currentBattery  = 40;
static volatile bool  triggerSensorPolling = false;

static BatteryService *batteryServicePtr;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    printf("Disconnected!\r\n");
    ble.startAdvertising(); // restart advertising
}

void periodicCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */

    /* Note that the periodicCallback() executes in interrupt context, so it is safer to do
     * heavy-weight sensor polling from the main thread. */
    triggerSensorPolling = true;
}

void connectionCallback(const Gap::ConnectionCallbackParams_t *params)
{
    printf("Connected!\r\n");
}

void passkeyDisplayCallback(Gap::Handle_t handle, const SecurityManager::Passkey_t passkey)
{
    printf("Input passKey: ");
    for (unsigned i = 0; i < Gap::ADDR_LEN; i++) {
        printf("%c ", passkey[i]);
    }
    printf("\r\n");
}

void securitySetupCompletedCallback(Gap::Handle_t handle, SecurityManager::SecurityCompletionStatus_t status)
{
    if (status == SecurityManager::SEC_STATUS_SUCCESS) {
        printf("Security success\r\n");
    } else {
        printf("Security failed\r\n");
    }
}

int main(void)
{
    unsigned long dev1 = (unsigned long)NRF_FICR->DEVICEID[0];
    printf("DeviceID_1: ");
    printf("%lu", dev1);
    unsigned long dev2 = (unsigned long)NRF_FICR->DEVICEID[1];
    printf(" DeviceID_2: ");
    printf("%lu\n", dev2);
    unsigned long dev3 = (unsigned long)NRF_FICR->DEVICEADDR[0];
    printf("DeviceAddress_1: ");
    printf("%lu", dev3);
    unsigned long dev4 = (unsigned long)NRF_FICR->DEVICEADDR[1];
    printf(" DeviceAddress_2: ");
    printf("%lu\n", dev4);
    
    unsigned long devaddr = dev1 + dev2 + dev3 + dev4;
    printf("DeviceID: ");
    printf("%lu\n", devaddr);
       
    for (int i = 0; i < 6; i++){
        int k = devaddr % 10;
        devaddr = devaddr / 10;
        pass[5-i] = k + '0';
    }
    
    printf("passKey: ");
    for (unsigned i = 0; i < Gap::ADDR_LEN; i++) {
        printf("%c ", pass[i]);
    }
    printf("\r\n");

    
    led1 = 1;
    Ticker ticker;
    ticker.attach(periodicCallback, 1); // blink LED every second
    
    /* Initialize BLE module */
    ble.init();
    
    /* Initialize BLE security */
    bool enableBonding = true;
    bool requireMITM   = true;
    ble.securityManager().init(enableBonding, requireMITM, SecurityManager::IO_CAPS_DISPLAY_ONLY, pass);
    
    /* Set callback functions */
    ble.gap().onConnection(connectionCallback);
    ble.gap().onDisconnection(disconnectionCallback);
    ble.securityManager().onPasskeyDisplay(passkeyDisplayCallback);
    ble.securityManager().onSecuritySetupCompleted(securitySetupCompletedCallback);
    
    batteryServicePtr = new BatteryService(ble, currentBattery);


    /* Setup auxiliary service. */
    DeviceInformationService deviceInfo(ble, "ARM", "Model1", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");

    /* Setup advertising. */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    //ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_HEART_RATE_SENSOR);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000);
    
    ble.gap().startAdvertising();

    // infinite loop
    while (1) {
        // check for trigger from periodicCallback()
        if (triggerSensorPolling && ble.getGapState().connected) {
            triggerSensorPolling = false;

           currentBattery = ain.read()*100;
            batteryServicePtr->updateBatteryLevel(currentBattery);
            
        } else {
            ble.waitForEvent(); // low power wait for event
        }
    }
}