/*
 * mbed Microcontroller Library
 * Copyright (c) 2006-2018 ARM Limited
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Based on mbed-stress-test by Marcus Chang @ Arm Mbed - http://github.com/ARMmbed/mbed-stress-test
*/

#include "mbed.h"
#include "utest/utest.h"
#include "unity/unity.h"
#include "greentea-client/test_env.h"
#include "common_defines_test.h"
#include "download_test.h"
#include <string>

#ifdef MBED_CONF_APP_BASICS_TEST_FILENAME
  #include MBED_CONF_APP_BASICS_TEST_FILENAME
#else
  #include "alice.h"
#endif

#ifndef MBED_CONF_APP_TESTS_FS_SIZE
  #define MBED_CONF_APP_TESTS_FS_SIZE (2*1024*1024)
#endif

using namespace utest::v1;

#if !defined(MBED_CONF_APP_NO_LED)
DigitalOut led1(LED1);
DigitalOut led2(LED2);
void led_thread() {
    led1 = !led1;
    led2 = !led1;
}
#endif

#define MAX_RETRIES 3
NetworkInterface* net = NULL;
static control_t setup_network(const size_t call_count) {
    net = NetworkInterface::get_default_instance();
    TEST_ASSERT_NOT_NULL_MESSAGE(net, "failed to initialize network");

    nsapi_error_t err = -1;
    for (int tries = 0; tries < MAX_RETRIES; tries++) {
        err = net->connect();
        if (err == NSAPI_ERROR_OK) {
            break;
        } else {
            printf("[ERROR] Connecting to network. Retrying %d of %d.\r\n", tries, MAX_RETRIES);
        }
    }
    TEST_ASSERT_EQUAL(NSAPI_ERROR_OK, err);
    printf("[NET] IP address is '%s'\n", net->get_ip_address());
    printf("[NET] MAC address is '%s'\n", net->get_mac_address());
    return CaseNext;
}

static uint32_t thread_counter = 0;
void download_fn() {
    uint32_t thread_id = core_util_atomic_incr_u32(&thread_counter, 1);
    download_test(net, story, sizeof(story), 1024, thread_id);
}

static control_t download_1_thread(const size_t call_count) {
    thread_counter = 0;
 
    Thread t1;
    t1.start(download_fn);
    t1.join();
 
    return CaseNext;
}

static control_t download_2_threads(const size_t call_count) {
    thread_counter = 0;

    Thread t1;
    Thread t2;
    t1.start(download_fn);
    wait(0.5);
    t2.start(download_fn);
    t2.join();
    t1.join();
 
    return CaseNext;
}

static control_t download_3_threads(const size_t call_count) {
    thread_counter = 0;

    Thread t1;
    Thread t2;
    Thread t3;
    t1.start(download_fn);
    t2.start(download_fn);
    t3.start(download_fn);
    t1.join();
    t2.join();
    t3.join();
 
    return CaseNext;
}

static control_t download_4_threads(const size_t call_count) {
    thread_counter = 0;

    Thread t1;
    Thread t2;
    Thread t3;
    Thread t4;
    t1.start(download_fn);
    t2.start(download_fn);
    t3.start(download_fn);
    t4.start(download_fn);
    t1.join();
    t2.join();
    t3.join();
    t4.join();
 
    return CaseNext;
}

utest::v1::status_t greentea_setup(const size_t number_of_cases) {
    GREENTEA_SETUP(10*60, "default_auto");
    return greentea_test_setup_handler(number_of_cases);
}

Case cases[] = {
    Case(TEST_NETWORK_TYPE " network setup", setup_network),
    Case(TEST_NETWORK_TYPE " 1 thread", download_1_thread),
#if MBED_CONF_TARGET_NETWORK_DEFAULT_INTERFACE_TYPE != CELLULAR
    Case(TEST_NETWORK_TYPE " 2 threads", download_2_threads),
#endif
    //Case(TEST_NETWORK_TYPE " 3 threads", download_3_threads),
    // 4 threads may fail due to http host limits
    //Case(TEST_NETWORK_TYPE " 4 threads", download_4_threads),
};

Specification specification(greentea_setup, cases);

int main() {
    //Create a thread to blink an LED and signal that the device is alive
#if !defined(MBED_CONF_APP_NO_LED)
    Ticker t;
    t.attach(led_thread, 0.5);
#endif

    return !Harness::run(specification);
}
