/* Copyright (c) 2013 Henry Leinen (henry[dot]leinen [at] online [dot] de)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#ifndef __UDPSOCKET_H__
#define __UDPSOCKET_H__

#include "Socket/Socket.h"
#include "Socket/Endpoint.h"

/** Class UDPSocket, inherits from class Socket and implements the functionality of a UDP Socket when using a WifiPlusClick module.
  * @note Please note that the functionality provided by a WifiPlusClick module is limited. Neither broadcasting nor multicasting functionality is possible.
  */
class UDPSocket : public Socket
{
public:
    /** UDPSocket constructor to instantiate a UDPSocket object.  */
    UDPSocket();
    
    /** Function init provides initialization of the UDPSocket object.
      * @returns : true if successfull, or false otherwise.
      */
    int init(void);
    
    /** Function bind will bind a UDPSocket to a local port. The socket object should not already have been used for other purposes before.
      * @note Please note that there is currently no check implemented as to wheather or not the socket has already been used before.
      * @parameter port : The port to which to bind the UDPSocket to. If the port is specified as zero, a non-zero value beginning with 1024 will be selected.
      * @returns : 0 if successfull, or -1 otherwise.
      */
    int bind(unsigned int port = 0);
    
    /** Function join_multicast_group.
      * @note This function is not implemented as the WifiPlusClick module does not support this functionality.
      */
    int join_multicast_group(const char *address);
    
    /** Function set_broadcasting.
      * @note This fnction is not implemented as the WifiPlusClick module does not support this functionality.
      */
    int set_broadcasting(bool broadcast=true);
    
    /** Function sendTo implements sending of a packet to a specific endpoint. Make sure to use it on an initialized or bound socket.
      * @param remote : a reference to a valid endpoint specifying the remote ip-address and the remote port where to send the packet.
      * @param packet : a pointer to a valid buffer containing the packet data to send.
      * @param length : Specifies the number of data bytes to send.
      * @returns : the number of databytes actually sent or -1 on failure.
      */
    int sendTo(Endpoint &remote, char *packet, int length);
    
    /** Function receiveFrom implements receiving a packet of data on a socket. The remote address will be provided on return. Make sure to use it on an initialized or bound socket.
      * @param remote : a reference to an endpoint, which will receive the remote ip-address and the remote port from where data was returned.
      * @param packet : a data buffer which will receive the received data. The buffer must be at least as larget as length bytes.
      * @param length : The maximum number of bytes to receive. Please make sure that the buffer packet is large enough to hold this data completely.
      * @returns : the number of databytes actually received or -1 on failure. @note there may be more data received than what the buffer can store. Any data that does not fit in the buffer is discarded !
      */
    int receiveFrom(Endpoint &remote, char *packet, int length);
};

#endif // __UDPSOCKET_H__