/* Copyright (c) 2013 Henry Leinen (henry[dot]leinen [at] online [dot] de)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#ifndef __WIFI_H__
#define __WIFI_H__

#include "mbed.h"

typedef enum {
    //  Control Messages
    RESET_MSG   = 170,  /*!< RESET_MSG */
    GET_VERSION_MSG = 23, /*!< GET_VERSION_MSG */
    GPIO_MSG = 172,
    //  Network Configuration Messages
    SET_IP_ADDRESS_MSG = 41,
    SET_NETWORK_MASK_MSG = 42,
    SET_GATEWAY_IP_ADDRESS_MSG = 44,
    GET_NETWORK_STATUS_MSG = 48,
    SET_MACADDRESS_MSG = 49,
    SET_ARP_TIME_MSG = 173,
    //  Wi-Fi General Configuration Messages
    SET_CP_NETWORK_MODE_MSG = 55,
    SET_CP_SSID_MSG = 57,
    SET_REGIONAL_DOMAIN_MSG = 56,
    SET_CHANNEL_LIST_MSG = 58,
    SET_LIST_RETRY_COUNT_MSG = 59,
    //  Wi-Fi Power Management Messages
    SET_POWER_SAVE_MODE_MSG = 102,
    //  Wi-Fi Security Configuration Messages
    SET_CP_SECURITY_OPEN_MSG = 65,
    SET_CP_SECURITY_WEP40_MSG = 66,
    SET_CP_SECURITY_WEP104_MSG = 67,
    SET_CP_SECURITY_WPA_MSG = 68,
    GET_CP_WPAKEY_MSG = 71,
    //  Wi-Fi Scanning Messages
    SCAN_START_MSG = 80,
    SCAN_GET_RESULTS_MSG = 81,
    //  Wi-Fi Connection Messages
    WIFI_CONNECT_MSG = 90,
    WIFI_DISCONNECT_MSG = 91,
    //  ICMP (Ping) Messages
    PING_SEND_MSG = 121,
    //  Socket Messages
    SOCKET_CREATE_MSG = 110,
    SOCKET_CLOSE_MSG = 111,
    SOCKET_BIND_MSG = 112,
    SOCKET_CONNECT_MSG = 113,
    SOCKET_LISTEN_MSG = 114,
    SOCKET_ACCEPT_MSG = 115,
    SOCKET_SEND_MSG = 116,
    SOCKET_RECV_MSG = 117,
    SOCKET_SEND_TO_MSG = 118,
    SOCKET_RECV_FROM_MSG = 119,
    SOCKET_ALLOCATE_MSG = 122,

} CMD_MSG_t;


typedef enum {
    ACK_MSG = 0,
    EVENT_MSG = 1,
    
    //  Control Message Responses
    GPIO_RESPONSE_MSG = 50,
    //  Network Configuration Message Responses
    NETWORK_STATUS_RESPONSE_MSG = 48,
    //  Wi-Fi Security Configuration Message Responses
    WPAKEY_RESPONSE_MSG = 0x31,
    //  Wi-Fi Scanning Message Responses
    SCAN_RESULT_MSG = 22,
    //  Socket Message Responses
    SOCKET_CREATE_RESPONSE_MSG = 23,
    SOCKET_BIND_RESPONSE_MSG = 24,
    SOCKET_CONNECT_RESPONSE_MSG = 25,
    SOCKET_LISTEN_RESPONSE_MSG = 26,
    SOCKET_ACCEPT_RESPONSE_MSG = 27,
    SOCKET_SEND_RESPONSE_MSG = 28,
    SOCKET_RECV_RESPONSE_MSG = 29,
    SOCKET_SEND_TO_RESPONSE_MSG = 30,
    SOCKET_RECV_FROM_RESPONSE_MSG = 31,
    SOCKET_ALLOCATE_RESPONSE_MSG = 32,
        
} RSP_MSG_t;

typedef enum {
    Event_IP_Address_Assigned = 16,
    Event_WiFi_Connection_Status_Changed = 8,
    Event_WiFi_Scan_Results_Ready = 9,
    Event_Ping_Response = 26,
    Event_Error_Event = 255,
    Event_Startup_Event = 27,
    
} EVT_MSG_t;


typedef enum {
    GPIO0 = 0,
    GPIO1,
    GPIO2,
    GPIO3,
    GPIO4,
    GPIO5,
    GPIO6,
    GPIO7
} GPIO_IDX_t;

typedef enum {
    GPIO_OP_OUTPUT_LOW = 0,
    GPIO_OP_OUTPUT_HIGH = 1,
    GPIO_OP_READ_INPUT=2
} GPIO_OP_t;

typedef enum {
    GPIO_RES_OUTPUT_LOW = 0,
    GPIO_RES_OUTPUT_HIGH= 1,
    GPIO_RES_INPUT_LOW  = 2,
    GPIO_RES_INPUT_HIGH = 3,
    GPIO_RES_INVALID_INDEX= 255
} GPIO_RES_t;

typedef struct {
    union {
        struct {
            char o1, o2, o3, o4;
        };
        char o[4];
    } sin_addr;
    operator char* () { return (char*)this; }
} IPADDRESS_t;

typedef enum {
    NoFailure=0,
    
    JoinFailure=2,
    AuthenticationFailure,
    AssociationFailure,
    WEPHandshakeFailure,
    PSKCalculationFailure,
    PSKHandshakeFailure,
    AdHocJoinFailure,
    SecurityMismatchFailure,
    NoSuitableAPFoundFailure,
    RetryForeverNotSupportedFailure,
    
    BeaconTimeOut=21,
    DeauthReceivedDE,
    DisassociateReceived
} CONN_ERROR_t;

typedef enum {
    NotConnected_StaticIP = 0,
    Connected_StaticIP = 1,
    NotConnected_DHCP = 2,
    Connected_DHCP = 3
} NET_STAT_t;

typedef enum {
    INFRASTRUCTURE = 1,
    ADHOC = 2
} NETW_MODE_t;

typedef enum {
    NoError = 0,
    BaudRateGeneratorError = 60,
    InvalidConnectionProfileError = 61,
    WiFiAlreadyConnectedError = 62,
    WiFiAlreadyDisconnectedError = 63,
    CloseSocketFailedError = 64,
    SocketSendToTimeOutError = 65,
    ScanIndexOutOfRangeError = 66,
    ICMPPingFloodError = 67,
    ICMPPingInUseError = 68,
    SocketRecvFromError = 69,
    SerialTransmitBufferAllocationError = 71,
    GeneralAssertError = 72,
    InvalidPowersaveModeError = 73,
    BusyInHibernateModeError = 74,
    BusyInScanModeError = 75,
    Unknown = 0xFFFF
}  ERR_t;

typedef enum {
    WPA_PSK = 3,
    WPA_PASSPHRASE = 4,
    WPA2_PSK = 5,
    WPA2_PASSPHRASE = 6,
    WPA_OR_WPA2_PSK = 7,
    WPA_OR_WPA2_PASSPHRASE = 8
} WPA_SECURITY_t;

typedef __packed struct {
    char IE:1;
    char res:3;
    char Privacy:1;
    char Preamble:1;
    char WPA:1;
    char WPA2:1;
} AP_CONFIG_t;

typedef enum {
    FCC = 0,
    IC,
    ETSI,
    SPAIN,
    FRANCE,
    JAPANA,
    JAPANB
} DOMAIN_COUNTRY_CODE_t;

typedef enum {
    HIBERNATE = 1,          // Radio off
    PSPOLL_AP_SUGGESTED_DTIM = 2,  
    PSPOLL_SELF_SET_DTIM = 3,
    FULLY_POWERED = 4
} POWERSAVE_MODE_t;

typedef enum {
    InvalidSocketHandle=254,
    UnknownSocketType=255
} SOCKET_HANDLE_t;

typedef enum {
    UDP=0,
    TCP=1
} SOCKET_TYPE_t;



/** Class Wifi encapsulates communication and access to Wifi Plus Click module. */
/*!\copyright { Henry Leinen } */
class Wifi
{
public:
    typedef enum {
        Idle,                  
        Header1_received,           
        Header2_received,           
        MessageType_low_received,
        MessageType_high_received,
        DataLen_low_received,
        Data_receiving,
        Awaiting_trailer
    } RX_STATE_t;

protected:
    Serial          m_wifi;
    DigitalOut      m_reset;
    char*           m_buffer;
    
    bool            m_msgReceived;
    RSP_MSG_t       m_msgResponse;
    short           m_msgDataLen;
    bool            m_ackReceived;

//  Event data
    IPADDRESS_t     m_ipAddress;
    bool            m_bWifiConnected;
    CONN_ERROR_t    m_WifiConnectionError;
    
    ERR_t           m_lastError;

    bool            m_bScanResultsReady;
    char            m_NumLastScanResults;
    
public:
    static Wifi*     _Wifi;
    static Wifi*    getInstance()
    {
        if (_Wifi==NULL) {
            error("NOWIFI");
        }
        return _Wifi;
    }
                
public:
    Wifi(PinName tx, PinName rx, PinName rst);

    bool AwaitScanResults(int timeout) {
        Timer t;
        t.start();
        while(t.read() < timeout) {
            if (m_bScanResultsReady) {
                return true;
            }
        }
        return false;
    }

    bool AwaitConnected(int timeout, ERR_t *err = NULL) {
        Timer t;
        t.start();
        while(t.read() < timeout) {
            if (m_bWifiConnected) {
                return true;
            }
            wait(0.1);
        }
        if (err)
            *err = m_lastError;
        return false;
    }        
    
    bool AwaitDisconnected(int timeout, ERR_t *err = NULL) {
        Timer t;
        t.start();
        while(t.read() < timeout) {
            if (!m_bWifiConnected) {
                return true;
            }
            wait(0.1);
        }
        if (err)
            *err = m_lastError;
        return false;
    }
    
    void Reset();

    /** Function to reset the Wifi Plus Click Device. PLEASE NOTE, it will loose all its current settings
      * @returns true if successful or false otherwise 
      */
    bool ResetMsg();
    /** Function retrieves the software and radio version of the Wifi Plus Click Board.
      * @param sw_ver_lo : receives the low WiComm Socket Version number
      * @param sw_ver_hi : recevies the high WiComm Socket Version number
      * @param rad_ver_lo: receives the low Radio version number
      * @param rad_ver_hi: receives the high Radio version number
      * @returns true if successful or false otherwise
      */
    bool GetVersionMsg(char &sw_ver_lo, char &sw_ver_hi, char &rad_ver_lo, char &rad_ver_hi);
    /** Function will set a GPIO pin to a specific output level or will set it to input.
      * The function will return the result of the operation.
      * @param pin : Reference to the GPIO pin of interest
      * @param operation: the operation to perform on the GPIO pin. Set to specific output level or read as input
      * @param result : a reference to a variable which receives the actual status of the GPIO pin
      */
    bool GpioMsg(GPIO_IDX_t pin, GPIO_OP_t operation, GPIO_RES_t &result);
    
    /** Function to set the IP address of the Wifi Plus Click device.
      * @param bUseDHCP : set to true if DHCP shall be used, or set to false if you want to specify a static IP address
      * @param IPAddress: pointer to an IP address.
      *                   Please note that the IP adress will only be used if bUseDHCP is set to false
      * @returns : true if successfull, or false otherwise. The function will not check for any asynchroneous result.
      */
    bool SetIpAddress(bool bUseDHCP, IPADDRESS_t *IPAddress);
    /** Function to set the network mask of the Wifi Plus Click device.
      * @param NetworkMask : the Network mask to use.
      * @returns : true if successfull, or false otherwise.
      */
    bool SetSubnetMask(IPADDRESS_t *NetworkMask);
    /** Function to set the gateway ip address to use. If the gateway ip address is set to 0.0.0.0 the Wifi Plus Click will not use
      * a gateway.
      * @param IPAddress : Pointer to an IPADDRESS specifying the IP Address of the gateway to use. 
      * @returns : true if successfull, or false otherwise.
      */
    bool SetGatewayIpAddress(IPADDRESS_t *IPAddress);
    /** Function to retrieve the current network status of the Wifi Plus Click device.
      * @param MacAddress : pointer to an array of 6 Bytes to retrieve the MAC Address. Set to NULL if not required.
      * @param IPAddress  : pointer to an IPADDRESS_t structure which receives the IP address. Set to NULL if not required.
      * @oaram NetworkMask : pointer to an IPADDRESS_t structure which receives the network mask. Set to NULL if not requried.
      * @param GatewayAddress : pointer to an IPADDRESS_t structure which receives the gateway address. Set to NULL if no required.
      * @param stat : will receive the actual status.
      * @returns : true if successfull, or false otherwise.
      */
    bool GetNetworkStatus(char *MacAddress, IPADDRESS_t *IPAddress, IPADDRESS_t *NetworkMask, IPADDRESS_t *GatewayAddress, NET_STAT_t &stat);
    /** Function to set the MAC Address of the device. This command should only be used during initialization
      * @param MacAddress : the MAC address to use.
      * @returns : true if successfull, or false otherwise.
      */
    bool SetMACAddress(char MacAddress[6]);
    /** Function to set the address resolution time from 1 to 65535 seconds.
      * @param ARPTime : either 0 to disable ARP or time in seconds.
      * @returns : true if successfull, or false otherwise.
      */
    bool SetARPTime(unsigned short ARPTime);
    /** Function to set the network mode for one of two available profiles.
      * @param Profile : can be profile number one or 2
      * @param NetMode : can be either INFRASTRUCTURE or ADHOC
      * @returns : true if successfull, or false otherwise.
      */
    bool SetNetworkMode(char Profile, NETW_MODE_t NetMode);
    /** Function to set the SSID for one of two profiles.
      * @param Profile : can be profile number one or two.
      * @param ssid : a string descibing the ssid.
      * @returns : true if successfull, or false otherwise.
      */
    bool SetSSID(char Profile, const char* ssid);
    /** Function switches device into one of 3 power saving modes, or sets it into awake mode.
      * @param pwrsave : specifies which powersaving mode to set.
      * @param DTIM_Listen : if pwrsave is in mode PSPOLL_SELF_DTIM uses this value as the DTIM listen interval. Value is given in units of 100ms.
      * @returns : true if successfull, or false otherwise
      */
    bool SetPowerSaveMode(POWERSAVE_MODE_t pwrsave, short DTIM_Listen);
    /** Function to set the regional domain for all CPs.
      * @param dcode : domain country code valid for all CPs.
      * @returns : true if successfull, or false otherwise.
      */
    bool SetRegionalDomain(DOMAIN_COUNTRY_CODE_t dcode);
    /** Function to set the channels on which to scan.
      * @param numListItems : the number of channels given in the list.
      * @param ListItems : array consisting of numListItems channel numbers.
      * @returns : true if successfull, or false otherwise
      */
    bool SetChannelList(char numListItems, char ListItems[]);
    /** Function to specify the connection managaer retry count. There are separate values for infrastructure and ad hoc mode.
      * Please note that a number of 0 means no retries and 255 means retry forever.
      * @param infrastructureRetryCount : number of retries in infrastructure mode
      * @param adhocRetryCount : number of retries in ad-hoc mode.
      * @returns : true if successfull, or false otherwise
      */
    bool SetRetryCount(char infrastructureRetryCount=255, char adhocRetryCount=5);
    /** Function to set the Security mode to OPN 
      * @param Profile : can be profile number one or two.
      * @returns : true if successfull, or false otherwise.
      */
    bool SetSecurityOpen(char Profile);
    /** Function to set the WEP40 security mode.
      * @param Profile : can be profile number one or two.
      * @param bShareKey : set to true for shared key, or false for open key
      * @param DefaultWEPKeyIdx : the index (0-3) into the key array
      * @param SecurityKeys : array or 4 security keys with 5 bytes each
      * @returns : true if successfull, or false otherwise
      */
    bool SetSecurityWEP40(char Profile, bool bSharedKey, char DefaultWEPKeyIdx, char SecurityKeys[20]);
    /** Function to set the WEP104 security mode.
      * @param Profile : can be profile number one or two.
      * @param bShareKey : set to true for shared key, or false for open key
      * @param DefaultWEPKeyIdx : the index (0-3) into the key array
      * @param SecurityKeys : array or 4 security keys with 13 bytes each
      * @returns : true if successfull, or false otherwise
      */
    bool SetSecurityWEP104(char Profile, bool bSharedKey, char DefaultWEPKeyIdx, char SecurityKeys[52]);
    /** Function to set the WPA security mode.
      * @param Profile : can be profile number one or two.
      * @param sec : can be one of the define security types.
      * @param len : length of the security key or passphrase.
      * @param secKeyOrPSK : security key or passphrase.
      * @returns : true if successfull, or false otherwise.
      */
    bool SetSecurityWPA(char Profile, WPA_SECURITY_t sec, int len, const char* secKeyOrPSK);
    /** Function to retrieve the WPA Key from the Wifi Plus Click device for a selected profile.
      * @param Profile : can be profile number one or two.
      * @param Key : will receive the key.
      * @returns : true if successfull, or false otherwise.
      */
    bool GetWPAKey(char Profile, char Key[32]);
    /** Function to initiate the scan on one or both Profiles using the settings downloaded to the Wifi Plus Click.
      * Please NOTE that the device will only accept the REBOOT message while a scan is active. A scan can only be
      * started if the device is not connected.
      * @param Profile : can be profile number one or two.
      * @returns : true if successfull, or false otherwise.
      */
    bool ScanStartMsg(char Profile);
    /** Function to retrieve a specific scan result. Will only work after scan has been started and finished.
      * @param index : specify the result item to return. When this index is too high the function returns with a false return value.
      * @param ssid : will receive the SSID of the item.
      * @param apCfg : will receive the access points configuration.
      * @param beaconInterval : will receive the beacon interval time.
      * @param ATIMWindow : will receive the ATIM Window.
      * @param RSSI : will receive the RSSI level.
      * @param bssType : will receive information about the BSS type (INFRASTRUCTURE oR ADHOC).
      * @param channelNo : will receive on which channel the AP transmits.
      * @returns : true if successfull, false otherwise.
      */
    bool ScanGetResults(char index, char ssid[32], AP_CONFIG_t &apCfg, short &beaconInterval, short &ATIMWindow, char &RSSI, NETW_MODE_t &bssType, char &channelNo);
    
    /** Function to connect using the specified profile and the settings previously downloaded to the device. Please Note that the connection
      * state will be transmitted asynchroneously be the device. You need to monitor the connected state.
      * @param Profile : can be profile number one or two.
      * @returns : true if successfully submitted, or false otherwise.
      */
    bool Connect(char Profile);
    /** Function to disconnect from any connected AP.
      * @returns : true if successful, or false otherwise.
      */
    bool Disconnect();
    
    
    /** @name Socket Functions
        In the following section, all socket relevant functions are contained.
       */
    ///@{
    
    /** Function to allocate the internal memory of the device and the number of sockets (which in total is limited to 8).
      * @param nTCPSvr : Number of server sockets to allocate.
      * @param nTCPClnt: Number of client sockets to allocate.
      * @param TCPSvrRxBuf : Size (in bytes) to allocate for the receive buffer for each server socket.
      * @param TCPSvrTxBuf : Size (in bytes) to allocate for the transmit buffer for each server socket.
      * @param TCPClntRxBuf: Size (in bytes) to allocate for the receive buffer for each client socket.
      * @param TCPClntTxBuf: Size (in bytes) to allocate for the transmit buffer for each client socket.
      * @returns : true if successful, or false otherwise. Can return false if the requestet internal memory exceededs the total of 8192 internal byte orif the maximum number of
      * 8 sockets is exceeded.
      */
    bool SocketAllocate(char nTCPSvr, char nTCPClnt, unsigned short TCPSvrRxBuf, unsigned short TCPSvrTxBuf, unsigned short TCPClntRxBuf, unsigned short TCPClntTxBuf);
    
    /** Function to create a new socket object of the given type.
      * @param sockType : Specifies the type of socket to create (UDP or TCP).
      * @returns : Either a valid socket handle or InvalidSocketHandle or UnknownSocketType.
      */
    SOCKET_HANDLE_t SocketCreate( SOCKET_TYPE_t sockType );
    /** Function to close a previously created socket.
      * @param hSock: Valid socket handle, which has been created by a previous call to SocketCreate.
      * @returns : true if successful, or false if an error occured
      */
    bool SocketClose(SOCKET_HANDLE_t hSock);
    /** Function to bind to a specific port using the given socket.
      * @param hSock : Valid socket handle.
      * @param Port  : local port to bind to.
      * @returns : true if successfull, or false otherwise.
      */
    bool SocketBind(SOCKET_HANDLE_t hSock, int Port);
    /** Function to prepare the a number of sockets for accepting remote connections 
      * @param hSock : Valid socket handle.
      * @param Backlog : Number of sockets to set in listening mode, will receive the number of new unassigned backlog count
      * @return : true if successfull, or false otherwise.
      */
    bool SocketListen(SOCKET_HANDLE_t hSock, int &Backlog);
    /** Function to accept a new remote connection. Will work asynchroneously, means if no new connection was accepted, will return immediately.
      * @param hSock : Valid socket handle.
      * @param client : reference to a variable which receives the new socket handle of the connection socket or InvalidSocketHandle if no connection could be made at this time.
      * @param remotePort : receives the remote port of the remote connection.
      * @param remoteAddress : receives the remote address of the remote connection.
      * @returns : true if successfull, or false otherwise. PLEASE NOTE that even if no connection was made, the return result may be true.
      */
    bool SocketAccept(SOCKET_HANDLE_t hSock, SOCKET_HANDLE_t &client, int &remotePort, IPADDRESS_t &remoteAddress);
    /** Function to connect the socket to a remote socket specified by IpAddress and Port.
      * @param hSock : Valid socket handle.
      * @param IpAddress : valid IP Address of the remote host to connect with.
      * @param Port : Port of the remote host to connect with.
      * @returns : true if a connection has been made successfully or false otherwise. 
      */
    SOCKET_HANDLE_t SocketConnect(SOCKET_HANDLE_t hSock, IPADDRESS_t *IpAddress, int Port);    
    /** Function to send data on the connected socket
      * @param hSock : Valid socket handle.
      * @param data : Pointer to a buffer containing the data to send.
      * @param length : Number of bytes in the buffer to send.
      * @returns : Number of bytes actually sent over the connection.
      */
    int SocketSend(SOCKET_HANDLE_t hSock, char* data, int length);
    /** Function to receive data from the connected socket.
      * @param hSock : Valid socket handle of a socket which is already connected.
      * @param data : Pointer to a valid buffer which shall receive the data.
      * @param length : Maximum number of bytes which can be received and stored in the provided buffer.
      * @returns : Number of bytes actually received over the connection or -1 if a fault occured.
      */
    int SocketRecv(SOCKET_HANDLE_t hSock, char* data, int length);
    /** Function to send data over a connectionless socket.
      * @param hSock : Valid socket handle.
      * @param remoteAddress : The ip address of the remote destination to send the data to.
      * @param remotePort : The port of the remote destination to send the data to.
      * @param data : Pointer to a buffer which contains the data to send.
      * @param length : The number of bytes to sent to the specified address.
      * @returns : Number of bytes actually sent or -1 if a fault occured.
      */
    int SocketSendTo(SOCKET_HANDLE_t hSock, IPADDRESS_t* remoteAddress, int remotePort, char *data, int length);
    /** Function to receive data over a connectionless socket.
      * @param hSock : Valid socket handle.
      * @param remoteAddress : The ip address of the remote origin from where to receive the data.
      * @param remotePort : Point to receive the port of the remote origin from where to receive the data or NULL if not needed.
      * @param data : Pointer to a buffer which will receive the data.
      * @param length : Number of bytes to receive and which fit into the provided buffer.
      * @returns : Number of bytes actually received or -1 if a fault occured.
      */
    int SocketRecvFrom(SOCKET_HANDLE_t hSock, IPADDRESS_t *remoteAddress, int *remotePort, char *data, int length);
    
    ///@}
    
    
    /** This is the main cyclic function, which needs to be called as often as possible.
     * The function will check if a new message has been received and will decode and dispatch it.
     */        
    void Poll();
    
    
    int gethostbyname(const char* host, IPADDRESS_t *IpAddress);
    int convert(const char *hostip, IPADDRESS_t *IpAddress);
protected:    
    void SendCommand(CMD_MSG_t cmd, int msgLen1, char *pData1, int msgLen2 = 0, char* pData2 = NULL);
    void rx_int();
    
    void SubmitResponse(RSP_MSG_t rsp, int msgLen);
    bool WaitMessage(RSP_MSG_t msg, int timeout);
    bool WaitEvent(EVT_MSG_t evt, int timeout);
    bool WaitAck(int timeout);
};

#endif //__WIFI_H__