/* Copyright (c) 2013 Henry Leinen (henry[dot]leinen [at] online [dot] de)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#ifndef __TCPSOCKETCONNECTION_H__
#define __TCPSOCKETCONNECTION_H__

#include "Socket.h"
#include "Endpoint.h"

/** Class TCPSocketConnection inherits from Socket and Endpoint. Implements the client TCP socket functionality.
  */
class TCPSocketConnection : public Socket, public Endpoint
{
    friend class TCPSocketServer;
public:
    /** Public constructor to instantiate an object of class TCPSocketConnection */
    TCPSocketConnection();
    
    /** Function connect allows to create an unused socket object to a remote host on a remote port.
      * @param host : remote host to which to initiate a TCP connection.
      * @param port : remote port to which to initiate a TCP connection.
      * @returns : 0 if successfull or -1 on failure.
      */
    int connect(const char *host, const int port);
    
    /** Function to query wheather the socket is already connected with a remote instance.
      * @returns : true if already connected or false otherwise.
      */
    bool is_connected(void);
    
    /** Function send allows to send data to an already connected TCP socket.
      * @param data : pointer to a valid data buffer.
      * @param length : Number of bytes from data buffer to send.
      * @returns : number of bytes actually sent or -1 on failure.
      */
    int send(char *data, int length);
    
    /** Function send_all allows to send data to an already connected TCP socket. The function will try to send all the data specified by length.
      * If a timeout occurs in a blocking configuration the function returns the number of bytes actually sent.
      * @param data : pointer to a valid data buffer.
      * @param length : Number of bytes from data buffer to send.
      * @returns : number of bytes actually send or -1 on failure.
      */
    int send_all(char *data, int length);
    
    /** Function recv allows to receive data from an already connected TCP socket. Please note that this function will return once a first bunch
      * of data has been received. It will not wait until all the requested data is available.
      * @param data : pointer to a valid data buffer which will receive the received data. The buffer must be big enough to hold all the requested data.
      * @param length : Maximum numnber of bytes to receive.
      * @returns : number of bytes actually received and stored in data buffer or -1 on failure.
      */
    int recv(char *data, int length);
    
    /** Function recv_all allows to receive data entirely from an already connected TCP socket. Please note that this function will only return if
      * all the requested data was received or a timeout occured.
      * @param data : pointer to a valid data buffer which will receive the received data. The buffer must be big enough to hold all the requested data.
      * @param length : Number of bytes to receive.
      * @returns : number of bytes actually received and stored in data buffer or -1 on failure.
      */
    int recv_all(char * data, int length);
    
private:
    bool _is_connected;
};

#endif // __TCPSOCKETCONNECTION_H__