/* Copyright (c) 2013 Henry Leinen (henry[dot]leinen [at] online [dot] de)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "mbed.h"
#include "TCPSocketConnection.h"

TCPSocketConnection::TCPSocketConnection() 
    : _is_connected(false)
{
}

int TCPSocketConnection::connect(const char *host, const int port)
{
    _socket = Wifi::getInstance()->SocketCreate(TCP);
    if (_socket == InvalidSocketHandle)
        return -1;
        
    if (set_address(host, port) != 0) 
        return -1;
        
    if (!Wifi::getInstance()->SocketConnect(_socket, reinterpret_cast<IPADDRESS_t*>(get_address()), port)) {
        close();
        return -1;
    }
    _is_connected = true;
    
    return 0;
}

bool TCPSocketConnection::is_connected()
{
    return _is_connected;
}

int TCPSocketConnection::send(char *data, int length)
{
    if ((_socket == InvalidSocketHandle) || !_is_connected)
        return -1;
        
    int n = Wifi::getInstance()->SocketSend(_socket, data, length);
    
    return n;
}

int TCPSocketConnection::send_all(char *data, int length)
{
    if ((_socket == InvalidSocketHandle) || !_is_connected)
        return -1;
        
    int n = 0;
    
    Timer t;
    t.start();
    
    while( n < length) {
        int r = Wifi::getInstance()->SocketSend(_socket, &data[n], length-n);
        if (r == -1) {
            return n;
        }
        n += r;
        
        if (_blocking && (t.read_ms() > _timeout)) {
            break;
        }
    }
    
    return n;  
}

int TCPSocketConnection::recv(char *data, int length)
{
    if ((_socket == InvalidSocketHandle) || !_is_connected)
        return -1;
        
    int n = Wifi::getInstance()->SocketRecv(_socket, data, length);
    
    return n;
}

int TCPSocketConnection::recv_all(char *data, int length)
{
    if ((_socket == InvalidSocketHandle) || !_is_connected)
        return -1;
        
    int n = 0;
    
    Timer t;
    t.start();
    
    while( n < length) {
        int r = Wifi::getInstance()->SocketRecv(_socket, &data[n], length-n);
        if (r == -1) {
            return n;
        }
        n += r;
        
        if (_blocking && (t.read_ms() > _timeout)) {
            break;
        }
    }
    
    return n;
}
