/* Copyright (c) 2013 Henry Leinen (henry[dot]leinen [at] online [dot] de)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#ifndef __DNSQUERY_H__
#define __DNSQUERY_H__

#include "Wifi.h"


/** Class DnsQuery implements DNS over UDP functionality.
  * Example as a typical use case :
  * @code
  * #include "mbed.h"
  * #include "DnsQuery.h"
  *
  * void main(void)
  * {
  *     IPADDRESS_t ipAddress;  // will receive the ip address of the host
  *     IPADDRESS_t dnsIp = { 192, 168, 178, 1 };      // Ip Address of the DNS server
  *     
  *     DnsQuery dns(Wifi::getInstance(), &dnsIp);
  *     if (dns.gethostbyname("mbed.org", ipAddress)) {
  *         printf("Ip-Address of mbed.org is %d.%d.%d.%d\n", ipAddress.sin_addr.o1, ipAddress.sin_addr.o2, ipAddress.sin_addr.o3, ipAddress.sin_addr.o4);
  *     } else {
  *         printf("Unable to obtain IP-Address\n");
  *     }
  * }
  * @endcode
  */
class DnsQuery
{
public:
    /** Constructor to instantiate a DnsQuery object.
      * @param wifi : A valid pointer to a Wifi Object, which can be used to obtain a UDP socket object.
      * @param dnsip : A valid pointer which holds the IPAddress of the DNS server to query.
      */
    DnsQuery(Wifi* wifi, IPADDRESS_t* dnsip);
    
    /** Function gethostbyname implements the functionality to query a domain name server for an IP-Address of a given hostname.
      * @param hostname : the hostname of interest as a string.
      * @param ipaddress : a reference to a IPADDRESS_t object which will receive the resolved IP Address of the host in question.
      * @returns true if successfull, or false otherwise.
      */
    bool gethostbyname(const char* hostname, IPADDRESS_t &ipaddress);
protected:
    bool resolve(char* resp, IPADDRESS_t &ipaddress);
    bool parseRR(char *resp, int& c, IPADDRESS_t& adr );

protected:
    Wifi* _wifi;
    IPADDRESS_t _dnsip;
};

#endif // __DNSQUERY_H__