/* HTTPConnection.h */
/*
Copyright (c) 2013 Henry Leinen (henry[dot]leinen [at] online [dot] de)
 
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
#ifndef __HTTPConnection_H__
#define __HTTPConnection_H__

#include "mbed.h"

#include <string>
#include <map>

class HTTPServer;

/** Type HTTPRequestType enumerates request types
*/
typedef enum 
{
    HTTP_RT_GET,        /*!< GET request */
    HTTP_RT_POST,       /*!< POST request */
    HTTP_RT_PUT,        /*!< PUT request */
    HTTP_RT_OPTIONS,    /*!< OPTIONS request */
    HTTP_RT_HEAD,       /*!< HEAD request */
    HTTP_RT_DELETE,     /*!< DELETE request */
    HTTP_RT_TRACE,      /*!< TRACE request */
    HTTP_RT_CONNECT     /*!< CONNECT request */
} HTTPRequestType;

/** class HTTPConnection, encapsulates one connection being made throught the HTTPServer
 *
 */
class HTTPConnection {
    public:
        
        /** HTTPMessage contains all the details of the request received by external HTTP client.
        */
        typedef struct 
        {
            /** Specifies the request type received
            */
            HTTPRequestType                         request;
            /** The uri associated with the request.
            */
            std::string                             uri;
            /** Contains the HTTP/1.1 or HTTP/1.0 version requested by client.
            */
            std::string                             version;
            /** Map of headers provided by the client in the form <HeaderName>:<HeaderValue>
            */
            std::map<std::string, std::string>      headers;
            /** Map of arguments that came with the uri string
            */
            std::map<std::string, std::string>      args;
        } HTTPMessage;
        
        /** Public constructor for HTTPConnection objects.
         *
         */
        HTTPConnection ();
        
        /** Destructor for HTTPConnection objects.
        *
        */
        ~HTTPConnection();
        
        /** Query if this connection is already closed and can be deleted.
        @returns true, if connection is closed.
        */
        bool is_closed();
        
    protected:
        
        /** Function to close this connection. To be called from internally.
        */
        void close();
                        
        HTTPMessage m_Msg;

        /** parse a HTTP request line from the content of the buffer. 
        * @param buffer : the request received from the client in the form <RequestType> <uri> <Http Version>
        * @returns -1 if the request is invalid or 0 if the request is valid
        */
        int parse(char *buffer);
        int receiveHeaders(const char* buffer, int nBuffSize);

    
        /** parse the receoved \c uri_string for arguments which will be stored in the \c args map.
        * @parse uri_string : the uri string which was received from the client.
        * @parse args : the args map which will receive the argument:value touples from the \c uri_string.
        * @returns -1 if an error occured, otherwise returns 0.
        */
        int parseUriArgs(char *uri_string, map<string,string>& args);

};

#endif // __HTTPConnection_H__