#include "mbed.h"
#include "ESP8266Interface.h"
#include "TCPSocketConnection.h"
#include "ifttt.h"
#include "INA226.hpp"
#include "float.h"
#include "SoftSerialSendOnry.h"

// ROOP_CNT * SAMPLING[sec] = 1ROOP_TIME[sec]
#define ROOP_CNT 100
#define SAMPLING 0.1

// Please change your enviroment.
const char WIFI_SSID[] = "ssid";
const char WIFI_PASSWORD[] = "password";
const char IFTTT_EVENT_NAME[] = "event_name";
const char IFTTT_SECRET_KEY[] = "secret_key";

// WIFI setting
ESP8266Interface wifi(dp16, dp15, dp4, WIFI_SSID, WIFI_PASSWORD, 115200); // TX,RX,Reset,SSID,Password,Baud
TCPSocketConnection socket;
IFTTT *ifttt;

//Serial pc(USBTX,USBRX);
SoftSerialSendOnry pc(dp10);
I2C i2c(dp5, dp27);
INA226 VCmonitor(i2c);

// Timer
Timer timer;
float nextUpdateTime = 0.0f;

// Data params
double minW = DBL_MAX, maxW = 0.0, totalW = 0.0; // Power[mW]
double minV = DBL_MAX, maxV = 0.0, totalV = 0.0; // Voltage[mV]
double minC = DBL_MAX, maxC = 0.0, totalC = 0.0; // Current[mA]
unsigned int count = 0;

void postIFTTT(){
    char postData[128];
    sprintf(postData, "%f,%f,%f,%f,%f,%f,%f,%f,%f", totalV/ROOP_CNT, maxV, minV, totalC/ROOP_CNT, maxC, minC, totalW/ROOP_CNT, maxW, minW);
    pc.printf("postData:%s\r\n",postData);
    ifttt->addIngredients(postData);
    ifttt->trigger(IFTTT_POST);
}

void record(){
    nextUpdateTime += SAMPLING;
    double V,C,W;
    if((VCmonitor.getVoltage(&V) != 0) || (VCmonitor.getCurrent(&C) != 0)){
        pc.printf("Error:VCmonitor does not get voltage or current.\r\n");
        return;
    }
    count++;
    
    // Calculate power.
    W = V * C * 0.001;//[mW]
//    pc.printf("%7.2f [mV],%7.2f [mA], %f [mW]\r\n",V ,C ,W);
    
    // update Max ,Min and Average
    if(maxW < W){maxW = W;}
    if(maxV < V){maxV = V;}
    if(maxC < C){maxC = C;}
    if(minW > W){minW = W;}
    if(minV > V){minV = V;}
    if(minC > C){minC = C;}
    totalW += W;
    totalV += V;
    totalC += C;

    if(count >= ROOP_CNT){ // 0.1[sec] * 1000 = 10[sec]
        pc.printf("AVG %7.2f [mV],%7.2f [mA], %f [mW]\r\n",totalV/ROOP_CNT ,totalC/ROOP_CNT ,totalW/ROOP_CNT);
        pc.printf("MIN %7.2f [mV],%7.2f [mA], %f [mW]\r\n",minV, minC, minW);
        pc.printf("MAX %7.2f [mV],%7.2f [mA], %f [mW]\r\n",maxV, maxC, maxW);
        
        postIFTTT();

        // clear params.
        minW = DBL_MAX, maxW = 0.0, totalW = 0.0;
        minV = DBL_MAX, maxV = 0.0, totalV = 0.0;
        minC = DBL_MAX, maxC = 0.0, totalC = 0.0;
        count = 0;
        nextUpdateTime = 0.0f;
        timer.reset();
    }
}

void initIFTTT(){
    wifi.init(); //Reset
    wifi.connect(); //Use DHCP
    
    pc.printf("IP Address is %s \n\r", wifi.getIPAddress());
    // Initialize ifttt object, add up to 3 optional values, trigger event. 
    ifttt = new IFTTT(IFTTT_EVENT_NAME, IFTTT_SECRET_KEY, &socket); // EventName, Secret Key, socket to use    
}

void initINA226PRC(){
    pc.printf("VCmonitor INA226 TEST Program. (BUILD:[" __DATE__ "/" __TIME__ "])\r\n");

    if(!VCmonitor.isExist()){
        pc.printf("VCmonitor NOT FOUND\r\n");
        while(1){}
    }
    pc.printf("VCmonitor FOUND\r\n");

    unsigned short val = 0;
    if(VCmonitor.rawRead(0x00,&val) != 0){
        pc.printf("VCmonitor READ ERROR\r\n");
        while(1){}
    }
    pc.printf("VCmonitor Reg 0x00 : 0x%04x\r\n",val);

    VCmonitor.setCurrentCalibration();
}

void init(){
    pc.baud(9600);
    initIFTTT();
    initINA226PRC();
    timer.start();
}

int main() {
    init();
    
    while(1){
        // wait
        while(timer.read() < nextUpdateTime){}
        record();
    }
}
