/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#ifndef MICROBIT_VALUE_H
#define MICROBIT_VALUE_H

#include "mbed.h"
#include "MicroBitConfig.h"
#include "MicroBitComponent.h"
#include "MicroBitStorage.h"

/**
  * Value events
  */
#define MICROBIT_VALUE_EVT_DATA_UPDATE          1

struct ValueSample
{
    int     v;

    ValueSample()
    {
        this->v = 0;
    }

    ValueSample(int v)
    {
        this->v = v;
    }

    bool operator==(const ValueSample& other) const
    {
        return v == other.v;
    }

    bool operator!=(const ValueSample& other) const
    {
        return !(v == other.v);
    }
};

/**
  * Class definition for MicroBit Value.
  *
  * Represents an implementation of shared value.
  * Also includes basic caching.
  */
class MicroBitValue : public MicroBitComponent
{

    ValueSample             sample;                   // The latest sample data recorded.
    MicroBitStorage*        storage;                  // An instance of MicroBitStorage used for persistence.

    public:

    /**
      * Constructor.
      * Create a software representation of an e-value.
      *
      *
      * @param _storage an instance of MicroBitStorage, used to persist calibration data across resets.
      *
      * @param id the ID of the new MicroBitCompass object. Defaults to MICROBIT_ID_VALUE.
      *
      * @code
      *
      * MicroBitStorage storage;
      *
      * MicroBitValue uBitValue(storage);
      * @endcode
      */
    MicroBitValue(MicroBitStorage& _storage, uint16_t id = MICROBIT_ID_VALUE);

    /**
      * Constructor.
      * Create a software representation of an e-value.
      *
      * @param id the ID of the new MicroBitCompass object. Defaults to MICROBIT_ID_VALUE.
      *
      * @code
      * MicroBitI2C i2c(I2C_SDA0, I2C_SCL0);
      *
      * MicroBitCompass compass(i2c);
      * @endcode
      */
    MicroBitValue(uint16_t id = MICROBIT_ID_VALUE);

    /**
      * Reads the value of the V integer.
      *
      * @return The value V.
      *
      * @code
      * value.getV();
      * @endcode
      */
    int getV();

    /**
      * Set the value of the V integer.
      *
      * @param The value V.
      *
      * @code
      * value.setV(v);
      * @endcode
      */
    void setV(int v);

    /**
      * Updates the local sample, only if the compass indicates that
      * data is stale.
      *
      * @note Can be used to trigger manual updates, if the device is running without a scheduler.
      *       Also called internally by getV() member functions.
      */
    int updateSample();

    /**
      * Periodic callback from MicroBit idle thread.
      *
      * Calls updateSample().
      */
    virtual void idleTick();


    /**
      * Destructor for MicroBitCompass, where we deregister this instance from the array of fiber components.
      */
    ~MicroBitValue();

    private:

    /**
      * An initialisation member function used by the many constructors of MicroBitValue.
      *
      * @param id the unique identifier for this value instance.
      *
      */
    void init(uint16_t id);
};

#endif