/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

/**
  * Class definition for MicroBit Value.
  *
  * Represents an implementation of the Freescale MAG3110 I2C Magnetmometer.
  * Also includes basic caching.
  */
#include "MicroBitConfig.h"
#include "MicroBitValue.h"
#include "MicroBitFiber.h"
#include "ErrorNo.h"

/**
  * An initialisation member function used by the many constructors of MicroBitValue.
  *
  * @param id the unique identifier for this value instance.
  *
  */
void MicroBitValue::init(uint16_t id)
{
    this->id = id;

    if(this->storage != NULL)
    {
        /*
        KeyValuePair *calibrationData =  storage->get("compassCal");

        if(calibrationData != NULL)
        {
            CompassSample storedSample = CompassSample();

            memcpy(&storedSample, calibrationData->value, sizeof(CompassSample));

            setCalibration(storedSample);

            delete calibrationData;
        }
        */
    }

    // Indicate that we're up and running.
    status |= MICROBIT_COMPONENT_RUNNING;
}



/**
  * Constructor.
  * Create a software representation of an e-compass.
  *
  * @param _i2c an instance of i2c, which the compass is accessible from.
  *
  * @param _storage an instance of MicroBitStorage, used to persist calibration data across resets.
  *
  * @param address the default address for the compass register on the i2c bus. Defaults to MAG3110_DEFAULT_ADDR.
  *
  * @param id the ID of the new MicroBitCompass object. Defaults to MAG3110_DEFAULT_ADDR.
  *
  * @code
  * MicroBitI2C i2c(I2C_SDA0, I2C_SCL0);
  *
  * MicroBitStorage storage;
  *
  * MicroBitCompass compass(i2c, storage);
  * @endcode
  */
MicroBitValue::MicroBitValue(MicroBitStorage& _storage, uint16_t id) :
    sample(),
    storage(&_storage)
{
    init(id);
}

/**
  * Constructor.
  * Create a software representation of an e-compass.
  *
  * @param _i2c an instance of i2c, which the compass is accessible from.
  *
  * @param address the default address for the compass register on the i2c bus. Defaults to MAG3110_DEFAULT_ADDR.
  *
  * @param id the ID of the new MicroBitCompass object. Defaults to MAG3110_DEFAULT_ADDR.
  *
  * @code
  * MicroBitI2C i2c(I2C_SDA0, I2C_SCL0);
  *
  * MicroBitCompass compass(i2c);
  * @endcode
  */
MicroBitValue::MicroBitValue(uint16_t id) :
    sample(),
    storage(NULL)
{
    init(id);
}

/**
  * Updates the local sample, only if the compass indicates that
  * data is stale.
  *
  * @note Can be used to trigger manual updates, if the device is running without a scheduler.
  *       Also called internally by all get[X,Y,Z]() member functions.
  */
int MicroBitValue::updateSample()
{
    /**
      * Adds the compass to idle, if it hasn't been added already.
      * This is an optimisation so that the compass is only added on first 'use'.
      */
    if(!(status & 0x01))
    {
        fiber_add_idle_component(this);
        status |= 0x01;
    }

    return MICROBIT_OK;
}

/**
  * Periodic callback from MicroBit idle thread.
  *
  * Calls updateSample().
  */
void MicroBitValue::idleTick()
{
    updateSample();
}

/**
  * Attempts to set the sample rate of the compass to the specified value (in ms).
  *
  * @param period the requested time between samples, in milliseconds.
  *
  * @return MICROBIT_OK or MICROBIT_I2C_ERROR if the magnetometer could not be updated.
  *
  * @code
  * // sample rate is now 20 ms.
  * compass.setPeriod(20);
  * @endcode
  *
  * @note The requested rate may not be possible on the hardware. In this case, the
  * nearest lower rate is chosen.
  */
void MicroBitValue::setV(int v)
{
    this->sample.v = v;
}

/**
  * Reads the currently configured sample rate of the compass.
  *
  * @return The time between samples, in milliseconds.
  */
int MicroBitValue::getV()
{
    return (int)sample.v;
}
/**
  * Destructor for MicroBitCompass, where we deregister this instance from the array of fiber components.
  */
MicroBitValue::~MicroBitValue()
{
    fiber_remove_idle_component(this);
}
