/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

/**
  * Class definition for the MicroBit BLE Magnetometer Service.
  * Provides access to live magnetometer data via BLE, and provides basic configuration options.
  */
#include "MicroBitConfig.h"
#include "ble/UUID.h"

#include "MicroBitValueService.h"

/**
  * Constructor.
  * Create a representation of the MagnetometerService.
  * @param _ble The instance of a BLE device that we're running on.
  * @param _compass An instance of MicroBitCompass to use as our Magnetometer source.
  */
MicroBitValueService::MicroBitValueService(BLEDevice &_ble, uint16_t *_value) :
        ble(_ble), value(_value)
{
#if CONFIG_ENABLED(MICROBIT_DBG)
    if(SERIAL_DEBUG) SERIAL_DEBUG->printf("MicroBitValueService::MicroBitValueService value = %i\r\n", *value);
#endif
    // Create the data structures that represent each of our characteristics in Soft Device.

    GattCharacteristic  valueCharacteristic(MicroBitValueServiceDataUUID, (uint8_t *)&valueCharacteristicBuffer, 0,
    sizeof(valueCharacteristicBuffer),
    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE);

    // Initialise our characteristic values.
    valueCharacteristicBuffer = *value;

    // Set default security requirements
    valueCharacteristic.requireSecurity(SecurityManager::MICROBIT_BLE_SECURITY_LEVEL);

    GattCharacteristic *characteristics[] = {&valueCharacteristic};
    GattService         service(MicroBitValueServiceUUID, characteristics, sizeof(characteristics) / sizeof(GattCharacteristic *));

    ble.addService(service);

    valueCharacteristicHandle = valueCharacteristic.getValueHandle();

    ble.gattServer().write(valueCharacteristicHandle, (const uint8_t *)&valueCharacteristicBuffer, sizeof(valueCharacteristicBuffer));

    ble.gattServer().onDataWritten(this, &MicroBitValueService::onDataWritten);

    if (EventModel::defaultEventBus)
    {
        EventModel::defaultEventBus->listen(MICROBIT_ID_VALUE, MICROBIT_VALUE_EVT_UPDATE, this, &MicroBitValueService::valueUpdate, MESSAGE_BUS_LISTENER_IMMEDIATE);
    }
}

/**
  * Callback. Invoked when any of our attributes are written via BLE.
  */
void MicroBitValueService::onDataWritten(const GattWriteCallbackParams *params)
{

    if (params->handle == valueCharacteristicHandle && params->len >= sizeof(valueCharacteristicBuffer))
    {
        valueCharacteristicBuffer = *((uint16_t *)params->data);
        *value = valueCharacteristicBuffer;
        //MicroBitEvent e(41, MICROBIT_VALUE_EVT_UPDATE);
    }
}

/**
  * Magnetometer update callback
  */
void MicroBitValueService::valueUpdate(MicroBitEvent)
{
    
#if CONFIG_ENABLED(MICROBIT_DBG)
    if(SERIAL_DEBUG) SERIAL_DEBUG->printf("MicroBitValueService::valueUpdate\r\n");
#endif

    if (ble.getGapState().connected)
    {
        valueCharacteristicBuffer = *value;
        ble.gattServer().write(valueCharacteristicHandle, (const uint8_t *)&valueCharacteristicBuffer, sizeof(valueCharacteristicBuffer));
    }

}

const uint8_t  MicroBitValueServiceUUID[] = {
    0xab,0x05,0xf2,0xd8,0x25,0x1d,0x47,0x0a,0xa0,0x62,0xfa,0x19,0x22,0xdf,0xa9,0xa8
};

const uint8_t  MicroBitValueServiceDataUUID[] = {
    0xab,0x05,0xfb,0x11,0x25,0x1d,0x47,0x0a,0xa0,0x62,0xfa,0x19,0x22,0xdf,0xa9,0xa8
};

const uint8_t  MicroBitValueServicePeriodUUID[] = {
    0xab,0x05,0x38,0x6c,0x25,0x1d,0x47,0x0a,0xa0,0x62,0xfa,0x19,0x22,0xdf,0xa9,0xa8
};

const uint8_t  MicroBitValueServiceBearingUUID[] = {
    0xab,0x05,0x97,0x15,0x25,0x1d,0x47,0x0a,0xa0,0x62,0xfa,0x19,0x22,0xdf,0xa9,0xa8
};