#include "mbed.h"
#include "QEI.h"
#include "USBSerial.h"
#include "PTC3471.h"

#define Ts 0.01                                                                 //periodo de amostragem
#define pi 3.141592653589793

/******************************************************************************/
/****************  Definição de Variaveis, Objetos e Funções ******************/
/******************************************************************************/

USBSerial  pc;                                                                  // Objeto de comunicação serial com o TeraTerm

Ticker Control_Interrupt;                                                       // Interrupção de Tempo para acionamento do algoritmo de controle

QEI Encoder_Motor (PTD0,PTB17,NC, 300, QEI::X4_ENCODING);   // Objeto de leitura do encoder do motor
QEI Encoder_Pendulo (PTA12,PTA13,NC, 600, QEI::X4_ENCODING);// Objeto de leitura do encoder do pêndulo

DigitalOut Horario(PTC1);                                   // DigitalOut que sinaliza se deve virar o motor no sentido horário
DigitalOut AntiHorario(PTD5);                               // DigitalOut que sinaliza se deve virar o motor no sentido anti-horário
PwmOut Motor(PTD6);      


bool Flag_Controle = false;
int PlotCount = 0;

double phi0 = 0;                                                                // phi0 -> Ângulo lido pelo Encoder_Braco
double phi1 = 0;   

double th0 = 0;                                                                 // th0 -> Ângulo do braço
double th1 = 0;                                                                 // th1 -> Ângulo do pêndulo

double th0_f = 0;                                                                 // th0 -> Ângulo do braço
double th1_f = 0;                                                                 // th1 -> Ângulo do pêndulo

double tau = 5e-2; 

double th0_a = 0;                                                               // Valor de th0 um período de amostragem anterior
double th1_a = 0;                                                               // Valor de th1 um período de amostragem anterior

float  u=0;
                                                                                
void Init(void);                                                                // Função de Inicialização
void Sensor_Read(void);                                                         // Função de leitura dos sensores
void Control_Function(void);                                                    // Função de flag do controle, a ser chamada pela interrupção


/******************************************************************************/
/*************************** Corpo de Funções *********************************/
/******************************************************************************/

/*************************** Função Principal *********************************/
// A main chama todas as inicializações e então aguarda o sinal de que deve
// realizar controle. Esse sinal é dado pela flag "Controle" e é setada por uma
// interrupção de tempo.
//
// Para garantir a execução imediata do algoritmo de controle nenhum wait deve
// ser chamado durante a execução do controle e o uso de printfs deve ser 
// esporádico.
int main() {

    /*********************************************************************************/
    /** Inicialização do algoritmo de proteção. NUNCA DEVE SER RETIRADO DO PROGRAMA **/
    /**/                                wait(5);                                   /**/
    /**/         Protecao_Init(&Encoder_Motor, &Control_Interrupt, pi);        /**/
    /** Inicialização do algoritmo de proteção. NUNCA DEVE SER RETIRADO DO PROGRAMA **/
    /*********************************************************************************/
    
    Init();
    while(1) {
        
        if(Flag_Controle){
            
            Sensor_Read();                                                      // Executa a leitura dos sensores
            
            PlotCount++;
            if(PlotCount>=5){                                                  // Controla para que o printf ocorra apenas uma vez a cada 10 iterações
                PlotCount = 0;
                pc.printf("%f \t %f\n\r", phi0, phi1);
               
            }
            
            Flag_Controle = false;                                                   // Sinaliza que deve-se esperar o próximo sinal da interrupção de tempo para executar o próximo passo de controle
        }
    }
}


/************************* Função de Inicialização *****************************/
// Esta função concentra todas as inicializações do sistema
void Init(void){
    
    Motor.period(0.0001);
    Horario = 0;
    AntiHorario = 0;
    Motor = 0.0;
    Control_Interrupt.attach(&Control_Function, Ts);
   
}

/********************** Função de leitura dos sensores *************************/
// Cada vez que esta função é chamada deve-se calcular os ângulos e velocidades
// angulares por algum método conhecido
void Sensor_Read(void){
    
    th0_a=th0;                                                    
    th1_a=th1;                                                    
    
    /** Leituras cruas dos ângulos do encoder **/
    phi0 = pi*Encoder_Motor.getPulses()/600.0;                                  // (eventos_lidos/eventos_por_volta)*2*pi = angulo_em_radianos
    phi1 = pi*Encoder_Pendulo.getPulses()/1200.0;                               // (eventos_lidos/eventos_por_volta)*360  = angulo_em_graus
    
    th0 = phi0;
    /** Tratamento do ângulo lido para ser zero na vertical para cima **/       // Como o encoder é incremental quando inicializamos o programa com o pêndulo na posição
    if(phi1>0)                                                                   // vertical para baixo esta passa a ser lida como 0º. Porém, para o algoritmo de controle
        th1 = phi1-pi;                                                           // funcionar corretamente 0º deve ser o pêndulo na posição vertical para cima. Para 
                                                                                // garantir que isso aconteça subido o pêndulo no sentido horário ou anti-horário fazemos
    else if(phi1<=0)                                                             // th1 = th1-sgn(th1)*pi, onde sgn(x) é o sinal de x.
        th1 = phi1+pi;                                                           // Para ficar mais claro o funcionamento destes "if else" plote o sinal de th1 no tera term
    
    // Filtro (1/tau*s +1) nos angulos
    th0_f = (tau/(Ts+tau))*th0_f + (Ts/(Ts+tau))*th0;
    th1_f = (tau/(Ts+tau))*th1_f + (Ts/(Ts+tau))*th1;    
    
}

/**************** Função de flag do algoritmo de controle ******************/
// Esta função avisa a main quando executar o próximo passo do algoritmo de 
// controle. O uso de uma interrupção para o acionamento da flag garante que
// haja exatamente Ts segundos entre execuções.
void Control_Function(void){
     Flag_Controle = true;
 }