// ********************************
// ***  KeplerBRAIN V4 Library  ***
// ********************************


// TASTER

// value = READ_TASTER_DOWN(port)
// value: 0, 1

// value = READ_TASTER_PRESSED(port)
// value: 0, 1


// LEDs

// void WRITE_LED(int port, int value)
// port: 1, 2, 3
// value: 0, 1


// DISPLAY

// void WRITE_DISPLAY(int zeile, int zeichen, const char *value)
// zeile: 1, 2
// zeichen: 1, 2, ..., 16
// value: char[]


// MOTOREN

// void WRITE_MOTOR_SPEED(int port, float value)
// port: 1, 2, 3, 4
// value:  -0.9 bis 0.9 (Vorzeichen: Drehrichtung)

// WRITE_MOTOR_STOP(int port)
// port: 1, 2, 3, 4


// IOS

// void SET_IOS(int port, value)
// port: 1, 2, 3, 4
// value: IN, OUT, SERVO

// value = READ_IOS(port)
// value: 0, 1

// WRITE_IOS(value)
// value (OUT): 0, 1
// value (SERVO): 1000 .. 1500 .. 2000


// ADC

// int READ_ADC(int port)
// port: 1, 2, ..., 8
// value: 0 .. 1023


// I2C

// WRITE_I2C(adress, type, register, value)
// adress: 
// type: SRF10, HMC6352, MLX90614 BNO55
// register:
// value:

// int READ_I2C(adress, type, register)
// adress:
// type: SRF10, HMC6352, MLX90614 BNO55
// register:


// SPI

// WRITE_SPI(port, type, register, value)
// port: 1, 2, 3
// type: 
// register:
// value:

// value = READ_SPI(port, type, register)
// port: 1, 2, 3
// type:
// register:


#include "mbed.h"

// *** Allgemeine Funktionen ***

void SLEEP(int value)
{
  wait_ms(value);
}


Ticker timer;

int COUNTER;

void counter_count()
{
  COUNTER++;  
}  
  

// *** LEDs ***

// PA_4 Led grün
// PC_2 Led gelb
// PC_3 Led rot

DigitalOut led1(PA_4);
DigitalOut led2(PC_2);
DigitalOut led3(PC_3);

void WRITE_LED(int port, int value)
{
   switch(port)
    {
      case 1:
        if (value==0) led1 = 0;
        else led1 = 1;
      break;
      case 2:
        if (value==0) led2 = 0;
        else led2 = 1;
      break;
      case 3:
        if (value==0) led3 = 0;
        else led3 = 1;
      break;
    }
}


// *** TASTER ***

// PA_10 Taster 1
// PB_5  Taster 2
// PB_4  Taster 3

DigitalIn taster1(PA_10, PullUp);
DigitalIn taster2(PB_5, PullUp);
DigitalIn taster3(PB_4, PullUp);

bool taster1_down; 
bool taster2_down; 
bool taster3_down; 

int READ_TASTER_DOWN(int port)
{
   int value = 0;
   switch(port)
    {
      case 1:
        if (!taster1) value = 1;
      break;
      case 2:
        if (!taster2) value = 1;
      break;
      case 3:
        if (!taster3) value = 1;
      break;
    }
    return value;
}

int READ_TASTER_PRESSED(int port)
{
  int value = 0;
  switch(port)
    {
      case 1:
        if (!taster1)
        {
          if (taster1_down==false)
          {
            value = 1;
          }
          taster1_down = true;
        }
        else
        {
          taster1_down = false;
        } 
      break;
      case 2:
        if (!taster2)
        {
          if (taster2_down==false)
          {
            value = 1;
          }
          taster2_down = true;
        }
        else
        {
          taster2_down = false;
        } 
      break;
      case 3:
        if (!taster3)
        {
          if (taster3_down==false)
          {
            value = 1;
          }
          taster3_down = true;
        }
        else
        {
          taster3_down = false;
        } 
      break;
    }
    return value;
}    
      
  
// *** Display 16x2 ***

// PA_15 Display RS
// PC_13 Display E
// PC_11 Display D4
// PC_10 Display D5
// PC_12 Display D6
// PA_1  Display D7

DigitalOut display_rs(PA_15);
DigitalOut display_e(PC_13);
BusOut display_d(PC_11, PC_10, PC_12, PA_1);
    
void display_writeByte(int value)
{
  display_d = value >> 4;
  wait(0.000040f);
  display_e = 0;
  wait(0.000040f);
  display_e = 1;
  display_d = value >> 0;
  wait(0.000040f);
  display_e = 0;
  wait(0.000040f);
  display_e = 1;
}   
   
void display_writeCommand(int command)
{
  display_rs = 0;
  display_writeByte(command);
}   

void display_writeData(int data)
{
  display_rs = 1;
  display_writeByte(data);
}

void display_location(int column, int row) 
{
  int a = 0x80 + (row * 0x40) + column;
  display_writeCommand(a);
}

void display_cls()
{
  display_writeCommand(0x01);
  wait(0.00164f);
}

void display_string( const char *data )
{
    while( *data != '\0' )
        display_writeData( *data++ );
}
void display_init()
{
  display_e  = 1;
  display_rs = 0;
  wait(0.015); 

  for (int i=0; i<3; i++)
  {
    display_writeByte(0x3);
    wait(0.00164);
  }
  display_writeByte(0x2);
  wait(0.000040f);

  display_writeCommand(0x28);
  display_writeCommand(0x0C);
  display_writeCommand(0x6);
  
  display_cls();
}

void WRITE_DISPLAY(int zeile, int zeichen, const char *value)
{
  if (zeile<1) zeile = 1;
  if (zeile>2) zeile = 2;
  if (zeichen<1) zeichen = 1;
  if (zeichen>16) zeichen = 16;
  zeile--;
  zeichen--;
  display_location(zeichen, zeile);
  display_string(value);
}

void WRITE_DISPLAY_CLS()
{
  display_cls();    
}


// *** PWM Motoren ***

// PC_7 Motor 1 PWM
// PC_4 Motor 1 IN1
// PB_2 Motor 1 IN2

// PB_10 Motor 2 PWM
// PC_5  Motor 2 IN1
// PC_8  Motor 2 IN2

// PB_0 Motor 3 PWM
// PH_1 Motor 3 IN1
// PD_2 Motor 3 IN2

// PA_0 Motor 4 PWM
// PC_1 Motor 4 IN1
// PC_0 Motor 4 IN2

PwmOut m1_pwm(PC_7);
DigitalOut m1_in1(PC_4);
DigitalOut m1_in2(PB_2);

PwmOut m2_pwm(PB_10);
DigitalOut m2_in1(PC_5);
DigitalOut m2_in2(PC_8);

PwmOut m3_pwm(PB_0);
DigitalOut m3_in1(PH_1);
DigitalOut m3_in2(PD_2);

PwmOut m4_pwm(PA_0);
DigitalOut m4_in1(PC_1);
DigitalOut m4_in2(PC_0);

void WRITE_MOTOR_SPEED(int port, double value)
{
  if (value>0.9) value = 0.9;
  if (value<-0.9) value = -0.9;
  double dutycycle = abs(value);

  switch(port)
    {
      case 1:
        if (value==0) { m1_in1 = 0; m1_in2 = 0; m1_pwm.write(0); }
        if (value>0) { m1_in1 = 1; m1_in2 = 0; m1_pwm.write(dutycycle); }
        if (value<0) { m1_in1 = 0; m1_in2 = 1; m1_pwm.write(dutycycle); }
      break;
      case 2:
        if (value==0) { m2_in1 = 0; m2_in2 = 0; m2_pwm.write(0); }
        if (value>0) { m2_in1 = 1; m2_in2 = 0; m2_pwm.write(dutycycle); }
        if (value<0) { m2_in1 = 0; m2_in2 = 1; m2_pwm.write(dutycycle); }
      break;
      case 3:
        if (value==0) { m3_in1 = 0; m3_in2 = 0; m3_pwm.write(0); }
        if (value>0) { m3_in1 = 1; m3_in2 = 0; m3_pwm.write(dutycycle); }
        if (value<0) { m3_in1 = 0; m3_in2 = 1; m3_pwm.write(dutycycle); }
      break;
      case 4:
        if (value==0) { m4_in1 = 0; m4_in2 = 0; m4_pwm.write(0); }
        if (value>0) { m4_in1 = 1; m4_in2 = 0; m4_pwm.write(dutycycle); }
        if (value<0) { m4_in1 = 0; m4_in2 = 1; m4_pwm.write(dutycycle); }
      break;
    }
}

void WRITE_MOTOR_STOP(int port)
{
  switch(port)
    {
      case 1:
        m1_in1 = 0;
        m1_in2 = 0;
        m1_pwm.write(0);
      break;
      case 2:
        m2_in1 = 0;
        m2_in2 = 0;
        m2_pwm.write(0);
      break;
      case 3:
        m3_in1 = 0;
        m3_in2 = 0;
        m3_pwm.write(0);
      break;
      case 4:
        m4_in1 = 0;
        m4_in2 = 0;
        m4_pwm.write(0);
      break;
    }
}


// *** IOS ***
   
// PB_6  Digital In Out Servo 
// PB_7  Digital In Out Servo 
// PB_8  Digital In Out Servo 
// PB_9  Digital In Out Servo 

/*
DigitalOut ios1(PB_6);
DigitalOut ios2(PB_7);
DigitalOut ios3(PB_8);
DigitalOut ios4(PB_9);

DigitalIn ios1(PB_6, PullUp);
DigitalIn ios2(PB_7, PullUp);
DigitalIn ios3(PB_8, PullUp);
DigitalIn ios4(PB_9, PullUp);

PwmOut ios1(PB_6);
PwmOut ios2(PB_7);
PwmOut ios3(PB_8);
PwmOut ios4(PB_9);
*/

int READ_IOS_IN(int port)
{
  int value = 0;
  if (port==1)
  {
    DigitalIn ios1(PB_7, PullUp);
    if (!ios1) value = 1;
  }
  if (port==2)
  {
    DigitalIn ios2(PB_8, PullUp);
    if (!ios2) value = 1;
  }
  if (port==3)
  {
    DigitalIn ios3(PB_6, PullUp);
    if (!ios3) value = 1;
  }
  if (port==4)
  {
    DigitalIn ios4(PB_9, PullUp);
    if (!ios4) value = 1;
  }

  return value;
}

void WRITE_IOS_OUT(int port, int value)
{
  if (port==1)
  {
    DigitalOut ios1(PB_7);
    if (value==0) ios1 = 0;
    else ios1 = 1;
  }
  if (port==2)
  {
    DigitalOut ios2(PB_8);
    if (value==0) ios2 = 0;
    else ios2 = 1;
  }
   if (port==3)
  {
    DigitalOut ios3(PB_6);
    if (value==0) ios3 = 0;
    else ios3 = 1;
  }
   if (port==4)
  {
    DigitalOut ios4(PB_9);
    if (value==0) ios4 = 0;
    else ios4 = 1;
  }
 
}

void WRITE_IOS_SERVO(int port, int value)
{
  float dutycycle = 100.0/value;
  if (port==1)
  {
        PwmOut ios1(PB_7);
        ios1.period(0.02);
        ios1.write(dutycycle);
  }
  if (port==2)
  {
        PwmOut ios2(PB_8);
        ios2.period(0.02);
        ios2.write(dutycycle);
  }
  if (port==3)
  {
        PwmOut ios3(PB_6);
        ios3.period(0.02);
        ios3.write(dutycycle);
  }
  if (port==4)
  {
        PwmOut ios4(PB_9);
        ios4.period(0.02);
        ios4.write(dutycycle);
  }
   
}

// *** I2C ***

// PA_8 I2C SCL
// PC_9 I2C SDA

#define HMC6352 0x01
#define HMC6352_HEADING 0x41

I2C i2c(PC_9, PA_8); 

int READ_I2C(int address, int component, int reg)
{
  int value = 0;
  switch(component)
  {
      case HMC6352:
        char cmd[1];
        char bytes[2];
        cmd[0] = 0x41;
        i2c.write(address, cmd, 1);
        i2c.read(address, bytes, 2);
        value = (bytes[0] * 256 + bytes[1]) / 10;
      break;
  }  
  return value;
} 

// *** I2C HMC6352 ***

int READ_I2C_HMC6352()
{ 
  char cmd[2];
  char bytes[2];
  int wert = 0;
 
  cmd[0] = 0x41;
  i2c.write(0x42, cmd, 1);
  i2c.read(0x42, bytes, 2);
  wert = (bytes[0] * 256 + bytes[1]) / 10;
  
  return wert;
}
    

// *** I2C SRF10 ***

void SET_I2C_SRF10(int i2c_addr, char gain, char range)
{
  char cmd[2];
  
  cmd[0] = 0x01; // gain
  cmd[1] = gain; // gain (0x00 - 0x10, default 0x04)
  i2c.write(i2c_addr, cmd, 2);
 
  cmd[0] = 0x02; // range
  cmd[1] = range; // range (0x00 - 0xFF, default 0x80)
  i2c.write(i2c_addr, cmd, 2);
}

int READ_I2C_SRF10(int i2c_addr)
{
  char cmd[2];
  char bytes[2];
  int wert = 0;
  
  // Software Revision Register auslesen 
  cmd[0] = 0x00; 
  i2c.write(i2c_addr, cmd, 1);
  i2c.read(i2c_addr, bytes, 1);
  wert = bytes[0];
  
  // Abfrage, ob Messung läuft  
  if (wert!=0xFF)
  {
    // Messwert auslesen
    cmd[0] = 0x03; 
    i2c.write(i2c_addr, cmd, 1);
    i2c.read(i2c_addr, bytes, 1);
    wert = bytes[0];
    // Messung starten
    cmd[0] = 0x00; 
    cmd[1] = 0x51; 
    i2c.write(i2c_addr, cmd, 2);
  }
  return wert;
}


// *** I2C BNO055 ***

#define BNO055_ADDR      (0x28 << 1) 
#define BNO055_CHIP_ID          0x00
#define BNO055_CHIP_ID_VALUE    0xa0   
#define BNO055_AXIS_MAP_CONFIG  0x41
#define BNO055_OPR_MODE         0x3d
#define CONFIGMODE              0x00
#define MODE_NDOF               0x0c 
#define ACCEL_OFFSET_X_LSB      0x55
#define ACCEL_OFFSET_X_MSB      0x56
#define ACCEL_OFFSET_Y_LSB      0x57
#define ACCEL_OFFSET_Y_MSB      0x58
#define ACCEL_OFFSET_Z_LSB      0x59
#define ACCEL_OFFSET_Z_MSB      0x5a
#define MAG_OFFSET_X_LSB        0x5b
#define MAG_OFFSET_X_MSB        0x5c
#define MAG_OFFSET_Y_LSB        0x5d
#define MAG_OFFSET_Y_MSB        0x5e
#define MAG_OFFSET_Z_LSB        0x5f
#define MAG_OFFSET_Z_MSB        0x60
#define GYRO_OFFSET_X_LSB       0x61
#define GYRO_OFFSET_X_MSB       0x62
#define GYRO_OFFSET_Y_LSB       0x63
#define GYRO_OFFSET_Y_MSB       0x64
#define GYRO_OFFSET_Z_LSB       0x65
#define GYRO_OFFSET_Z_MSB       0x66
#define ACCEL_RADIUS_LSB        0x67
#define ACCEL_RADIUS_MSB        0x68
#define MAG_RADIUS_LSB          0x69
#define MAG_RADIUS_MSB          0x6a
#define BNO055_EULER_H_LSB      0x1a
#define BNO055_EULER_H_MSB      0x1b
#define BNO055_EULER_R_LSB      0x1c
#define BNO055_EULER_R_MSB      0x1d
#define BNO055_EULER_P_LSB      0x1e
#define BNO055_EULER_P_MSB      0x1f

void SET_I2C_BNO055()
{
  char data[7];
  char chip_id;
    
  char accel_offset_x_lsb_value = 8; 
  char accel_offset_x_msb_value = 0;
  char accel_offset_y_lsb_value = 34;
  char accel_offset_y_msb_value = 0;
  char accel_offset_z_lsb_value = 7;
  char accel_offset_z_msb_value = 0;

  char mag_offset_x_lsb_value = 45;
  char mag_offset_x_msb_value = 255;
  char mag_offset_y_lsb_value = 116;
  char mag_offset_y_msb_value = 0;
  char mag_offset_z_lsb_value = 90;
  char mag_offset_z_msb_value = 1;

  char gyro_offset_x_lsb_value = 1;
  char gyro_offset_x_msb_value = 0;
  char gyro_offset_y_lsb_value = 1;
  char gyro_offset_y_msb_value = 0;
  char gyro_offset_z_lsb_value = 0;
  char gyro_offset_z_msb_value = 0;

  char accel_radius_lsb_value = 0; 
  char accel_radius_msb_value = 3;
  char mag_radius_lsb_value = 66;
  char mag_radius_msb_value = 2;
    
  data[0] = BNO055_CHIP_ID;
  i2c.write(BNO055_ADDR, data, 1, true);
  i2c.read(BNO055_ADDR, data, 7, false);
  chip_id = data[0];
  
  while (chip_id != BNO055_CHIP_ID_VALUE)
  {
    data[0] = BNO055_CHIP_ID;
    i2c.write(BNO055_ADDR, data, 1, true);
    i2c.read(BNO055_ADDR, data, 7, false);
    chip_id = data[0];
  }
  
  data[0] = BNO055_OPR_MODE;
  data[1] = CONFIGMODE;
  i2c.write(BNO055_ADDR, data, 2);
  wait_ms(50);
  
  data[0] = BNO055_AXIS_MAP_CONFIG; 
  data[1] = 0x24; 
  data[2] = 0x00;
  i2c.write(BNO055_ADDR, data, 2);
    
  data[0] = ACCEL_OFFSET_X_LSB; data[1] = accel_offset_x_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = ACCEL_OFFSET_X_MSB; data[1] = accel_offset_x_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = ACCEL_OFFSET_Y_LSB; data[1] = accel_offset_y_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = ACCEL_OFFSET_Y_LSB; data[1] = accel_offset_y_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = ACCEL_OFFSET_Z_LSB; data[1] = accel_offset_z_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = ACCEL_OFFSET_Z_LSB; data[1] = accel_offset_z_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
   
  data[0] = MAG_OFFSET_X_LSB; data[1] = mag_offset_x_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = MAG_OFFSET_X_MSB; data[1] = mag_offset_x_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = MAG_OFFSET_Y_LSB; data[1] = mag_offset_y_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = MAG_OFFSET_Y_LSB; data[1] = mag_offset_y_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = MAG_OFFSET_Z_LSB; data[1] = mag_offset_z_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = MAG_OFFSET_Z_LSB; data[1] = mag_offset_z_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
   
  data[0] = GYRO_OFFSET_X_LSB; data[1] = gyro_offset_x_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = GYRO_OFFSET_X_MSB; data[1] = gyro_offset_x_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = GYRO_OFFSET_Y_LSB; data[1] = gyro_offset_y_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = GYRO_OFFSET_Y_LSB; data[1] = gyro_offset_y_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = GYRO_OFFSET_Z_LSB; data[1] = gyro_offset_z_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = GYRO_OFFSET_Z_LSB; data[1] = gyro_offset_z_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
   
  data[0] = ACCEL_RADIUS_LSB; data[1] = accel_radius_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = ACCEL_RADIUS_MSB; data[1] = accel_radius_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = MAG_RADIUS_LSB; data[1] = mag_radius_lsb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
  data[0] = MAG_RADIUS_MSB; data[1] = mag_radius_msb_value; i2c.write(BNO055_ADDR, data, 2); wait_ms(20);
   
  data[0] = BNO055_OPR_MODE;
  data[1] = MODE_NDOF;
  i2c.write(BNO055_ADDR, data, 2);
  wait_ms(10);
}

int READ_I2C_BNO055_YAW()
{ 
  int16_t wert;
  char data[3];
  
  data[0] = BNO055_EULER_H_LSB;
  i2c.write(BNO055_ADDR, data, 1, true);
  i2c.read(BNO055_ADDR, data, 2, false);
  
  wert = data[1] << 8 | data[0];
  wert = (double)wert/16;
  return wert;
}

int READ_I2C_BNO055_PITCH()
{ 
  int16_t wert;
  char data[3];
  
  data[0] = BNO055_EULER_P_LSB;
  i2c.write(BNO055_ADDR, data, 1, true);
  i2c.read(BNO055_ADDR, data, 2, false);
  
  wert = data[1] << 8 | data[0];
  wert = (double)wert/16;
  return wert;
}

int READ_I2C_BNO055_ROLL()
{ 
  int16_t wert;
  char data[3];
  
  data[0] = BNO055_EULER_R_LSB;
  i2c.write(BNO055_ADDR, data, 1, true);
  i2c.read(BNO055_ADDR, data, 2, false);
  
  wert = data[1] << 8 | data[0];
  wert = (double)wert/16;
  return wert;
}

// *** SPI ***

// PA_6 SPI MISO
// PA_7 SPI MOSI
// PA_5 SPI SCK
// PB_1 SPI ADC CS
// PB_12 SPI 1 CS
// PA_11 SPI 2 CS
// PA_12 SPI 3 CS

SPI spi(PA_7,PA_6,PA_5);
DigitalOut spi_adc_cs(PB_1);
DigitalOut spi_1_cs(PB_12);
DigitalOut spi_2_cs(PA_11);
DigitalOut spi_3_cs(PA_12);

// *** SPI MCP3008 ***

int READ_ADC(int port)
{
  int address = 0x80;
  int data1;
  int data2;
  int value = 0;
  
  switch (port)
  {
    case 1:
      address = 0xF0;
    break;
    case 2:
      address = 0xE0;
    break;
    case 3:
      address = 0xD0;
    break;
    case 4:
      address = 0xC0;
    break;
    case 5:
      address = 0xB0;
    break;
    case 6:
      address = 0xA0;
    break;
    case 7:
      address = 0x90;
    break;
    case 8:
      address = 0x80;
    break;
  }
  
  spi_adc_cs=0;
  spi.write(0x01);
  data1=spi.write(address);
  data2=spi.write(0x00);           
  value =(data1<<8) | data2;
  value = value & 0x03FF;                
  spi_adc_cs=1;
  
  return value; 
}

// *** SPI LINE ***

int READ_SPI_LINE(int spi_port, int sensor_port)
{
  int wert1 = 0;
  int wert2 = 0;
  int wert = 0;
  int spi_address = 0;
  
  switch(sensor_port)
  {
    case 1:
      spi_address = 0x80;   
    break;
    case 2:
      spi_address = 0x90;   
    break;
    case 3:
      spi_address = 0xA0;   
    break;
    case 4:
      spi_address = 0xB0;   
    break;
    case 5:
      spi_address = 0xC0;   
    break;
    case 6:
      spi_address = 0xD0;   
    break;
    case 7:
      spi_address = 0xE0;   
    break;
    case 8:
      spi_address = 0xF0;   
    break;
  }
  
  switch(spi_port)
  {
    case 1:
      spi_1_cs=0;
    break;
    case 2:
      spi_2_cs=0;
    break;
    case 3:
      spi_3_cs=0;
    break;
  }

  spi.write(0x01);
  wert1=spi.write(spi_address);
  wert2=spi.write(0x00);           
  wert =(wert1<<8) | wert2;
  wert = wert & 0x03FF;                
 
  switch(spi_port)
  {
    case 1:
      spi_1_cs=1;
    break;
    case 2:
      spi_2_cs=1;
    break;
    case 3:
      spi_3_cs=1;
    break;
  }
   
  return wert;   
}

// *** SPI Pixy ***

// PB_14 SPI Pixy MISO
// PB_15 SPI Pixy MOSI
// PB_13 SPI Pixy SCK
// PC_6 SPI Pixy CS

SPI spi_pixy(PB_15,PB_14,PB_13);
DigitalOut spi_pixy_cs(PC_6);


// *** UART PC ***

// PA_2 TX
// PA_3 RX

Serial UART(PA_2, PA_3);


// *** Initialisierung ***

void KEPLERBRAIN_INIT()
{
  // Counter
  timer.attach(counter_count, 0.1);
  
  // LEDs
  led1 = 0;
  led2 = 0;
  led3 = 0;
  
  // Display
  display_init();
  char textzeile[16];
  sprintf(textzeile," KeplerBRAIN V4 ");
  WRITE_DISPLAY(1, 1, textzeile);
  sprintf(textzeile,":-) WELLCOME (-:");
  WRITE_DISPLAY(2, 1, textzeile);

  // Motoren
  m1_pwm.period(0.00005);
  m2_pwm.period(0.00005);
  m3_pwm.period(0.00005);
  m4_pwm.period(0.00005);
  
  // SPI
  spi.format(8,0); 
  spi.frequency(1000000);
  spi_adc_cs = 1;
  spi_1_cs = 1;
  spi_2_cs = 1;
  spi_3_cs = 1;
  
  // SPI Pixy
  spi_pixy.format(8,0); 
  spi_pixy.frequency(1000000);
  spi_pixy_cs = 1;
  
  // UART PC
  UART.baud(19200);
    
}




// Code ...


/*
  
  int p1,p2,p3;          //to store adc high and low and PEC(packet error correction)
  int ch=0,temp_val;     //to store temparature value
  i2c.frequency(100000);  //I2C frequency 20000hz (20khz)


  // I2C SRF10 0xE0 Setup
  // gain
  cmd[0] = 0x01;
  cmd[1] = 0x04; // gain (0x00 - 0x10, default 0x04)
  i2c.write(0xE0, cmd, 2);
  // range
  cmd[0] = 0x02;
  cmd[1] = 0x80; // range (0x00 - 0xFF, default 0x80)
  i2c.write(0xE0, cmd, 2);
 

void READ_SPI_MCP3008()
{
  int data1;
  int data2;
  
  spi_cs=0; spi.write(0x01); data1=spi.write(0x80);data2=spi.write(0x00);           
  INT_SPI_MCP3008_VALUE =(data1<<8) | data2; INT_SPI_MCP3008_VALUE = INT_SPI_MCP3008_VALUE & 0x03FF;                
  spi_cs=1; 
    
}


  m4_pwm.write(0.5);

  ios1.period(0.02);
  ios2.period(0.02);
  ios3.period(0.02);
  ios4.period(0.02);
  
   // READ_I2C_HMC6352();
  //  UART.printf("%d\n",INT_I2C_HMC6352_VALUE);
   
   
  // READ_SPI_MCP3008();
 //  UART.printf("%d\n",INT_SPI_MCP3008_VALUE);
   
    m1_in1 = 0;
    m1_in2 = 0;
      
    m2_in1 = 0;
    m2_in2 = 0;
      
    m3_in1 = 0;
    m3_in2 = 0;
      
    m4_in1 = 0;
    m4_in2 = 0;
    
    
    ios1 = 0;
    ios2 = 0;
    ios3 = 0;
    ios4 = 0;
   
    if (ios1) led = 0; else led =1;
    if (ios2) led = 0; else led =1;
    if (ios3) led = 0; else led =1;
    if (ios4) led = 0; else led =1;
    
    
    ios1.write(0.1);
    ios2.write(0.1);
    ios3.write(0.1);
    ios4.write(0.1);
    
    wait(0.1);
    
    m1_in1 = 1;
    m1_in2 = 0;
      
    m2_in1 = 1;
    m2_in2 = 0;
      
    m3_in1 = 1;
    m3_in2 = 0;
      
    m4_in1 = 1;
    m4_in2 = 0;
      
    
    ios1 = 1;
    ios2 = 1;
    ios3 = 1;
    ios4 = 1;
    

   ios1.write(0.05);
    ios2.write(0.05);
    ios3.write(0.05);
    ios4.write(0.05);
    
 
    wait(0.1);
    
    m1_in1 = 0;
    m1_in2 = 1;
      
    m2_in1 = 0;
    m2_in2 = 1;
      
    m3_in1 = 0;
    m3_in2 = 1;
      
    m4_in1 = 0;
    m4_in2 = 1;
      
    wait(0.1);
    
    
  
*/

