// I2CU - Search for devices on an I2C bus
// Copyright (c) 2009, sford
// Released under the MIT License: http://mbed.org/license/mit
//
// Goes through each device address, seeing if it gets a response
//  - for every device if finds, it prints the 8-bit address
//  - if the program hangs, the bus isn't working (check pull-ups etc)
//  - if it doesn't find your device, check power/bus wiring etc
//
//  - Note: if you plug / unplug devices try cycling power

#include "mbed.h"
#include "MAX11410.h"

#define CS1 D9
#define CS2 D10
#define MOSI_1 D11
#define MISO_1 D12
#define SCLK_1 D13
#define VDD 3.3
AnalogOut vOut1(A3);
AnalogOut vOut2(A4);
DigitalOut cs_pin1(CS1);
DigitalOut cs_pin2(CS2);
SPI spi(MOSI_1, MISO_1, SCLK_1);
MAX11410 adc1(&spi,&cs_pin1);
MAX11410 adc2(&spi,&cs_pin2);
//Serial pc(USBTX, USBRX,115200);
Serial pc(USBTX, USBRX, 9600);

void starting()
{
    pc.printf("this program has started\r\n");
}





void print8bitRegsAdc1(char start_reg,char end_reg)
{
    bool int_status;
    char val;
    for(int n=start_reg;n<=end_reg;n++)
    {
        val = adc1.read8bits(n,&int_status);
        pc.printf("reg %02x, val =%02x\r\n",n,val);
    }
    
}

void print8bitRegsAdc2(char start_reg,char end_reg)
{
    bool int_status;
    char val;
    for(int n=start_reg;n<=end_reg;n++)
    {
        val = adc2.read8bits(n,&int_status);
        pc.printf("reg %02x, val =%02x\r\n",n,val);
    }
}

int main() 
{
    int32_t channel_data[10];
    float vdiff=0.01;
    vOut1 = 0.5-vdiff/2;
    vOut2 = 0.5+vdiff/2;
    starting();
    bool int_state;
    spi.format(8,MAX11410_SPI_MODE); //configure number of spi bits 8, 16
    spi.frequency(8000000); // Max 8MHz
    //read chip ID
    
    pc.printf("chip ID %d\r\n",adc1.read8bits(REG_PART_ID,&int_state) );
    
    //config ADC 1
    adc1.reset(); //POR
//    adc2.reset();
        
    //check default register configs
    pc.printf("default regs for ADC1\r\n");
    print8bitRegsAdc1(REG_PD,REG_WAIT_START);
    
    pc.printf("default regs for ADC2\r\n");
    print8bitRegsAdc2(REG_PD,REG_WAIT_START);
    
    
    //config interrupts, PGA, Buffer, polarity, reference    
    char mode_config = MODE_NORMAL;
    adc1.write8bitReg(REG_PD,mode_config);      //got to normal mode
    adc2.write8bitReg(REG_PD,mode_config);      //got to normal mode
    
    char filter_config = FIR_SIXTY | _RATE(4) ; //
    adc1.write8bitReg( REG_FILTER,filter_config );
    adc2.write8bitReg( REG_FILTER,filter_config );
 
    char ctrl_config = INT_CLOCK | BIPOLAR | TWOS_COMP | _PBUF_EN(0) | _NBUF_EN(0) | REF_AVDD ; //ADC configuration
    adc1.write8bitReg( REG_CTRL, ctrl_config );
    adc2.write8bitReg( REG_CTRL, ctrl_config );
    
    char source_config = VBIAS_ACTIVE | BRN_OFF | _IDAC(0); //not sourcing current
    adc1.write8bitReg( REG_SOURCE,source_config );
    adc2.write8bitReg( REG_SOURCE,source_config );

    uint32_t status_ie_config = DATA_RDY_INT | CAL_RDY_INT | CONV_RDY_INT;   
    adc1.write24bitReg(REG_STATUS_IE,status_ie_config); 
    adc2.write8bitReg( REG_SOURCE,source_config );
    
    char gain_setting = 0;
    char pga_config = PGA | _GAIN_EXP(gain_setting) ; //no gain
    adc1.write8bitReg(REG_PGA,pga_config);
    adc2.write8bitReg(REG_PGA,pga_config);

    pc.printf("PGA gain = %d\r\n",1<<gain_setting);
    
    //check register writes
    pc.printf("checking register writes\r\n");
    pc.printf("reg %02x, val %02x, set to %02x\r\n",REG_PD,adc1.read8bits(REG_PD,&int_state),mode_config);
    pc.printf("reg %02x, val %02x, set to %02x\r\n",REG_FILTER,adc1.read8bits(REG_FILTER,&int_state),filter_config);     
    pc.printf("reg %02x, val %02x, set to %02x\r\n",REG_CTRL,adc1.read8bits(REG_CTRL,&int_state),ctrl_config);
    pc.printf("reg %02x, val %02x, set to %02x\r\n",REG_SOURCE,adc1.read8bits(REG_SOURCE,&int_state),source_config);
    pc.printf("reg %02x, val %06x, set to %06x\r\n",REG_STATUS_IE,adc1.read24bits(REG_STATUS_IE,&int_state),status_ie_config);     
    pc.printf("reg %02x, val %02x, set to %02x\r\n",REG_PGA,adc1.read8bits(REG_PGA,&int_state),pga_config);    
    
        // for each channel 1-5
        for(int n=0;n<5;n++)
        {
            //select channel
            char p_ch = 2*n<<4;
            char n_ch = 2*n+1;
            adc1.write8bitReg(REG_MUX_CTRL0, p_ch | n_ch );
            
            //select data output register and begin conversion
            adc1.write8bitReg(REG_CONV_START, (_DEST(n) | SINGLE_CONV) );
            
            //optional: cal Gain
            
            //optional: cal Offset
            
            //optional: store cal parameters
            
            //begin conversion
            
            //wait for interrupt
            while(!adc1.interrupt() )
            {
                wait_ms(CONV_DELAY_MS);//do nothing
            }
            
            //read conversion
           channel_data[n] = adc1.read24bitsSigned(REG_DATA0+n,&int_state);
           pc.printf("%d, ",channel_data[n]);    
        } //channel sweep
        pc.printf("\r\n");

    
    //config ADC 2: repeat above
    
    
    
//    while(1) 
//    {
//       
//    }

} //END MAIN
