/* mbed Ring Buffer for Servos
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SERVORINGBUFFER_H
#define SERVORINGBUFFER_H

// Define variables for structure
#define NUMBER_OF_SPINDLES         2
#define LEFT_SERVO_INDEX           0
#define RIGHT_SERVO_INDEX          1

// Define variables for ring buffer
#define BUFFER_SIZE     512            //This sets the ring buffer size to 512 elements

// Define a structure hierarchy for storing data
struct servoData {
    unsigned short pos;
    unsigned int force;
};
struct spindleData {
    servoData myServoData[NUMBER_OF_SPINDLES];
    unsigned char direction;
    unsigned char cycle;
    int time;
};

/** Servo ring buffer class, to store data between acquisiton and export
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "ServoRingBuffer.h"
 *
 * int main() {
 *
 *   ServoRingBuffer Buffer;     //Set up ring buffer
 *   AnalogIn Ain(p20);          //Set up potentiometer on pin 20
 *
 *   while (1) {
 *
 *       // This section of code should run whenever data needs to be acquired
 *       Buffer.write(Ain.read_u16(),Ain.read_u16());
 *
 *       // This section of code should run whenever there is free time to print to the screen
 *       Buffer.dumpBufferToSerial();
 *       wait (0.1);
 *   }
 * }
 * @endcode
 */
class ServoRingBuffer
{

public:
    ServoRingBuffer();

    void purge();
    void write(spindleData data);
    void write(unsigned short pos0, unsigned short force0, unsigned short pos1=0, unsigned short force1=0);
    spindleData read(void);
    void dumpBufferToSerial(void);
    void dumpBufferToSD(FILE * txtFile);
    unsigned int writesRemaining(void);
    unsigned int readsRemaining(void);
    float percentFull(void);

private:

    volatile unsigned int lastWritten;  //This is the variable to track the last written index
    volatile unsigned int lastRead;     //This is the variable to track the last read index
    unsigned int idx;                       //Buffer index counter
    Timer t;
    spindleData ringBuffer[BUFFER_SIZE]; //This creates a structure of structures with indeces from 0 to BUFFER_SIZE - 1 (i.e.-There are BUFFER_SIZE amount of indeces )

};

#endif // SERVORINGBUFFER_H