/**
* @file Max7221.h
* @brief Header file for the MAX7219/7221 driver class.
* 
* @author Grzegorz Kaczmarek
* @comment Code inspired on earlier Dwayne S. Dilbeck's work
* @date 20/11/2015
*/

#ifndef Max7221_H
#define Max7221_H

// MAX7219/7221 register map
#define max7221_reg_noop         0x00
#define max7221_reg_digit0       0x01
#define max7221_reg_digit1       0x02
#define max7221_reg_digit2       0x03
#define max7221_reg_digit3       0x04
#define max7221_reg_digit4       0x05
#define max7221_reg_digit5       0x06
#define max7221_reg_digit6       0x07
#define max7221_reg_digit7       0x08
#define max7221_reg_decodeMode   0x09
#define max7221_reg_intensity    0x0A
#define max7221_reg_scanLimit    0x0B
#define max7221_reg_shutdown     0x0C
#define max7221_reg_displayTest  0x0F

// Logical values
#define LOG_0 0
#define LOG_1 1

class Max7221 {
private:
    static SPI *mp_spi;             // SPI used for this display (no data transmission when NULL)
    static DigitalOut *mp_cs;       // /CS signal (it should be handled manually when NULL)
    static unsigned int m_counter;  // Total number of displays on SPI chain
    unsigned int m_position;        // Display position on SPI chain (0 = 1'st)
public:
    /**
    *  Default constructor.
    *  At least one instance should set SPI and /CS pointers for display chain to work.
    *  Use SetSpi() and SetCs() or parametrized constructor.
    */
    Max7221();

    /**
    *  Constructor.
    *  Sets SPI and /CS pointers. Display chain is operational.
    */
    Max7221(SPI *spi, DigitalOut *cs);

    /**
    *  Sets pointer to configured SPI.
    *  SPI is used by all display instances.
    */
    void SetSpi(SPI *spi);

    /**
    *  Sets pointer to configured /CS signal.
    *  /CS signal is used by all display instances.
    */    
    void SetCs(DigitalOut *cs);

    /**
    *  Method for setting /CS signal LOW (enable data reception).
    */
    void CsLow(void);

    /**
    *  Method for setting /CS signal HIGH (received data is latched and processed).
    */
    void CsHigh(void);

    /**
    *  Method used to send data over SPI (no use of /CS signal).
    *  To be used with display chains and when building display libraries
    *  where amount of data needs to be pushed via SPI in one transaction.
    */
    void WriteRaw(unsigned int reg, unsigned int data);
   
    /**
    *  Method for sending register-value pair for this diasplay.
    *  Supports display chain on single SPI (m_position > 0).
    */
    void Write(unsigned int reg, unsigned int data);

    /**
    *  Sets test mode which causes all display LEDs to emit light with full brightess.
    *  @param mode  true - test mode, false - normal operation.
    */
    void TestMode(bool mode);
    
    /**
    *  Sets amount of digits to be used by this display.
    *  @param digits_no  1 - one digit, (2-7), 8 - all 8 digits.
    */
    void UseDigitsNo(unsigned int digits_no);

    /**
    *  Selects for which digits Code-B data decoding is used.
    *  @param mode  0x00 - no decoding, 0x0F - digits 0-3, 0xFF - all digits.
    */
    void DecodeMode(unsigned int mode);

    /**
    *  Sends data to selected digit.
    *  @param digit_no  Selected digit (0-7).
    *  @param digit_no  Data to display.
    */
    void WriteDigit(unsigned int digit_no, unsigned int value);

    /**
    *  Sets display brightness.
    *  @param intensity  0x00 - minimum brightness, 0x0F - maximum brightness.
    */
    void Intensity(unsigned int intensity);

    /**
    *  Sets display operation mode.
    *  @param mode  true - normal operation, false - shutdown mode.
    */
    void OperationMode(bool mode);
    
    /**
    *  Initial setup of this display.
    *  8-digit blank decimal display set to nimimal brightness.
    *  Supports display chain on single SPI (m_position > 0).
    */   
    void Setup(void);
};

#endif