/**
* @file Max7221.cpp
* @brief MAX7219/7221 driver class.
* 
* @author Grzegorz Kaczmarek
* @comment Code inspired on earlier Dwayne S. Dilbeck's work
* @date 20/11/2015
*/

#include "mbed.h"
#include "Max7221.h"

// Static members initialization
SPI *Max7221::mp_spi = NULL;
DigitalOut *Max7221::mp_cs = NULL;
unsigned int Max7221::m_counter = 0;

Max7221::Max7221() {
    m_position = m_counter;
    m_counter++;
}

Max7221::Max7221(SPI *spi, DigitalOut *cs) {
    mp_spi = spi;
    mp_cs = cs;
    m_position = m_counter;
    m_counter++;
}

void Max7221::SetSpi(SPI *spi) {
    mp_spi = spi;
}

void Max7221::SetCs(DigitalOut *cs) {
    mp_cs = cs;
}
    
    
void Max7221::CsLow()
{
    if(mp_cs != NULL) {
        *mp_cs = LOG_0;
    }
}

void Max7221::CsHigh()
{
    if(mp_cs != NULL) {
        *mp_cs = LOG_1;
    }
}

void Max7221::WriteRaw(unsigned int reg, unsigned int data) {
    if(mp_spi != NULL) {
        mp_spi->write(reg);
        mp_spi->write(data);
    }
}

void Max7221::Write(unsigned int reg, unsigned int data) {
    unsigned int i;
    
    CsLow();
    if((m_counter > 1) && ((m_counter - 1) > m_position)) {
        i = m_counter - 1 - m_position;
        while(i--) {
            WriteRaw(max7221_reg_noop, 0x00);
        }        
    }
    WriteRaw(reg, data);
    if(m_position > 0) {
        i = m_position;
        while(i--) {
            WriteRaw(max7221_reg_noop, 0x00);
        }
    }
    CsHigh();
}

void Max7221::TestMode(bool mode) {
    if(mode) {
        Write(max7221_reg_displayTest, 0x01);
    } else {
        Write(max7221_reg_displayTest, 0x00);
    }
}

void Max7221::UseDigitsNo(unsigned int digits_no) {
    if((digits_no > 0) && (digits_no < 9)) {
        Write(max7221_reg_scanLimit, (digits_no - 1));
    }
}

void Max7221::DecodeMode(unsigned int mode) {
    Write(max7221_reg_decodeMode, mode);
}

void Max7221::WriteDigit(unsigned int digit_no, unsigned int value) {
    unsigned int digit_reg;

    if(digit_no < 8) {
        switch(digit_no) {
            case 0:
                digit_reg = max7221_reg_digit0;
                break;
            case 1:
                digit_reg = max7221_reg_digit1;
                break;
            case 2:
                digit_reg = max7221_reg_digit2;
                break;
            case 3:
                digit_reg = max7221_reg_digit3;
                break;
            case 4:
                digit_reg = max7221_reg_digit4;
                break;
            case 5:
                digit_reg = max7221_reg_digit5;
                break;
            case 6:
                digit_reg = max7221_reg_digit6;
                break;
            case 7:
                digit_reg = max7221_reg_digit7;
                break;
            default:
                digit_reg = max7221_reg_noop;
                break;
        }
        Write(digit_reg, value);
    }    
}

void Max7221::Intensity(unsigned int intensity) {
    if(intensity > 0x0F) {
        intensity = 0x0F;
    }
    Write(max7221_reg_intensity, intensity);
}

void Max7221::OperationMode(bool mode) {
    if(mode) {
        Write(max7221_reg_shutdown, 0x01);
    } else {
        Write(max7221_reg_shutdown, 0x00);
    }
}

void Max7221::Setup () {
    unsigned int i;

    TestMode(false);            // Disable display test
    UseDigitsNo(8);             // Use all 8 digits
    DecodeMode(0xFF);           // Turn on Code B font decode for all digits
    for(i=0;i<8;i++) {          // Clean all digits
        WriteDigit(i, 0x0F);
    }
    Intensity(0x01);            // Set lowest display intensity(0x00-0xFF)
    OperationMode(true);        // Enable operation mode
}
