/* 
   Created by Konstantinos Fane 
   copy this code to Mbed online simulator 
   to run the designed application
*/

#include "mbed.h"
#include <string>
#include "C12832.h" 

// LCD display
C12832 lcd(p5, p7, p6, p8, p11);

//variables used for frequency, period and time interval measruements
float m_freq_hz;
float m_t_interval_ms;
float period = 0.0;
float count1 = 0.0;

//result to be displayed
float m_result;

const char* range_scale [3][3] = {
    {"MHz","KHz","Hz"},
    {"ms","ms","s"},
    {"ms","ms","s"}
    };

/*  flag to hold which measurement mode to display, 0 = Frequency Measurement, 
 *  1 = Period Measurement, 2 = Time interval Measurement
 *  to match the index of message display arrays */
int dsp_mode = 0; 
const char* mode_msg [] = {"Frequency", "Period", "Time Interval"};

/*  flag to hold which measurement range to display
 *  0 = High Range, 1 = Medium Range, 2 = Low Range */
int dsp_range = 0;
const char* range_msg [] = {"High Resolution", "Medium Resolution", "Low Resolution"};

int dsp_time_interval = 0; //0=R-R, 1=F-F, 2=R-F, 3=F-R

//result string
const char* res[] = {""};

//Interrupt to be used in actual implementation to read the input square wave signal
InterruptIn squareIN(p17);

//User interface measurement mode and range selection buttons
InterruptIn mode(p12);
InterruptIn range(p15);
InterruptIn time_interval(p13);

//interrupt service routines
void counter1_isr();
void freq_calc_isr();
void mode_select();
void range_select();
void time_interval_type_select();

//Timeout task2;
Ticker freq_meas; //generate a tick
Ticker ui_disp; //generate the 1s interrupt, to measure frequency (direct frequency measurement)

//user defined functions to implement the user interface and result calculation
void user_interface_disp();
void result_calculator();

int main()
{
    
    //ticker, where it is done periodically, timeout only does it once
    //Direct frequency measurement, measures the number of rising edges every time they occur
    squareIN.rise(&counter1_isr); //maximum frequency related to interrupt operating frequency
    freq_meas.attach(&freq_calc_isr,1); //obtain number of sums per second
    
    ui_disp.attach(&user_interface_disp, 1); //update the screen every 1 second
    
    //interrupt routine for mode and range selection user input
    mode.rise(&mode_select);
    range.rise(&range_select);
    time_interval.rise(&time_interval_type_select);
    
    while(1)
    {
        wait_ms(0.1);
    }
 

}

//ISR to count the occurences of incoming signal rising edges
void counter1_isr()
{
    //increment counter by 1
    count1 = count1 + 1.0;
}

//ISR to calculate the period of incoming signal, every 1 second
void freq_calc_isr()
{
    m_freq_hz = count1;
    count1 = 0;
    //call the result calculator function, to calculate the required period and time interval results
    result_calculator();
}

void user_interface_disp()
{

    lcd.cls(); //clear screen
    lcd.locate(1,1);
    lcd.printf("Mode: %s", mode_msg[dsp_mode]);
    
    //display the type of time interval measurement if the mode is selected
    if(dsp_mode ==2)
    {
        lcd.locate(90,1);
        if(dsp_time_interval == 0)
        {
            lcd.printf("(R->R)");
        }
        else if(dsp_time_interval == 1)
        {
            lcd.printf("(F->F)");
        }
        else if(dsp_time_interval == 2)
        {
            lcd.printf("(R->F)");
        }
        else if(dsp_time_interval == 3)
        {
            lcd.printf("(F->R)");
        }
        
    }
    lcd.locate(1,10);
    lcd.printf("Range: %s", range_msg[dsp_range]);
    lcd.locate(1,20);
    //print the result
    lcd.printf(*res,m_result,range_scale[dsp_mode][dsp_range]);
    
}

void time_interval_type_select()
{
    if (dsp_time_interval <3)
    {
        dsp_time_interval++;
    }
    else
    {
        dsp_time_interval = 0;
    }
}

//Interrupt service routine to handle the measurement mode selection
void mode_select()
{
    if (dsp_mode < 2)
    {
        dsp_mode++;
    }
    else
    {
        dsp_mode = 0;
    }
}

//Interrupt service routine to handle the measurement range selection
void range_select()
{
    if (dsp_range < 2)
    {
        dsp_range++;
    }
    else
    {
        dsp_range = 0;
    }
}

void result_calculator()
{
    
    //calculate period from frequency
    float m_period_ms = (1/m_freq_hz)*1000; //times by 1000 to convert to ms
    
    if(dsp_mode == 0) //frequency measurement 
    {   
        if(dsp_range == 0) //High resolution (MHz)
        {
            m_result = m_freq_hz/1000000;
            if(m_result <= 10.001)
            {
                //assigns the decimal point precision for each range
                *res="%3.3f %s";
            }
            else
            {
                *res="Out-of-range... '>10 MHz !'";
            }
            
        }
        else if(dsp_range == 1) //medium resolution (KHz)
        {
            m_result = m_freq_hz/1000;
            if(m_result <= 100.001)
            {
                //assigns the decimal point precision for each range
                *res="%3.3f %s";
            }
            else
            {
                *res="Out-of-range... '>100 KHz !'";
            }
        }
        else //low resolution (Hz)
        {
            m_result = m_freq_hz;
            if(m_result <= 100.001)
            {
                //assigns the decimal point precision for each range
                *res="%3.3f %s";
            }
            else
            {
                *res="Out-of-range... '>100 Hz !'";
            }
            
        }
        
        /* '*res' assignment was moved inside the ifs, to that in case the current 
        nuber to be displayed is higher that acceptable upper limit, to delete 
        the displayed measurement, and show warning message*/
        
    }
    else if (dsp_mode == 1) //period measurement
    {
        if(dsp_range == 0) //High resolution (2.000 +- 0.0001 ms)
        {
            m_result = m_period_ms;
            if(m_result <= 2.0001)
            {
                //assigns the decimal point precision for each range
                *res="%1.4f %s";
            }
            else
            {
                *res="Out-of-range... '>2.000 ms !'";
            }
        }
        else if(dsp_range == 1) //medium resolution (20.00 +- 0.01 ms)
        {
            m_result = m_period_ms;
            if(m_result <= 20.01)
            {
                //assigns the decimal point precision for each range
                *res="%2.2f %s";
            }
            else
            {
                *res="Out-of-range... '>20.00 ms !'";
            }
        }
        else //low resolution (2.000 +- 0.001 s)
        {
            m_result = m_period_ms/1000;
            if(m_result <= 2.001)
            {
                //assigns the decimal point precision for each range
                *res="%1.3f %s";
            }
            else
            {
                *res="Out-of-range... '>2.000 sec'";
            }
        }
    }
    else // "(dsp_mode == 2)" time interval measurement 
    {
        if(dsp_time_interval == 0 || dsp_time_interval == 1) //rising->rising, falling -> falling
        {
            m_result = m_period_ms;
        }
        else if(dsp_time_interval == 2 || dsp_time_interval == 3) //rising->falling, falling->rising
        {
            m_result = m_period_ms/2; //based on the assumption that the square wave has 50% DUTY CYCLE
        }
        
        if(dsp_range == 0) //High resolution (2.0000 +- 0.0001 ms)
        {
            if(m_result <= 2.0001)
            {
                //assigns the decimal point precision for each range
                *res="%1.4f %s";
            }
            else
            {
                *res="Out-of-range... '>2.000 ms !'";
            }
        }
        else if(dsp_range == 1) //medium resolution (20.00 +- 0.01 ms)
        {
            if(m_result <= 20.01)
            {
                //assigns the decimal point precision for each range
                *res="%2.2f %s";
            }
            else
            {
                *res="Out-of-range... '>20.00 ms !'";
            }
        }
        else //low resolution (2.000+-0.0001ms)
        {
            //m_result = m_t_interval_ms/10e3;
            m_result = m_result/1000;
            if(m_result <= 2.001)
            {
                //assigns the decimal point precision for each range
                *res="%1.3f %s";
            }
            else
            {
                *res="Out-of-range... '>2.000 sec'";
            }
        }
    }    
}
