/* mbed USBHost Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef USBHOSTMIDI_H
#define USBHOSTMIDI_H

#include "USBHostConf.h"

#if USBHOST_MIDI

#include "USBHost.h"
#include "MtxCircBuffer.h"

/** 
 * A class to communicate a USB MIDI device
 */
class USBHostMIDI : public IUSBEnumerator {
public:
    /**
    * Constructor
    */
    USBHostMIDI();

    /**
    * Check if a USB MIDI device is connected
    *
    * @returns true if a midi device is connected
    */
    bool connected();
    
    /**
     * Try to connect a midi device
     *
     * @return true if connection was successful
     */
    bool connect();
    
    /**
     * Attach a callback called when note on is received
     *
     * @param ptr function pointer
     */
    inline void attachNoteOn(void (*fn)(unsigned char, unsigned char, unsigned char)) {
        if (fn != NULL) {
            noteOn = fn;
        }
    }

    /**
     * Attach a callback called when note off is received
     *
     * @param ptr function pointer
     */
    inline void attachNoteOff(void (*fn)(unsigned char, unsigned char, unsigned char)) {
        if (fn != NULL) {
            noteOff = fn;
        }
    }

    /**
     * Attach a callback called when control change is received
     *
     * @param ptr function pointer
     */
    inline void attachControlChange(void (*fn)(unsigned char, unsigned char, unsigned char)) {
        if (fn != NULL) {
            controlChange = fn;
        }
    }

    /**
     * Attach a callback called when program change is received
     *
     * @param ptr function pointer
     */
    inline void attachProgramChange(void (*fn)(unsigned char, unsigned char)) {
        if (fn != NULL) {
            programChange = fn;
        }
    }

    /**
     * Attach a callback called when pitch bend is received
     *
     * @param ptr function pointer
     */
    inline void attachPitchBend(void (*fn)(unsigned char, unsigned int)) {
        if (fn != NULL) {
            pitchBend = fn;
        }
    }

    /**
     * Send a note on event
     *
     * @param channel 0-15
     * @param note 0-127
     * @param velocity 0-127 (0 means note off)
     */
    int sendNoteOn(unsigned char channel, unsigned char note, unsigned char velocity);

    /**
     * Send a note off event
     *
     * @param channel 0-15
     * @param note 0-127
     * @param velocity 0-127
     */
    int sendNoteOff(unsigned char channel, unsigned char note, unsigned char velocity);

    /**
     * Send a control change event
     *
     * @param channel 0-15
     * @param key 0-127
     * @param value 0-127
     */
    int sendControlChange(unsigned char channel, unsigned char key, unsigned char value);

    /**
     * Send a program change event
     *
     * @param channel 0-15
     * @param program 0-127
     */
    int sendProgramChange(unsigned char channel, unsigned char program);

    /**
     * Send a control change event
     *
     * @param channel 0-15
     * @param key 0(lower)-8191(center)-16383(higher)
     */
    int sendPitchBend(unsigned char channel, unsigned int value);

protected:
    //From IUSBEnumerator
    virtual void setVidPid(uint16_t vid, uint16_t pid);
    virtual bool parseInterface(uint8_t intf_nb, uint8_t intf_class, uint8_t intf_subclass, uint8_t intf_protocol); //Must return true if the interface should be parsed
    virtual bool useEndpoint(uint8_t intf_nb, ENDPOINT_TYPE type, ENDPOINT_DIRECTION dir); //Must return true if the endpoint will be used

private:
    USBHost * host;
    USBDeviceConnected * dev;
    USBEndpoint * bulk_in;
    USBEndpoint * bulk_out;
    uint32_t size_bulk_in;
    uint32_t size_bulk_out;

    bool dev_connected;

    void init();

    MtxCircBuffer<uint8_t, 64> circ_buf;

    uint8_t buf[64];

    void rxHandler();

    void (*noteOn)(unsigned char, unsigned char, unsigned char);
    void (*noteOff)(unsigned char, unsigned char, unsigned char);
    void (*controlChange)(unsigned char, unsigned char, unsigned char);
    void (*programChange)(unsigned char, unsigned char);
    void (*pitchBend)(unsigned char, unsigned int);

    int midi_intf;
    bool midi_device_found;

};

#endif

#endif
