#ifndef LED_BAR_H
#define LED_BAR_H

/*  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  

Simple 6-LEDs bar library for using with MBed 2.0 library

Copyright 2017 Dmitry Makarenko

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  */

#include "mbed.h"



/**
*  LedBar class is used for controlling 6 led bar. 
*
*  Example using on, off, toggle methods:
*  
* @code
*  
*  #include "mbed.h"
*  #include "6LedBar.h"
*  
*  
*  LedBar leds (A0, A1, A2, A3, A4, A5);
*  
*  int main() {
*      leds.on(0);
*      leds.on(2);
*      leds.on(4);
*      
*      while(true) {
*          for(int i = 0; i < 6; i++) {
*              leds.toggle(i);
*          }
*          wait(0.5);
*      }
*  }
*  
*  
* @endcode
*  
*  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  
*  
*  Example using mask methods:
*  
* @code
*  
*  #include "mbed.h"
*  #include "6LedBar.h"
*  
*  
*  LedBar leds (A0, A1, A2, A3, A4, A5);
*  
*  int main() {
*  
*      leds.set_mask(0b010101);
*      
*      while(true) {
*          leds.toggle_mask(0b111111);
*          wait(0.5);
*      }
*  }
*  
* @endcode
*
**/

//!
class LedBar {

public:

    //Constructor, you need to specify pins to which
    //leds are connected
    LedBar(PinName p0,
           PinName p1,
           PinName p2,
           PinName p3,
           PinName p4,
           PinName p5);
           
    //! Turn n-s led on
    void on(int n);
    
    //! Turn n-s led off
    void off(int n);
    
    //! Toggle n-s led
    void toggle(int n);
    
    //! Set bit mask
    //i.e. mask 0b000101 will turn on leds 0 and 1, and off all others
    void set_mask(int m);
    
    //! Turn on leds specified by bit-mask
    void on_mask(int m);
    
    //! Turn off leds specified by bit-mask
    void off_mask(int m);
    
    //! Toggle leds specified by bit-mask
    void toggle_mask(int m);
private:
    static const int LEDS_COUNT = 6;
    DigitalOut l0, l1, l2, l3, l4, l5;
    DigitalOut *m_leds[LEDS_COUNT];

};


#endif //LED_BAR_H