/*
 * Created by Simon Krogedal
 * 11/05/2021
 * 
 * 
 * On launch, run stepper backwards until the contact switch triggered. Set this as joint position
 * Subscribe to joint topics
 * Set joint angles when a message is recieved
 * Publish joint angles
 * Publish FSR reading
 * 
 * This version was ported to mbed from the orignal Arduino version
 * 
 */

#include "mbed.h"

// Include Ros libs as needed
#include <ros.h>
#include <sensor_msgs/JointState.h>
#include <std_msgs/Float32.h>

// Lib for hardware interface
#include "Servo.h"
#include "Async_4pin_Stepper.h"

// Pin definitions for NUCLEO-F401RE
#define stepperPin1   D3     // IN1 for ULN2003 driver
#define stepperPin2   D4     // IN2 for ULN2003 driver
#define stepperPin3   D5     // IN3 for ULN2003 driver
#define stepperPin4   D6     // IN4 for ULN2003 driver

#define servo1Pin     D1     // Control pin for Servo 1 on main arm
#define servo2Pin     D2     // Control pin for Servo 2 on crank arm

#define SWITCH_PIN    D8     // Input pin for digital switch
#define fsrPin        A0     // Input pin for force sensor


// Define the AccelStepper interface type; 4 wire motor in half step mode
#define MotorInterfaceType 8
// Define steps per rev for stepper
#define SPR 4075.7728
// Pinion circumference at pitch combined with the above value gives step/m, a useful constant
#define SPM 128205
#define SPMM 128.205
// Degrees per radian, general matemathical constant
#define DPR 57.3
// FSR scale value, in N
#define FSR_SCALE 1
// Stepper speed, steps/s
#define STEP_V 833 // (max for 28BYJ-48)

// Declare node handle
ros::NodeHandle nh;

// Declare hardware objects
Servo   servo1(servo1Pin),
        servo2(servo2Pin);
// Initialise stepper with pin sequence IN1-IN3-IN2-IN4
Async_4pin_Stepper stepper(stepperPin1, stepperPin3, stepperPin2, stepperPin4, SPM);

// Declare digital input for stepper limit switch
DigitalIn switchPin(SWITCH_PIN);

// Declare publisher and message
std_msgs::Float32 fsr_val;
ros::Publisher fsrPublisher("FSR_value", &fsr_val);

// Joint related constants
//int offset[3] = {0, 0, 0};
//int restPos[3] = {0, 0, 0};

// Like the map function but takes floats as inputs and spits out ints
int floatmap(float x, float in_min, float in_max, long out_min, long out_max) {
  float temp = (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
  return temp;
}

// Function that can be called to calibrate prismatic joint by running it 
// until the switch is activated. Must be run on launch
void calibrateStepper() {
  nh.loginfo("Running stepper to limit");
  // Set target way way back and just overshoot
  stepper.setTarget(-40000);
  stepper.goToTarget();
  // Check switch until triggered
  while(!switchPin) {
    wait_ms(10);
  }
  stepper.stop();
  // Set current position as origin
  stepper.setZeroPos();
  nh.loginfo("Stepper at 0-pos");
}


// Subscriber callback function for message
void subscriberCallback(const sensor_msgs::JointState& msg) {
  nh.loginfo("Callback triggered");
  // Transfrom message for local hardware
//  int s1pos = msg.position[0] * DPR - offset[0];
//  int s2pos = msg.position[1] * DPR - offset[1];
//  int steps = (msg.position[2] - offset[2]) * SPM;

  // Kinematic angles
//  int s1pos = map(msg.position[0], 3.14, 5.85, 36, 145);
//  int s2pos = map(msg.position[1], 2.93, 5.85, 153, 24);
//  int steps = (msg.position[2] - 30) * SPMM;
  
  // URDF angles - URDF file doesn't use DH, so has a different transform and thus angle limits
  int s1pos = floatmap(msg.position[0], -1.92, 1.13, 36, 145);
  int s2pos = floatmap(msg.position[1], -0.44, 2.53, 153, 24);
  int steps = -(msg.position[2] + 100) * SPMM;

  // Write angles to acutators
  /*
  Serial.print("Recieved angles :");
  for (int i = 0; i < 3; i++) {
    Serial.print(msg.position[i]);
    Serial.print(" ");
  }
  Serial.print("\nMoving to angles: ");
  Serial.print(s1pos); Serial.print(" "); Serial.print(s2pos); Serial.print(" "); Serial.println(steps);
  */
  servo1.position(s1pos);
  servo2.position(s2pos);
  stepper.setTarget(steps);
  stepper.goToTarget();
  nh.loginfo("Joints written");

}

// Subscribe to topic /phys_joint_states, which is fed by a script that takes the desired non-parallel joint states
ros::Subscriber<sensor_msgs::JointState> jointControlSubscriber("phys_joint_states", &subscriberCallback);

int main() {
  
  // Setup stepper
  stepper.setSpeed(833);
  
    // Analogue input for FSR value reading
    AnalogIn FSR(fsrPin);

  // Center servos on rest positions
  int centerPos[2] = {42, 142};
  //Serial.print("\nInitialising servos to "); Serial.print(centerPos[0]); Serial.print(" and "); Serial.println(centerPos[1]);
  servo1.position(centerPos[0]);
  servo2.position(centerPos[1]);

  // Set the connection to rosserial
  nh.initNode();

  // Built-in lED used as a connection indicator
    DigitalOut myled(LED1);
    myled = 0;
    
    // Wait until connected to ROS, while blinking LED
    while(!nh.connected()) {
        nh.spinOnce();
        wait_ms(500);
        myled = !myled;
    }
    myled = 1;
    
  nh.loginfo("mbed node connected");
  nh.subscribe(jointControlSubscriber);
  nh.loginfo("Arduino node subscribed");
  nh.advertise(fsrPublisher);

  // Find stepper position
  calibrateStepper();
  
  // Main loop
  while(true) {
      
        // Read FSR pin and publish value
        fsr_val.data = FSR * FSR_SCALE;
        fsrPublisher.publish(&fsr_val);
            
        nh.spinOnce();
        
        // Enter a tight for loop for ~1 second
        for(int i=0; i<1000; i++) {
            wait_ms(1);
            nh.spinOnce();
        }
    }
}
