/**
 * @file encoder.h
 * This class is based upon the QEI class by Aaron Berk and the encoder class I wrote during ESP in 2nd year.
 *
 * @author Simon Krogedal
 *
 * @version 0.1
 */

#ifndef KARBOT_ENCODER_H
#define KARBOT_ENCODER_H


#include "mbed.h"
 
 
#define PREV_MASK 0x1 //Mask for the previous state in determining direction
//of rotation.
#define CURR_MASK 0x2 //Mask for the current state in determining direction
//of rotation.
#define INVALID   0x3 //XORing two states where both bits have changed.

/** encoder Class
 *
 * Class for measuring quadrature magnetic encoders requiring internal pullups
 *
 * @author Simon Krogedal
 *
 * 27/05/2021
 *
 * Team 9 4th Year project
 * 
 *
 * for NUCLEO-F401RE
 * 
 * @version 0.2
 *
 */
class encoder {
    
    public:
    
        /** Direction enumerator
         */
        typedef enum  {
            Left,         // Left side motor, CCW rotation is forward
            Right         // Right side motor, CW rotation is forward
        } Side;
    
        /** Create an instance
         *
         * @param chanA Encoder channel A pin
         * @param chanB Encoder channel B pin
         * @param CPR Encoder clicks per revolution
         * @param side Left side or right side motor, defines whether counter-clockwise rotation is forwards (left) or backwards (right) (when looking at the robot from outside)
         * @param Period Sampling period of the wheel speed
         * @note Too short samling period gives a noisy reading, too long and dynamics are lost
         * @param diameter Wheel diameter in meters
         */
        encoder(PinName chanA, PinName chanB, int CPR, Side side, double Period, double diameter);
        
        /** Get wheel speed
         * @returns Wheel speed in meters/second
         */
        double getSpeed(void);
        
        /** Get distance travelled
         * @returns Distance travelled in meters
         */
        double getDistance(void);
        
        /** Get distance travelled
         * @returns Distance travelled in meters since last reset
         */
        double tempDist(void);
        
        /// Reset distance measurement
        void distRst(void);
        
        /// Reset distance measurement
        void tempRst(void);
        
        /// Start encoder reading
        void start(void);
        
        /// Stop encoder reading
        void reset(void);
        
        /**
         * Read the state of the encoder.
         *
         * @return The current state of the encoder as a 2-bit number, where:
         *         bit 1 = The reading from channel B
         *         bit 2 = The reading from channel A
         */
        int getCurrentState(void);
    
        /**
         * Read the number of pulses recorded by the encoder.
         *
         * @return Number of pulses which have occured.
         */
        int getPulses(void);
    
    private:
        int         tot_clicks, temp_tot;   // clicks since last distance reset
        double      click_rate, click_store;// clickrate
        bool        c_d;                    // left or right bool
        
        /**
         * Update the pulse count.
         *
         * Called on every rising/falling edge of channels A/B.
         *
         * Reads the state of the channels and determines whether a pulse forward
         * or backward has occured, updating the count appropriately.
         */
        void encode(void);
    
        InterruptIn channelA_;
        InterruptIn channelB_;
    
        int          pulsesPerRev_;
        int          prevState_;
        int          currState_;
    
        volatile int pulses_;
        
        Side         side_;

    protected:
    
        Ticker      sampler;                /// ticker object to sample speed
        double      period, enc_const;      /// sampling period and wheel constant
        
        /// Sample function called by ticker object
        void sample_func(void);
        
        /** Get wheel speed
         * @returns Wheel speed in clicks/second
         */
        double getClicks(void);
    
};

#endif