/**
 * @file MotorControl.h
 *
 * Class to control a motor with an encoder, inherits from the encoder class.
 *
 * @author Simon Krogedal
 *
 * @version 0.1
 */

#ifndef KARBOT_MOTOR_CONTROL_H
#define KARBOT_MOTOR_CONTROL_H

/* Karbot motor control class
 * Written by Simon Krogedal
 * 27/05/21
 * Team 9 4th Year project
 * 
 * for NUCLEO-F401RE
 * 
 */
 
 #include "encoder.h"
 #include "motor.h"

 /** Motor Controller Class
 *
 * This class controls the speed of an individual motor, helping the system keep
 * symmetric with asymmetric components. It is based on the encoder class
 * and contains a pointer to a motor object. With two of these the motion
 * controller can drive the robot to the desired points.
 *
 * @author Simon Krogedal
 *
 * Written by Simon Krogedal
 *
 *
 * Team 9 4th Year project
 * 
 *
 * for NUCLEO-F401RE
 * 
 * @version 0.1
 *
 */
class MotorControl : public encoder {
    
    private:
    
        motor*      mot;            // motor object
        
        double      Kp,             // Proportional gain
                    Ti,             // Integral gain time
                    r_speed,        // Speed set point
                    r_clicks,       // Same but in the "encoder language"
                    max_speed,      // The max speed of the motor (not used)
                    output,         // PWM output duty cycle
                    prev_error;     // Previous error, used for intergral control
                    
        bool        dir,            // Drivign direction (not used)
                    max_out;        // Flag triggered when output is saturated (not used)
        
        /** Get the current tracking error
         * @return Current tracking error
         */
        double getError(void);
        
        /// Control algorithm, called intermittedly by ticker object
        void algorithm(void);
    
    public:
        /** Creates an instance
         *
         * @param EncoderChanA Encoder channel A pin, set up using internal pullup
         * @param EncoderChanB Encoder channel B pin, set up using internal pullup
         * @param CPR Encoder counts per revolution
         * @param side Left side or right side motor, defines whether counter-clockwise rotation is forwards (left) or backwards (right) (when looking at the robot from outside)
         * @param period Sampling period for control algorithm and speed calculation from encoder readings
         * @param Motor Pointer to the motor object to be controlled
         * @param MaxSpeed Maximum speed of the motor, no set points above this value will be accepted
         * @param kp Proportional control gain
         * @param ti Integral control time
         * @param diameter Wheel diameter in meters
         */
        MotorControl(PinName EncoderChanA, PinName EncoderChanB, int CPR, encoder::Side side, double period, motor* Motor, double MaxSpeed, double kp, double ti, double diameter);
        
        /** Set the speed set point of the controller
         * This can be done while driving or while stopped, though the update will only be reflected while driving.
         * @param speed Speed set point in m/s
         */
        void setSpeed(double speed);
        
        /** Start driving
         * Attaches a ticker object to the control algorithm, running it at the frequency specified in the constructor
         */
        void drive(void);
        
        /** Stops the control algorithm
         * Stops the motors and detaches the ticker callback
         */
        void stop(void);
        
        /** Drives the motor in open loop
         * No ticket object is attaced, the motors are just activated.
         * Note that this also means new speed set points will not be passed to the motor unless this function is called again!
         */
        void driveManual(void);
        
        /** Sample the encoders to read speed value
         * This function does not return anything, the speed must then be obtained through getSpeed()
         */
        void samplecall(void);
        
        /** Set proportional gain
         * This function is useful for Ziegler-Nichols tuning
         *
         * @param k Controller proportional gain
         */
        void setK(double k);        // Set gain (used for Z-N tuning
        
        //void start(void);           // This function is overridden to avoid bugs
        
        /** Returns the current duty cycle, useful for debugging
         * @return Current duty cycle
         */
        double getOutput(void);     // Returns current duty cycle
        
        /** Checks whether the output saturation flag is set
         *
         * @return True: Output is saturated
         * @return False: Output is not saturated
         */
        bool checkFlag(void);       // Check whether max out flag is set
};

#endif