/**
 * @file Async_4pin_Stepper.h
 * @author Simon Krogedal <simon.krogedal@student.manchester.ac.uk>
 * @version 0.1
 */

#ifndef ASYNC_4PIN_STEPPER_H
#define ASYNC_4PIN_STEPPER_H

#include "mbed.h"

/** Asynchronous Stepper with 4-pin half step output class.
 *
 
 *
 * 31/05/2021
 * Team 9 4th Year project
 * 
 * for NUCLEO-F401RE
 * 
 
 *
 */
class Async_4pin_Stepper {
    
    private:
    
        Ticker ticker;          /// Ticker object
        BusOut output;          /// Collection of output pins
        
        int     currentPos;     /// Current position of stepper
        int     targetPos;      /// Target position of stepper
        int     speed_;         /// Set speed for stepper movement
        double  stepInterval;   /// Step interval for movement at desired speed
        double  SPR;            /// Steps per revolution
        bool    running;        /// Boolean flag indicating whether the motor is running
        
        /** Do a single step, called by stepOnce functions
         * @param step The desired step
         */
        void    step(int step); /// Do a single step, called by step once funcs
        void    stepCB();       /// Callback function for stepper, called intermittedly to do a step towards the current target
        
    protected:
        
        void stepOnceCW();      /// Single step CW
        void stepOnceCCW();     /// Single step CCW
        
    public:
    
        typedef enum  {
            CW,         /// Clockwise
            CCW         /// Counter-clockwise
        } Dir;
    
        /** Constructor takes 4 pins and a steps/rev in float presicion
         *
         * @param StepperPin1 Pin 1 on stepper motor
         * @param StepperPin2 Pin 2 on stepper motor
         * @param StepperPin3 Pin 3 on stepper motor
         * @param StepperPin4 Pin 4 on stepper motor
         * @param StepsPerRev Steps per revolution. Defaults to 4075.8, for 28BYJ-48
         */
        Async_4pin_Stepper( PinName StepperPin1,
                            PinName StepperPin2,
                            PinName StepperPin3,
                            PinName StepperPin4,
                            float   StepsPerRev = 4075.7728);
                            
        /// Set current position as 0
        void setZeroPos();
        
        /** Set target
         * @param target Target step, can be positive or negative
         */
        void setTarget(int target);
        
        /// Move towards given target
        void goToTarget();
        
        // Run at a given speed in step/s
//        void run(float speed, Dir dir);   

        /** Set speed
         * @param speed Speed in steps/seconds
         */
        void setSpeed(int speed);               
        
        /** Get speed
         * @returns Speed in steps/seconds
         */
        int getSpeed();
        
        /** Get position
         * @returns Stepper position
         */
        int getPos();
        
        /** Get target position
         * @returns Target step position
         */
        int getTarget();
        
        /** Get distance left to target
         * @returns Number of steps away from target
         */
        int distanceToGo();
        
        /** Check if motor is running
         * @returns
         *   0 Motor is not running (ticker is not attached)
         *   1 Motor is running (ticker is attached)
         */
        bool isRunning();
        
        /// Close all gates, reducing current but losing holding torque
        void relax();
        
        /// Stop the motor and detach ticker
        void stop();

};

#endif