/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO Animation SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#include "Animator.h"
#include "MicroBit.h"
#include <math.h>
#include <mbed.h>

MicroBitImage pattern(5,5);

Serial pc(USBTX, USBRX);

int inx=0;
int delta = 1;
int pixel_value=255;
int flash_state=1;
int num_rnd_pixels = 4;

int ripple_pixels[4][25] = {
    {0,0,0,0,0,  0,0,0,0,0,  0,0,0,0,0,  0,0,0,0,0,  0,0,0,0,0}, 
    {0,0,0,0,0,  0,0,0,0,0,  0,0,1,0,0,  0,0,0,0,0,  0,0,0,0,0}, 
    {0,0,0,0,0,  0,1,1,1,0,  0,1,0,1,0,  0,1,1,1,0,  0,0,0,0,0},
    {1,1,1,1,1,  1,0,0,0,1,  1,0,0,0,1,  1,0,0,0,1,  1,1,1,1,1} 
};

int spiral_pixels[25] = {
    12, 13, 8, 7, 6,   11, 16, 17, 18, 19,   14, 9, 4, 3, 2,   1, 0, 5, 10, 15,   20, 21, 22, 23, 24
};

Animator::Animator(MicroBitDisplay &_display) :
  display(_display), sleep_time(500), playing(0), current_animation(1)
{
}

void Animator::setAnimationType(uint16_t animation) 
{
    pc.printf("Animator::setAnimationType %d\n",animation);
    if (animation <= NUMBER_OF_ANIMATIONS) {
        current_animation = animation;
        pattern.clear();
        display.clear();
        switch (current_animation) {
          case ANIMATION_TYPE_FLASH:
            pc.printf("Animator::setAnimationType FLASH\n");
            sleep_time = 500;
            break;
          case ANIMATION_TYPE_RIPPLE:
            pc.printf("Animator::setAnimationType RIPPLE\n");
            sleep_time = 100;
            inx=0;
            delta = 1;
            break;
          case ANIMATION_TYPE_SPIRAL:
            pc.printf("Animator::setAnimationType SPIRAL\n");
            sleep_time = 50;
            break;
          default:
            pc.printf("Animator::setAnimationType default=FLASH\n");
            current_animation = ANIMATION_TYPE_FLASH;
            sleep_time = 500;
            break;        }        
    }
}

void Animator::start()
{
    pc.printf("Animator::start\n");
    playing = 1;
    MicroBitEvent(ANIMATION_STATUS_EVENT,ANIMATION_STATUS_BUSY);
}
    
void Animator::stop()
{
    pc.printf("Animator::stop\n");
    playing = 0;
    display.clear();
    MicroBitEvent(ANIMATION_STATUS_EVENT,ANIMATION_STATUS_FREE);
}
    
void Animator::faster() 
{
    if (sleep_time > 0) {
        sleep_time = sleep_time - ANIMATION_SPEED_DELTA;
    }
}
    
void Animator::slower()
{
    sleep_time = sleep_time + ANIMATION_SPEED_DELTA;
}

void Animator::ripple() {
    int i=0;
    for (i=0;i<25;i++) {
        pattern.setPixelValue(i % 5 , floor(static_cast<double>(i / 5)), ripple_pixels[inx][i]);
    }
    display.image.paste(pattern);
    inx = inx + delta;
    if (inx == 3 || inx == 0) {
        delta = delta * -1;
    }
}

void Animator::flash() {
    int i;
    for (i=0;i<25;i++) {
        pattern.setPixelValue(i % 5 , floor(static_cast<double>(i / 5)), flash_state * 255);
    }
    display.image.paste(pattern);
    flash_state = !flash_state;
}

void Animator::spiral() {
    int x = spiral_pixels[inx] % 5;
    int y = floor(static_cast<double>(spiral_pixels[inx] / 5));
    pattern.setPixelValue(x,y, pixel_value);
    display.image.paste(pattern);
    inx = inx + delta;
    if (inx == 25 || inx == -1) {
        delta = delta * -1;
        inx = inx + delta;
        if (pixel_value == 255) {
            pixel_value = 0;
        } else {
            pixel_value = 255;
        }
    }
}