#include "mbed.h"
#include "dac1.h"

#define CODE_ALL_LOAD_ALL 0xC2      // CODE ALL, LOAD ALL command byte
#define CODE_N_LOAD_N     0xB0      // CODE N, LOAD N command byte (add DAC selection nibble)

#define UPPER_LIMIT_DAC1_VAL( x ) if( x > 4095 ) { x = 4095; }

// DAC1 Pin Setup
// 12-bits, 8-channels, zero-scale (0b000000000000 = 0V output)
// Drive CSB high then low, then send 24 bits for each command (bit 23 first, bit 0 last)  

struct {
    SPI spi;
    DigitalOut clr_bar;
    DigitalOut csb_bar;
    } dac1 = {
    SPI( PC_12, PC_11, PC_10 ), // mosi, miso, sclk for SPI3 on STM32F411RE
    DigitalOut( PD_2 ),         // assert to reset DAC state to default (active-low)
    DigitalOut( PA_15 ),        // chip select (active-low)
};

void dac1_init( void ) {
    dac1_reset( );
    dac1.csb_bar = 1;
    
    dac1.spi.format( 8, 2 );        // 8-bit data, polarity = 1, phase = 0 (mode 2)
    dac1.spi.frequency( 1000000 );  // 1MHz clock frequency
    
    dac1_send_command( 0x27, 0x00, 0x00 );  // Set REF to 4.096V internal
}

void dac1_reset( void ) {
    dac1.clr_bar = 0;   // Reset DAC to default state
    wait( 0.1 );
    dac1.clr_bar = 1;
}

void dac1_send_command( char byte_1, char byte_2, char byte_3 ) {
    dac1.csb_bar = 0;
    
    dac1.spi.write( byte_1 );
    dac1.spi.write( byte_2 );
    dac1.spi.write( byte_3 );
    
    dac1.csb_bar = 1;
}

void dac1_code_load_all( uint16_t val ) {
    UPPER_LIMIT_DAC1_VAL( val );
    dac1_send_command( CODE_ALL_LOAD_ALL, val / 16, ( val % 16 ) << 4 );    // CMD, MSB, LSB
}

void dac1_code_load_channel( uint16_t val, uint8_t channel ) {
    if( channel <= 7 ) {
        UPPER_LIMIT_DAC1_VAL( val );
        dac1_send_command( CODE_N_LOAD_N + channel, val / 16, ( val % 16 ) << 4 );  // CMD, MSB, LSB
    }
}