// Include mbed standard functionality library
#include "mbed.h"

// Include USB serial device library imported from mbed web repo
#include "USBSerial.h"

// Include h-files written specifically for this project
#include "dac1.h"
#include "dac2.h"
#include "state_control.h"
 
// Need to double check these colors when LEDs are installed
DigitalOut led_red( PC_0 );
DigitalOut led_green( PC_1 );
DigitalOut led_yellow( PC_2 );

DigitalOut precharge_en_bar( PA_3 );
DigitalOut boost_en( PA_2 );
DigitalOut cuk_en( PA_1 );

DigitalOut one_shot_clr_bar( PA_5 );    // Set this high for now, can later use as timer output

//Virtual serial port over USB
USBSerial serial;

int main() {
    one_shot_clr_bar = 1;   // Set this high for now, can later use as timer output
    
    // Default mode is IV test to prevent damage to device from misconfigured quenching circuit
    set_relay_mode_ivtest();
    
    // Precharge the DC/DC buscap, wait 3 seconds, write initial DAC vals, start DC/DC converters
    precharge_en_bar = 0;
    wait( 5.0 );
    
    cuk_en = 1;
    wait( 1.0 );
    boost_en = 1;
    
    wait( 1.0 );
    led_green = 1;
    
    // Initialize 8-channel DAC1
    dac1_init( );
    dac1_code_load_all( 0 );
    
    dac2_init( );
    
    dac1_code_load_channel( 3000, 0 );     // 3.5V on DAC1 channel 0 (DELTAV_SETPOINT)
    dac1_code_load_channel( 2000, 1 );     // 2.0V on DAC1 channel 1 (DIAMOND_IBIAS)
    dac1_code_load_channel( 3000, 2 );     // 4.0V on DAC1 channel 2 (ONE_SHOT_COMP)
    dac1_code_load_channel( 1320, 3 );     // 1.0V on DAC1 channel 3 (VA_SETPOINT)
    dac1_code_load_channel( 2000, 4 );     // 2.0V on DAC1 channel 4 (ONE_SHOT_CCS)
    dac1_code_load_channel( 4000, 5 );     // 4.0V on DAC1 channel 5 (EXTRA_1)
    dac1_code_load_channel( 4000, 6 );     // 4.0V on DAC1 channel 6 (EXTRA_2)
    dac1_code_load_channel( 4000, 7 );     // 4.0V on DAC1 channel 7 (EXTRA_#)
    
    dac2_write( 950 );  // Comparator reference voltage
    
    char buf[ 128 ];
    while( 1 ) {
        serial.scanf( "%s", buf );
        serial.printf( "Recv: %s\n", buf );
        
        led_red = 1;    // Red LED indicates serial activity
        
        switch( buf[ 0 ] ) {
            case 'W' :   // write to DAC
                if( buf[ 1 ] == '1' ) {   // write to DAC1
                    char temp[ 4 ] = { buf[ 2 ], buf[ 3 ], buf[ 4 ], buf[ 5 ] };
                    uint16_t val = atoi( temp );
                    uint8_t channel = buf[ 6 ] - '0';
                    dac1_code_load_channel( val, channel );
                    
                    serial.printf( "Success: wrote %d to DAC1 channel %d.\n", val, channel );
                }
                else if( buf[ 1 ] == '2' ) {  // write to DAC2
                    char temp[ 4 ] = { buf[ 2 ], buf[ 3 ], buf[ 4 ], buf[ 5 ] };
                    uint16_t val = atoi( temp );
                    dac2_write( val );
                    
                    serial.printf( "Success: wrote %d to DAC2.\n", val );
                }
                else{
                    serial.printf( "Error: invalid DAC number.\n" );
                }
                break;
            case 'M' :  // change the relay mode
                if( buf[ 1 ] == '0' ) {         // set mode to AQC (default)
                    set_relay_mode_aqc();
                    serial.printf( "Success: changed relay mode to AQC.\n" );
                }
                else if( buf[ 1 ] == '1' ) {    // set mode to IV TEST
                    set_relay_mode_ivtest();
                    serial.printf( "Success: changed relay mode to IV TEST.\n" );
                }
                else{
                    serial.printf( "Error: invalid relay mode.\n" );
                }
                break;
            default :
                serial.printf( "Error: invalid command.\n" );
        }
        
        led_red = 0;
    }
}