//  controller.cpp: 制御器
#include "mbed.h"   // mbedマイコンではstdio.hに相当
//#include "QEI.h"    // エンコーダ用ライブラリを使用

#include "controller.h" //  controller.cpp: モータ制御器（位置制御、電流制御）
PwmOut pwm0p(p21);   // IN1 of TA7291P for right wheel
PwmOut pwm0m(p22);   // IN2 of TA7291P for right wheel
PwmOut pwm1p(p23);   // IN1 of TA7291P for left wheel
PwmOut pwm1m(p24);   // IN2 of TA7291P for left wheel
unsigned char   fReverse[2];  // モータ逆回転フラグ：回転方向が順方向のとき0、逆方向のとき1。[0]が現在の値、[1]はその前の値    reverse direction?

Serial pc(USBTX, USBRX);    // PCのモニタ上のtera termに文字を表示する宣言

controller_parameters    K[2];      // 速度制御メインループの定数、変数
float                    volt[2];   // モータへの入力電圧
AnalogOut   analog_out(DA_PORT);    // デバッグ用DA(アナログ信号をDA_PORTに出力)

unsigned long _countTS0;   // TS0[s]ごとのカウント数
float   _time;          // [s], プログラム開始時からの経過時間
float   debug[20];      // デバッグ用変数
DigitalOut  led1(LED1); // mbedマイコンのLED1を点灯
DigitalOut  led2(LED2); // mbedマイコンのLED2を点灯
DigitalOut  led3(LED3); // mbedマイコンのLED3を点灯
DigitalOut  led4(LED4); // mbedマイコンのLED4を点灯


float data[1000][5];    // PC上のmbed USB ディスクにセーブするデータ   memory to save data offline instead of "online fprintf".
unsigned short _countTS3=0;

DigitalOut  debug_p26(p26); // p17 for debug
DigitalOut  debug_p25(p25); // p17 for debug

// PIDゲイン
float   Kp[2], Ki[2], Kd[2];
#define vMAX   3.3       // [V], モータ入力電圧の最大値（超えるとこの値に制限する）

void init_parameters(){
// 制御器の初期値の設定
//  親関数： main()
//  子関数： なし
    // 制御器0（人の距離）のPIDゲイン
    Kp[0] = 1;
    Ki[0] = 0;
    Kd[0] = 0;
    // 制御器1（人の向き）のPIDゲイン
    Kp[1] = 1;
    Ki[1] = 0;
    Kd[1] = 0;

    K[0].y = K[1].y = 0;        // K[1].yは人の距離[m]、K[1].yは人の向き[deg]
    K[0].r = 1;                 // [m],   人の距離の目標値
    K[0].r = 0;                 // [deg], 人の向きの目標値
    K[0].u = K[1].u =0;         // 制御入力
    K[0].eI= K[1].eI=0;         // eの積分値
    K[0].e_old = K[1].e_old =0; // 1サンプル過去の偏差e
    K[0].eI_old = K[1].eI_old =0; // 1サンプル過去の偏差の積分値eI
    volt[0] = volt[1] = 0;      // [V], モータ入力電圧
}


void controller(){
// 速度制御メインループ、サンプル時間TS1秒
//      親関数： timerTS0()
//      子関数： PID(), v2Hbridge0()
    void PID(int i), v2Hbridge(int i);
    int i, f_aw;

    K[0].y = 1; // [m],   人の距離koko
    K[1].y = 0; // [deg], 人の向きkoko

    PID(0);    // 人の距離の制御器
    PID(1);    // 人の向きの制御器

    volt[0] = K[0].u + K[1].u;  // 右車輪のモータへの電圧
    volt[1] = K[0].u - K[1].u;  // 左車輪のモータへの電圧

    // アンチワインドアップ：制御入力が飽和したとき積分項eIを減衰させる   anti-windup: if u=v_ref is saturated, then reduce eI. 
    f_aw = 0;
    for(i=0;i<2;i++){
        if( volt[i] > vMAX ){  // モータ入力電圧がvMAXを超えたとき
            volt[i] = vMAX;        // 電圧をvMAXにする
            f_aw = 1;
        }else if( volt[i] < -vMAX ){  // モータ入力電圧が-vMAXを超えたとき
            volt[1] = -vMAX;       // 電圧を-vMAXにする
            f_aw = 1;
        }
    }
    if( f_aw ){
        K[0].eI = K[0].eI_old; // 積分しなかったことにする
        K[1].eI = K[1].eI_old;
    }

    v2Hbridge(0);     // モータ０への入力電圧をPWMにしてHbridgeに出力 volt. to Hbridge
    v2Hbridge(1);     // モータ１への入力電圧をPWMにしてHbridgeに出力 volt. to Hbridge
}


void PID(int i){
//  制御器：偏差eが入力され、制御入力（モータ電圧）uを出力
//      入力：出力 K[i].y, 目標値 K[i].r, PID制御積分項 K[i].eI, サンプル時間 TS1 [s]
//      出力：制御入力（モータ電圧） K[i].u [V]
    float  e, ed;

    e = K[i].r - K[i].y;        // 偏差 e の計算

    K[i].eI = K[i].eI + TS1*e;  // e の積分値の計算
    K[i].eI_old = K[i].eI;      // eIの1サンプル過去の値を更新

    ed = (e-K[i].e_old)/TS1;    // e の微分値の計算
    K[i].e_old = e;             // e の1サンプル過去の値を更新

    K[i].u = Kp[i]*e + Ki[i]*K[i].eI + Kd[i]*ed; // ＰＩＤ制御器の出力を計算
}

void    v2Hbridge(int i){
// 指令電圧vより、PWM関数pwm_out()のパラメータ（dutyとフラグ）をセット。
//      親関数： timerTS0()
//      子関数： なし
//      入力：電圧指令 p.v [V]
//      出力：フルブリッジのfwdIN, rvsIN用duty,
//            デッドタイムフラグfDeadtime, モータ逆回転フラグfReverse[i]
    float           duty;      // 0-1, PWMデューティ    duty of H bridge

    duty = volt[i]/vMAX;    // 指令電圧p.vの値を最大電圧vMAXで割って-1～1にしてdutyに代入
    if( duty>=0 ){      // dutyがプラスでモータが順回転のとき
        if( fReverse[i]==1 ){       // モータが逆回転から順回転に切り替ったとき
            if( i==0 ){  pwm0p = 0;   pwm0m = 0;    // デッドタイム作成
            }else{       pwm1p = 0;   pwm1m = 0;
            }
            wait(TS1/10);           // make デッドタイム
        }
        fReverse[i] = 0;            // 逆回転フラグをオフにする
        if( i==0 ){    pwm0p = duty;    pwm0m = 0;   // dutyをPWM関数pwm_out()に渡す
        }else{         pwm1p = duty;    pwm1m = 0;
        }
    }else{             // dutyがマイナスでモータが逆回転のとき
        if( fReverse[i]==0 ){       // モータが順回転から逆回転に切り替ったとき
            if( i==0 ){  pwm0p = 0;   pwm0m = 0;    // デッドタイム作成
            }else{       pwm1p = 0;   pwm1m = 0;
            }
            wait(TS1/10);           // make デッドタイム
        }
        fReverse[i] = 1;            // 逆回転フラグをオンにする
        if( i==0 ){    pwm0p = 0;    pwm0m = -duty;   // dutyをPWM関数pwm_out()に渡す
        }else{         pwm1p = 0;    pwm1m = -duty;   // dutyをPWM関数pwm_out()に渡す
        }
    }
}


void data2mbedUSB(){    // PC上のmbed USB ディスクにセーブするためのデータをTS3[s]ごとに代入    save data to mbed USB drive 
    if( _countTS3<1000 ){   // データ数が1,000の5種類のデータをメモリーに貯める
        data[_countTS3][0]= debug[0];
        data[_countTS3][1]= debug[1];
        data[_countTS3][2]= K[0].y;
        data[_countTS3][3]= K[1].y;
        data[_countTS3][4]= _countTS0*TS0;
        _countTS3++;
    }
}
void timerTS0(){    // タイマーtimerTS0()はTS0[s]ごとにコールされる   timer called every TS0[s].
//  debug_p26 = 1;
    _countTS0++;    // カウンターに1足す
    _time += TS0;   // 現在の時間にTS0[s]足す

    controller();   // 制御器
//  debug_p26 = 0;
}

void timerTS1(void const *argument){    // タイマーtimerTS1()はTS1[s]ごとにコールされる   
}

void display2PC(){  //  PCのモニタ上のtera termに諸量を表示 display to tera term on PC
    pc.printf("%8.1f[s]\t%8.2f[V]\t%8.2f [Hz]\t%8.2f\t%8.2f\r\n",
        _time, K[0].y, K[1].y, K[0].u, debug[0]);
//    pc.printf("%8.1f[s]\t%8.5f[V]\t%4d [deg]\t%8.2f\r\n", _time, _u, (int)(_th/(2*PI)*360.0), _r);//debug[0]*3.3/R_SHUNT);  // print to tera term
}
void timerTS2(){    // タイマーtimerTS2()はTS2[s]ごとにコールされる   
}
void timerTS3(){    // タイマーtimerTS3()はTS3[s]ごとにコールされる   
    data2mbedUSB();  // PC上のmbed USB ディスクにセーブするためのデータをTS3[s]ごとに代入   data2mbedUSB() is called every TS3[s].
}
void timerTS4(){    // タイマーtimerTS4()はTS4[s]ごとにコールされる   
    display2PC();   //  PCのモニタ上のtera termに文字を表示 display to tera term on PC. display2PC() is called every TS4[s].
}
