#ifndef MBED_HYT_H
#define MBED_HYT_H

#include "mbed.h"

/**
 * IST HYT Humidity and Temperature Sensor
 *
 * @code
 * #include "mbed.h"
 * #include "HYT.h"
 * 
 * Serial              pc(USBTX, USBRX);
 * Ticker              timeKeeping;
 * HYT                 SENSOR (I2C_SDA, I2C_SCL);
 * 
 * // HYT sensor polling cycle
 * void dataUpdate(void)
 * {
 *     SENSOR.MRCommand();
 *     wait_ms(100);
 *     SENSOR.DFCommand();
 * }
 * 
 * void secondsCallback(void) {
 *     pc.printf("Humidity level: %.1f\r\n%", SENSOR.humidity);
 *     pc.printf("Temperature level: %.1f\r\n%", SENSOR.temperature);
 *     pc.printf("-------------------------------\r\n%", SENSOR.temperature);
 * }
 * 
 * int main()
 * {
 *     timeKeeping.attach(&secondsCallback, 1.0f);
 *     while(1) {
 *          dataUpdate();
 *     }
 * }
 * @endcode
 */


#define HYT_ADDR 0x50 // 01010000

class HYT
{
    
public:
    /**
     * HYT constructor.
     *
     * @param   sda mbed pin to use for SDA line of I2C interface.
     * @param   scl mbed pin to use for SCL line of I2C interface.
     * 
     * Remember about pull-up resistors on sda and scl. Recommended value is 2.4 kΩ
     */
    HYT(PinName sda, PinName scl);

    /**
     * @brief   The totals (temperature in Celsius, relative humidity in percentages)
     */
    float humidity;
    float temperature;
    
    /**
     * @brief   Send "Measuring Request" command
     * @details Initiates a measuring cycle of HYT sensor
     * @details More information: http://www.ist-ag.com/eh/ist-ag/resource.nsf/imgref/Download_AHHYTM_E2.2.5.pdf/$FILE/AHHYTM_E2.2.5.pdf
     */     
    void MRCommand(void);
    
    /**
     * @brief   Send "Data Fetch" command & processing the data
     * @details Fetch the last measured value of humidity and temperature from sensor
     * @details Calculate values of temperature in Celsius, relative humidity in percentages
     * @details More information: http://www.ist-ag.com/eh/ist-ag/resource.nsf/imgref/Download_AHHYTM_E2.2.5.pdf/$FILE/AHHYTM_E2.2.5.pdf
     * @returns 0 if no errors, -1 if no new value received from sensor.
     */    
     int DFCommand(void);
     
private:

    I2C _i2c;
};

#endif