#ifndef _LED_ARRAY_PIN_H_
#define _LED_ARRAY_PIN_H_

#include "mbed.h"
#include "LEDBus.h"

/**
    The LEDArray class allows for a bit more simplistic way of communication with an addressable LED strip
*/
class LEDArray
{

private:
    LEDBus _ledBus;
    Color** _leds;
    unsigned int _numberOfLEDs;
    
    void initialize(unsigned int numberOfLEDs);
protected:

public:


    /**
    *   Initializes the addressable led strip
    *
    *   @param wirePin - The output pin on wich the addressable leds are connected
    *   @param byteOrder - The order in wich the r, g and b bytes are expected
    *   @param numberOfLEDs - The number of leds in this strip
    *   @param t0h_us - T0H as found in the addressable led datasheet. The duration, in microseconds, the pin will stay high for sending a 0 bit
    *   @param t0l_us - T0L as found in the addressable led datasheet. The duration, in microseconds, the pin will stay low for sending a 0 bit
    *   @param t1h_us - T1H as found in the addressable led datasheet. The duration, in microseconds, the pin will stay high for sending a 1 bit
    *   @param t1l_us - T1L as found in the addressable led datasheet. The duration, in microseconds, the pin will stay low for sending a 1 bit
    *   @param tReset_us - TReset as found in the addressable led datasheet. The duration, in microsecond, the pin will stay low for sending a reset command,
    */
    LEDArray(PinName wirePin, ColorByteOrder byteOrder, unsigned int numberOfLEDs, float t0h_us, float t0l_us, float t1h_us, float t1l_us, float tReset_us);

    /**
    *   Initializes the addressable led array with default values (taken from WS2812 datasheet): 
    *   T0H = 0.35, 
    *   T0L = 0.8,
    *   T1H = 0.7,
    *   T1L = 0.6,
    *   TReset = 50 
    *
    *   @param wirePin - The output pin on wich the addressable leds are connected
    *   @param byteOrder - The order in wich the r, g and b bytes are expected
    *   @param numberOfLEDs - The number of leds in this strip
    */
    LEDArray(PinName wirePin, ColorByteOrder byteOrder, unsigned int numberOfLEDs);

    ~LEDArray();

    /**
    *   Sets the color of the given pixel (index) with r, g and b values
    */
    void setPixelColor(unsigned int pixel, uint8_t r, uint8_t g, uint8_t b);
    
    /**
    *   Sets the color of the given pixel (index) with a given color.
    */
    void setPixelColor(unsigned int pixel, Color& color);
    
    /**
    *   Sends all configured colors to the actual led strip, this method is required to call in order to update the LEDs
    */
    void show();
};

#endif