#ifndef _LED_BUS_PIN_H_
#define _LED_BUS_PIN_H_

#include "mbed.h"
#include "Color.h"

#define MICRO_SECOND 1000000.f



/**
    Callback method, supplying a Color struct to edit and the led index
*/
typedef void (*ColorGenerator)(Color* out, uint32_t index);


/**
    The LEDBus class allows for 'low' level communication to an addressable LED bus/strip 
*/
class LEDBus
{

private:
    DigitalOut _wire;
    ColorByteOrder _byteOrder;

    unsigned int _delayT1H, _delayT0H, _delayT1L, _delayT0L, _delayReset;

    void write(uint8_t byte);
    void delay(unsigned int ticks);
protected:

public:


    /**
    *   Initializes the addressable led bus
    *
    *   @param wirePin - The output pin on wich the addressable leds are connected
    *   @param byteOrder - The order in wich the r, g and b bytes are expected
    *   @param t0h_us - T0H as found in the addressable led datasheet. The duration, in microseconds, the pin will stay high for sending a 0 bit
    *   @param t0l_us - T0L as found in the addressable led datasheet. The duration, in microseconds, the pin will stay low for sending a 0 bit
    *   @param t1h_us - T1H as found in the addressable led datasheet. The duration, in microseconds, the pin will stay high for sending a 1 bit
    *   @param t1l_us - T1L as found in the addressable led datasheet. The duration, in microseconds, the pin will stay low for sending a 1 bit
    *   @param tReset_us - TReset as found in the addressable led datasheet. The duration, in microsecond, the pin will stay low for sending a reset command,
    */
    LEDBus(PinName wirePin, ColorByteOrder byteOrder, float t0h_us, float t0l_us, float t1h_us, float t1l_us, float tReset_us);

    ~LEDBus();

    /**
    *  Writes the byte buffer directly to the addressable leds
    */
    void write(uint8_t* buffer, unsigned int size);

    /**
    *  Writes the color buffer translated to bytes directly to the addressable leds
    * @code
    * // The led bus control class.
    * LEDBus ledBus(p9, RGB, 0.5f, 2.0f, 1.25f, 1.25f, 50.0f);
    *
    * Color* led1 = new Color(255,0,0);
    * Color* led2 = new Color(0,255,0);
    * Color* led3 = new Color(0,0,255);
    *
    * Color* buffer[] = { led1, led2, led3 };
    *
    * ledBus.write(buffer, 3);
    * @endcode
    */
    void write(Color** buffer, unsigned int size);

    /**
    *  Updates the leds by using a color generator callback method to generate a color for each led
        * @code
        *    int offset = 0;
        *
        *    // Sample generator:
        *    void generate(Color* color, uint32_t index)
        *    {
        *        switch((index+offset)%3) {
        *            case 0:
        *                color->red = 255;
        *                break;
        *            case 1:
        *                color->green = 255;
        *                break;
        *            case 2:
        *                color->blue = 255;
        *                break;
        *        }
        *    }
        *
        *    int main()
        *    {
        *        // The led bus control class.
        *        LEDBus ledBus(p9, RGB, 0.35f, 1.36f, 1.36f, 0.35f, 50.0f);
        *
        *        while (1) {
        *            ledBus.update(generate, 3);
        *            offset++;
        *            wait_ms(250);
        *        }
        *    }
        * @endcode
    */
    void update(ColorGenerator generator, unsigned int numberOfLEDs);

};

#endif