/**
 * @author Przemyslaw Kochanski <przemyslaw@kochanski.biz>
 *
 * @Section LICENSE
 *
 * Copyright (C) 2014 Przemyslaw Kochanski, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Library for Pololu Maestro Servo Controller
 * Serial Servo Commands: http://www.pololu.com/docs/0J40/5.e
 */

#ifndef MAESTRO
#define MAESTRO

#include "mbed.h"

/**
 * Compact Protocol Command Bytes
 */
#define SET_TARGET 0x84
#define SET_MULTIPLE_TARGETS 0x9F
#define SET_SPEED 0x87
#define SET_ACCELERATION 0x89
#define SET_PWM 0x8A
#define GET_POSITION 0x90
#define GET_MOVING_STATE 0x93
#define GET_ERRORS 0xA1
#define GO_HOME 0xA2
#define BAUD_RATE_IDICATION 0xAA

/**
 * Errors bits numbers
 */
#define SERIAL_SIGNAL_ERROR 0
#define SERIAL_OVERRUN_ERROR 1
#define SERIAL_RX_BUFFER_FULL_ERROR 2
#define SERIAL_CRC_ERROR 3
#define SERIAL_PROTOCOL_ERROR 4
#define SERIAL_TIMEOUT_ERROR 5
#define SCRIPT_STACK_ERROR 6
#define SCRIPT_CALL_STACK_ERROR 7
#define SCRIPT_PROGRAM_COUNTER_ERROR 8

/**
 * Pololu Maestro Servo Controller
 */
class Maestro
{

public:

    /**
     * Constructor
     *
     * @param tx - mbed pin to use for TX serial line
     * @param rx - mbed pin to use for RX serial line
     */
    Maestro(PinName tx, PinName rx);

    /**
     * Sets baud rate
     *
     * @param baud - Baud Rate to be set
     */
    void setBaudRate(uint16_t baud);

    /**
     * Sets the target of a channel to a specified value
     *
     * @param channel - number a servo channel
     * @param target - the pulse width to transmit in units of quarter-microseconds
     */
    void setTarget(uint8_t channel, uint16_t target);

    /**
     * Sets specified channel's servo to a specified angle
     *
     * @param channel - number a servo channel
     * @param angle - target angle of a servo
     */
    void setServoAngle(uint8_t channel, int8_t angle);

    /**
     * Simultaneously sets the targets for a contiguous block of channels
     *
     * @param count - number of channels in the contiguous block
     * @param firstChannel - lowest channel number in the contiguous block
     * @param target - target values (the pulse width to transmit in units of
     *  quarter-microseconds) for each of the channels, in order by channel number
     */
    void setMultipleTargets(uint8_t firstChannel, uint8_t count, uint16_t* targets);

    /**
     * Simultaneously sets specified contiguous block of servo channels to a specified angles
     *
     * @param count - number of servo channels in the contiguous block
     * @param firstChannel - lowest servo channel number in the contiguous block
     * @param angles - target values (the pulse width to transmit in units of
     *  quarter-microseconds) for each of the servo channels, in order by channel number
     */
    void setServosAngles(uint8_t firstChannel, uint8_t count, int8_t* angles);

    /**
     * Limits the speed at which a servo channel's output value changes
     *
     * @param channel - number of a servo channel
     * @param speed - speed of the servo in units of 0.25 us / (10 ms)
     */
    void setSpeed(uint8_t channel, uint16_t speed);
    
    /**
     * Limits the speed at which all servos channels output value changes
     *
     * @param speed - speed of the servo in units of 0.25 us / (10 ms)
     */
    void setSpeed(uint16_t speed);

    /**
     * Limits the acceleration of a servo channel's output
     *
     * @param channel - number of a servo channel
     * @param acceleration - acceleration of the servo in units of (0.25 us) / (10 ms) / (80 ms)
     */
    void setAcceleration(uint8_t channel, uint16_t acceleration);
    
    /**
     * Limits the acceleration of all servos channels output
     *
     * @param acceleration - acceleration of the servo in units of (0.25 us) / (10 ms) / (80 ms)
     */
    void setAcceleration(uint16_t acceleration);

    /**
     * Sets the PWM output to the specified on time and period
     *
     * @param channel - number of a servo channel
     * @param time - time in units of 1/48 us
     * @param period - period in units of 1/48 us
     */
    void setPWM(uint8_t channel, uint16_t time, uint16_t period);

    /**
     * Gets current servo position
     *
     * @param channel - number of a servo channel
     *
     * @return - current pulse width that the Maestro is transmitting on the channel
     */
    uint16_t getPosition(uint8_t channel);

    /**
     * Determine whether the servo outputs have reached their targets or are still changing
     *
     * @return - true if servos are moving, false otherwise
     */
    bool getMovingState();

    /**
     * Examine the errors that the Maestro has detected
     *
     * @return - error bits
     */
    uint16_t getErrors();

    /**
     * Send all servos and outputs to their home positions
     */
    void goHome();

private:

    Serial serial;
};

#endif // Maestro
