#ifndef MAP_H
#define MAP_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Paddle.h"
#include "Ball.h"
#include <vector>


/** Struct containing all parameters for the bricks  */
struct Brick {
    int x, y, w, h; // x and y coords, width, height
};


// Forward declaration
class Map;


/** Level Class
, Creates the levels in the Breakout game
@author Kostadin Chakarov, University of Leeds
@date April 2019
*/


class Level
{
public:
    /** Draws the bricks and stores them in the levels vector,
    * virtual as it changes for each level 
    */
    virtual void initBricks(Map &map) {} 
};

/** Inherits from level and doesn't have any additional functions  */
class Level1 : public Level
{
public:
    /** Draws the bricks for level 1 and stores them in the levels vector */
    virtual void initBricks(Map &map);
};

/** Inherits from level and doesn't have any additional functions  */
class Level2 : public Level
{
public:
    /** Draws the bricks for level 2 and stores them in the levels vector */
    virtual void initBricks(Map &map);
};

/** Inherits from level and doesn't have any additional functions  */
class Level3 : public Level
{
public:
    /** Draws the bricks for level 3 and stores them in the levels vector */
    virtual void initBricks(Map &map);
};


/*
Power ups Coding strategy:
1. Define the PowerUp and it's draw method, set up all it's parameters in ctor
2. Lifetime cycle management:
2.a. Identify the spawn event (e.g. when brick is killed), then go to that code
2.b. Create the power up and put it in the map's powerUps vector
2.c. During update, iterate over all powerUps and move them
2.d. In checkCollisions, iterate over all powerUps and check if any hit the pad
2.d.i.  If hit the paddle, remove and apply power-up effect
2.d.ii. else if fell below paddle height, just remove and don't apply effect
*/


// Forward declaration since needed in PowerUpType struct 
class PowerUp;

/** Struct which contains all the powerup parameters and functions */
struct PowerUpType {
    int type;       /** Stores the type of the powerup */
    int* sprite;    /** Stores the sprite of the powerup */
    
    /** Constructor */
    PowerUpType(int type, int* sprite) : type(type), sprite(sprite) {}
    
    /** Gives the powerup effect to the player */
    virtual void giveBonus(Paddle &paddle, Ball &ball) {}
    /** Chooses which powerup to draw on the LCD depending on its type */
    virtual void draw(N5110 &lcd, PowerUp &pUp);
};

/** Inherits from the powerUpType struct and only changes the giveBonus function */
struct PaddleSizePUpType : public PowerUpType {
    /** Constructor */
    PaddleSizePUpType(int type, int* sprite) : PowerUpType(type, sprite) {}
    /** Gives the paddle size increase powerup effect to the player */
    virtual void giveBonus(Paddle &paddle, Ball &ball);
};

/** Inherits from the powerUpType struct and only changes the giveBonus function */
struct PaddleSpeedPUpType : public PowerUpType {
    /** Constructor */
    PaddleSpeedPUpType(int type, int* sprite) : PowerUpType(type, sprite) {}
    /** Gives the paddle speed incrase powerup effect to the player */
    virtual void giveBonus(Paddle &paddle, Ball &ball);
};

/** Inherits from the powerUpType struct and only changes the giveBonus function */
struct BallSpeedPUpType : public PowerUpType {
    /** Constructor */
    BallSpeedPUpType(int type, int* sprite) : PowerUpType(type, sprite) {}
    /** Gives the ball speed decrease powerup effect to the player */
    virtual void giveBonus(Paddle &paddle, Ball &ball);
};

#define PowerUpW 3
#define PowerUpH 4
#define PowerUpDropChancePct 15


/** PowerUp Class
, Creates the power ups in the Breakout game
@author Kostadin Chakarov, University of Leeds
@date April 2019
*/

class PowerUp : public GameObject {
    //PowerUpType& pUpType;
    
    // we cannot store a reference because (for some reason) it is not copy-constructable
    // and for some reason, PowerUp needs to be that.
    PowerUpType* _pUpType;
    
    
    
public:
    /** Constructor 
    * @param x the initial x-position of the PowerUp
    * @param y the initial y-position of the PowerUp
    * @param pUpType the type of the PowerUp
    */
    PowerUp(float x, float y, PowerUpType& pUpType);
    /** Destructor - empty */
    ~PowerUp() {};
    
    /** Determines the bonus depending on the type of the powerup */
    void giveBonus(Paddle &paddle, Ball &ball) {
        _pUpType->giveBonus(paddle, ball);
        }
    /** Draws each powerup on the LCD after type has been assigned*/
    virtual void draw(N5110 &lcd);
};


/** Map Class
, Creates the map and controls collision between objects in the Breakout game
@author Kostadin Chakarov, University of Leeds
@date April 2019
*/

class Map
{
private:
    int currentLevel; /** Keeps track of the current level */
    std::vector<Level*> levels; /** Stores the levels in a vector */
    std::vector<Brick> bricks; /** Stores the bricks in a vector */
    std::vector<PowerUp> powerUps; /** Stores the powerups in a vector */

public:
    /** Constructor */
    Map();
    /** Destructor */
    ~Map();
    /** Gets the current level
    * @return the number of the current level
    */
    int getCurrentLevel() { return currentLevel; }
    /** Checks if all levels are defeated
    * @return true if current level == number of levels
    */
    bool hasWon() { return currentLevel >= levels.size(); }
    /** Get the level object based on the current level number and call initBricks() on it */
    void initBricks();
    /** Updates the moving powerups on map */
    void update();
    /** Creates powerUps depending on the PowerUpDropChancePct */
    void onBrickHit(const Brick&);
    /** Draws all the bricks on the level, as well as powerUps */
    void drawMap(N5110 &lcd);
    /** Checks if brick <-> ball collided */
    void checkCollision(Ball &ball, Paddle &paddle); 
    /** Resolves the collision */
    void resolveCollision(const Brick &brick, GameObject &obj); 
    /** Checks if we cleared the level and if we won 
    * @return true if level is cleared
    */
    bool checkLevel();
    /** Resets the whole map when game is (re-)started */
    void reset(); 
    /** Adds the brick to the vector */
    void addBrick(Brick& brick) { bricks.push_back(brick); }
    /** Stores the score of the game */
    int score;
};
#endif