#include "Map.h"
#include "Ball.h"

// Levels

void Level1::initBricks(Map & map)
{
    int w = 41;     //width 
    int h = 2;      //height 
    int gap = 1;    //gap between bricks
    int y = 0;      //first y coord of columns
    
    for (int j = 0; j < 9; ++j) {
        int x = 0;  //first x coord of rows
        for (int i = 0; i < 2; ++i) {
            Brick b;
            b.x = x;
            b.y = y;
            b.w = w;
            b.h = h;
            map.addBrick(b);

            x += w + gap;
        }
        y += h + gap;
    }
}


void Level2::initBricks(Map & map)
{
    int w = 20;     //width 
    int h = 2;      //height 
    int gap = 1;    //gap between bricks
    int y = 0;      //first y coord of columns

    for (int j = 0; j < 9; ++j) {
        int x = 0;  //first x coord of rows
        for (int i = 0; i < 4; ++i) {
            Brick b;
            b.x = x;
            b.y = y;
            b.w = w;
            b.h = h;
            map.addBrick(b);

            x += w + gap;
        }
        y += h + gap;
    }
}


void Level3::initBricks(Map & map)
{
    int w = 13;     //width 
    int h = 2;      //height 
    int gap = 1;    //gap between bricks
    int y = 0;      //first y coord of columns

    for (int j = 0; j < 9; ++j) {
        int x = 0;  //first x coord of rows
        for (int i = 0; i < 6; ++i) {
            Brick b;
            b.x = x;
            b.y = y;
            b.w = w;
            b.h = h;
            map.addBrick(b);

            x += w + gap;
        }
        y += h + gap;
    }
}

// Power-Ups

//Sprite for each powerup

//                              rows,cols
int powerUpPaddleSizeSprite[PowerUpH][PowerUpW] =   {
    { 0,1,0 },
    { 1,1,1 },
    { 0,1,0 },
    { 1,1,1 }
};

//                              rows,cols
int powerUpPaddleSpeedSprite[PowerUpH][PowerUpW] =   {
    { 0,1,0 },
    { 0,1,0 },
    { 1,1,1 },
    { 1,1,1 }
};

//                              rows,cols
int powerUpBallSpeedSprite[PowerUpH][PowerUpW] =   {
    { 0,1,0 },
    { 1,1,1 },
    { 1,1,1 },
    { 0,1,0 }
};


void PowerUpType::draw(N5110 &lcd, PowerUp &pUp) {
    //printf("%f, %f; %d,%d - %d\n", pUp.getPos().x,pUp.getPos().y,pUp.getH(),pUp.getW(), (int)((int*)sprite));
    lcd.drawSprite(pUp.getPos().x,pUp.getPos().y,pUp.getH(),pUp.getW(), (int*)sprite);
}

// Array containing all the powerup types 
PowerUpType* PowerUpTypes[3] = {
    new PaddleSizePUpType(0, *powerUpPaddleSizeSprite),
    new PaddleSpeedPUpType(1, *powerUpPaddleSpeedSprite),
    new BallSpeedPUpType(2, *powerUpBallSpeedSprite)
};


PowerUp::PowerUp(float x, float y, PowerUpType& pUpType) : _pUpType(&pUpType)
{
    pos.x = x;
    pos.y = y;
    velocity.x = 0;
    velocity.y = 0.5f;
    w = PowerUpW;
    h = PowerUpH;
}

void PowerUp::draw(N5110 &lcd) {
    //printf("powerup %.02f , %.02f, %d\n", pos.x, pos.y, powerUpSprite[0][2]);
    _pUpType->draw(lcd, *this);
}

void PaddleSizePUpType::giveBonus(Paddle &paddle, Ball &ball) {
    int add = 2;
    paddle.setW(paddle.getW()+add);
}

void PaddleSpeedPUpType::giveBonus(Paddle &paddle, Ball &ball) {
    float add = 0.2f;
    paddle.setSpeed(paddle.getSpeed() + add);
}

void BallSpeedPUpType::giveBonus(Paddle &paddle, Ball &ball) {
    float multiply = 0.9f;
    Vector2D& v = ball.getVelocity();
    
    // 1. create variable speed = magnitude (length) of v
    // 2. change speed by ...
    // 3. Then normalize v and multiply by speed
    float oldSpeed = sqrt(v.x*v.x + v.y*v.y);
    float speed = oldSpeed * multiply;
    
    v.x = (v.x / oldSpeed) * speed;
    v.y = (v.y / oldSpeed) * speed;
    
}

// Map

Map::Map()
{
    score = 0;                      // Initial score is zero 
    levels.push_back(new Level1()); // Stores level 1 in the vector 
    levels.push_back(new Level2()); // Stores level 2 in the vector 
    levels.push_back(new Level3()); // Stores level 3 in the vector 
    reset();                        // Initializes map 
}


Map::~Map()
{

}

void Map::initBricks()
{
    bricks.clear();

    Level *level = levels[currentLevel];
    level->initBricks(*this);
}

void Map::drawMap(N5110 &lcd)
{   
    vector<Brick>::size_type end = bricks.size();
    for (int i = 0; i < end; i++) {
        const Brick& b = bricks[i];
        lcd.drawRect(b.x,b.y,b.w,b.h,FILL_BLACK);
    }
    
    
    for (int i = 0; i < powerUps.size(); i++) {
        powerUps[i].draw(lcd);
    }
}

//See: https://stackoverflow.com/questions/31022269/collision-detection-between-two-rectangles-in-java
bool rectIntersect(
    int Ax, int Ay, int Aw, int Ah,
    float Bx, float By, float Bw, float Bh)
{
    return
        Bx + Bw > Ax &&
        By + Bh > Ay &&
        Ax + Aw > Bx &&
        Ay + Ah > By;
}

//Normals to each side of the rectangle
const Vector2D Up = {0, -1};
const Vector2D Down = {0, 1};
const Vector2D Left = {-1, 0};
const Vector2D Right = {1, 0};

//See: https://math.stackexchange.com/questions/13261/how-to-get-a-reflection-vector
// r = d - 2(d dot n)n
void reflect(Vector2D &d, const Vector2D &n)
{
    float dotProd = d.x * n.x + d.y * n.y;
    float s = 2 * dotProd;

    d.x = d.x - s * n.x;
    d.y = d.y - s * n.y;
}

void Map::resolveCollision(const Brick &b, GameObject &obj)
{
    // take velocity of the ball to determine direction and previous coords 
    float vx = obj.getVelocity().x;
    float vy = obj.getVelocity().y;
    // take the previous x and y coords 
    float x = obj.getPos().x - vx;
    float y = obj.getPos().y - vy;
    // sides of the bricks 
    float leftX = b.x;
    float rightX = b.x + b.w;

    // mid of x and y coords
    //float cx = b.x + b.w/2.0f; not used
    float cy = b.y + b.h/2.0f;

    
    Vector2D &vel = obj.getVelocity();
    float thresh = 0.1;
    // hit right 
    if (x > rightX - thresh) {
        reflect(vel, Right);
    } 
    // hit left
    else if (x < leftX + thresh) {
        reflect(vel, Left);
    } 
    // hit top
    else if (y < cy) {
        reflect(vel, Up);
    } 
    // hit bottom
    else if (y > cy) {
        reflect(vel, Down);
    } 
    else {
        // hit top
        if (vy > 0) {
            reflect(vel, Up);
        } 
        // hit bottom
        else {
            reflect(vel, Down);
        }
    }
}

void Map::checkCollision(Ball &ball, Paddle &paddle)
{
    // Check all bricks 
    vector<Brick>::size_type end = bricks.size();
    for (int i = 0; i < end; i++) {
        const Brick& b = bricks[i];
        if (rectIntersect(b.x, b.y, b.w, b.h,
                          ball.getPos().x, ball.getPos().y, ball.getW(), ball.getH())) {
            // Brick collision event! 
            
            // Bounce the ball 
            resolveCollision(b, ball);
            
            // (maybe) spawn power-up 
            onBrickHit(b);
            
            // Remove brick 
            bricks.erase(bricks.begin() + i);
            
            // Increment score 
            score = score + 10;
            break;
        
        }
    }
    
    // Check all power ups 
    for (int i = 0; i < powerUps.size(); i++) 
    {
        PowerUp& p = powerUps[i];
        // use rectIntersect to check for collisions between power-up and pad 
        if (rectIntersect(p.getPos().x, p.getPos().y, p.getW(), p.getH(),
                          paddle.getPos().x, paddle.getPos().y, paddle.getW(), paddle.getH())) {
            // If collided give bonus and remove powerup 
            p.giveBonus(paddle, ball);                
            powerUps.erase(&p);
            
            break;             
        }
                          
    }
}

void Map::onBrickHit(const Brick& brick) {
    // spawn power up 
    if ((rand() % 100) < PowerUpDropChancePct) {
        
        int type = rand() % 3;  // each power up has the same rarity 
        
        //printf("Power up - t=%d", type);
        PowerUpType& pUpType = *PowerUpTypes[type]; // assign the pUpType to the type we got 
        
        //printf(", pUpType=%d\n", pUpType.type);
        
        // store the powerup in the powerups vector 
        powerUps.push_back( 
            PowerUp(brick.x+(brick.w/2)-PowerUpW/2, brick.y, pUpType)
        );
       }
}

void Map::update() {
    for (int i = 0; i < powerUps.size(); i++) {
        powerUps[i].move();
    }
}

bool Map::checkLevel()
{   
    // cleared level 
    if (bricks.size() == 0) {
        ++currentLevel;     // increment to next level 
        // printf("cleared level! %d %d\n", currentLevel, hasWon());
        if (!hasWon()) {
            powerUps.clear();
            initBricks();   // initialize next level 
        }
        return true;
    }
    return false;
}

void Map::reset()
{
    score = 0;
    currentLevel = 0;
    initBricks();
    powerUps.clear();
}