#include "mbed.h"
#include "uart.h"


//---------------------------------------
// Hardware recources
//---------------------------------------
Serial uartPC(USBTX, USBRX);
Timeout uartTimeout;



//---------------------------------------
// Prototypes
//---------------------------------------
void UART_calcCRC16( void );
void UART_initStructure( void );
void UART_rxCallback( void );
void UART_rxTimeout( void );

//---------------------------------------
// Internal variables
//---------------------------------------
char uartBuffer[UART_BUFFER_LENGTH];



//---------------------------------------
// External variables
//---------------------------------------
sUART_handler UART_handler;



//---------------------------------------
// Global Functions
//---------------------------------------

void UART_init( void )
{
    uartPC.baud(UART_BAUDRATE);
    uartPC.format(UART_BITS, Serial::None, UART_STOPBIT );
    UART_reset();
    uartPC.attach( &UART_rxCallback, Serial::RxIrq );      // call UART_rx() on every RX interrupt
}


void UART_reset( void )
{
    UART_handler.ptrBuffer = &uartBuffer[0];
    UART_handler.ptrReadPositon = &uartBuffer[0];
    UART_handler.ptrWritePosition = &uartBuffer[0];
    UART_handler.bytesToRead = 0;
    UART_handler.bytesToWrite = 0;
    UART_handler.mode = UART_MODE_LISTEN;
}


// Only send, if UART is not listining because the same buffer is used for RX and TX
// Return 1: Sending was successfull
// Return 0: Could not send data, eithter to many bytes, zero bytes or UART is still listining
int UART_sendData( void )
{
    int i;

    // 2013-07-04 SJ: Remove check for UART_MODE_RX_READY because UART_reset() is called on UART_newFrame
   // if( (UART_handler.bytesToWrite > 0) && (UART_handler.mode == UART_MODE_RX_READY) &&  (UART_handler.bytesToWrite < UART_BUFFER_LENGTH - 2 )  ) 
    if( (UART_handler.bytesToWrite > 0) &&  (UART_handler.bytesToWrite < UART_BUFFER_LENGTH - 2 )  ) 
    {
        // Calc and add CRC16 to buffer
        UART_calcCRC16();
        // Send data
        for( i = 0; i < UART_handler.bytesToWrite; i++ ) {
            while( !uartPC.writeable() );
            uartPC.putc(uartBuffer[i]);
        }//for
        // Reset UART function, ready to receive again
        UART_reset();

        return 1;
    }//if
    else {
        return 0;
    }

}


int UART_newFrame( void )
{
    if( UART_handler.mode == UART_MODE_RX_READY ) 
    {
        // 2013-07-04 SJ: 
        #warning UART: This also resets the read pointer position!!!
        UART_reset();
        return 1;
    } 
    else 
    {
        return 0;
    }
}


// Return 0: CRC16 check failed
// Return 1: CRC16 check passed
int UART_checkReceivedCRC( void )
{
    // calculate CRC16 of received stream and compare with received CRC16

    return 1;
}



//---------------------------------------
// Internal Functions
//---------------------------------------

// Calculate CRC16 and add to buffer
void UART_calcCRC16( void )
{


}


// Function is called on RX interrupt
void UART_rxCallback( void )
{
    char tempChar;

    tempChar = uartPC.getc();

    // Only save data, if not in blocked mode
    if( UART_handler.mode == UART_MODE_LISTEN ) {
        if( UART_handler.bytesToRead < UART_BUFFER_LENGTH ) { // no buffer overflow
            // Restart timeout timer
            uartTimeout.attach_us( &UART_rxTimeout, UART_RX_TIMEOUT_US );
            // Save received byte
            *UART_handler.ptrReadPositon = tempChar;
            UART_handler.ptrReadPositon++;
            UART_handler.bytesToRead++;
        }
    }// if
}


void UART_rxTimeout( void )
{
    // Disable timeout
    uartTimeout.detach();
    UART_handler.ptrReadPositon = &uartBuffer[0];
    UART_handler.mode = UART_MODE_RX_READY;   // Transmission is ready
}

