#include "mbed.h"
#include "rs232.h"



//---------------------------------------
// Hardware recources
//---------------------------------------
Serial      rs232(p13, p14);    // TXD, RXD
Timeout     rs232Timeout;
DigitalOut  rs232RTS(p12);      // RTS 
DigitalIn   rs232CTS(p11);      // CTS


//---------------------------------------
// Prototypes
//---------------------------------------
static void RS232_reset( void );
static void RS232_rxCallback( void );
static void RS232_rxTimeout( void );


//---------------------------------------
// Internal variables
//---------------------------------------
static char RS232Buffer[RS232_BUFFER_LENGTH];
static sRS232_handler RS232_handler;



//---------------------------------------
// External variables
//---------------------------------------



//---------------------------------------
// Global Functions
//---------------------------------------
void RS232_init( void )
{
    rs232RTS = 0;
    rs232.baud(RS232_BAUDRATE);
    rs232.format(RS232_BITS, Serial::None, RS232_STOPBIT );
    RS232_reset();
    rs232.attach( &RS232_rxCallback, Serial::RxIrq );      // call RS232_rx() on every RX interrupt
}



// Return 1: Sending was successfull
// Return 0: Could not send data, eithter to many bytes or zero bytes
int RS232_sendData( char *ptrData, int NumBytes )
{
    int i;
    
    if( (NumBytes == 0) || (NumBytes >= RS232_BUFFER_LENGTH  )  )
    {
        return 0;
    }
    
    // Send data
    for( i = 0; i < NumBytes; i++ )
    {
        while( !rs232.writeable() );
        rs232.putc(*ptrData);
        ptrData++;
    }//for

    return 1;
}


// Return 1: New data available, get pointer to buffer and number of bytes to read
// Return 0: No new data available
int RS232_receiveData( char *& ptrData, int *ptrNumBytes )
{    
    // Dummy operation to avoid compiler warning
    ptrData++;

    if( RS232_handler.mode != RS232_MODE_RX_READY )
    {    
        *ptrNumBytes = 0;
        return 0;
    }

    *ptrNumBytes = RS232_handler.bytesToRead;
    ptrData = &RS232Buffer[0];
        
    RS232_reset();
    return 1;    
}



void RS232_setRTS( int level)
{
    if( level )
    {
        rs232RTS = 0;
    }
    else
    {
        rs232RTS = 1;
    }

}


int RS232_getCTS( void )
{
    return rs232CTS;
}


//---------------------------------------
// Internal Functions
//---------------------------------------

static void RS232_reset( void )
{
   RS232_handler.ptrReadPositon = &RS232Buffer[0];
   RS232_handler.bytesToRead = 0;
   RS232_handler.mode = RS232_MODE_LISTEN;
}


// Function is called on RX interrupt
void RS232_rxCallback( void )
{
    char tempChar;
    
    tempChar = rs232.getc();
    
    if( (RS232_handler.mode != RS232_MODE_LISTEN) || (RS232_handler.bytesToRead >= RS232_BUFFER_LENGTH) )
    {
        return;
    }

    // Restart timeout timer
    rs232Timeout.attach_us( &RS232_rxTimeout, RS232_RX_TIMEOUT_US );
    // Save received byte
    *RS232_handler.ptrReadPositon = tempChar;
    RS232_handler.ptrReadPositon++;
    RS232_handler.bytesToRead++;    
}


void RS232_rxTimeout( void )
{
    // Disable timeout
    rs232Timeout.detach();
    RS232_handler.mode = RS232_MODE_RX_READY;   // Transmission is ready
}
