#include "mbed.h"
#include "pwm.h"



//---------------------------------------
// Hardware recources
//---------------------------------------
PwmOut  PWM0(p26);
PwmOut  PWM1(p25);
PwmOut  PWM2(p24);
PwmOut  PWM3(p23);
PwmOut  PWM4(p22);
PwmOut  PWM5(p21);



//---------------------------------------
// Prototypes
//---------------------------------------
static int PWM_calcDutyCycle( unsigned char DutyCyclePerc );
static void PWM_updateAllDC( void );


//---------------------------------------
// Internal variables
//---------------------------------------
static int PWMPeriode_us = 0;                           // [us] actual periode time of pwm signal
static unsigned char PWMDutyCycle[CNT_ePWM_channel];    // [%] Scale: 1 DutyCycle ratio of each channel



//---------------------------------------
// External variables
//---------------------------------------



//---------------------------------------
// Global Functions
//---------------------------------------
void PWM_init( void )
{
    int i;
    
    for( i = 0; i < CNT_ePWM_channel; i++ )
    {
        PWMDutyCycle[i] = 0;
    }
    
    // Update Freq and duty cycle
    PWM_updateFreq( PWM_INIT_FREQ );
}


// Return:
// 0: freq_Hz is outside of limit
// 1: frequency was updated
int PWM_updateFreq( unsigned int freq_Hz )
{
    float tempFloat;

    // check limit
    if( (freq_Hz < PWM_FREQ_MIN) || (freq_Hz > PWM_FREQ_MAX) )
    {
        return 0;
    }
    
    // convert to period time in us
    tempFloat = 1.0 / (float)(freq_Hz);
    tempFloat *= 1000000.0;

    // Save period to global variable, update PWM and update duty cycle
    PWMPeriode_us = (int)tempFloat; 
    PWM0.period_us(PWMPeriode_us); // periode effects all 6 channels 
    PWM_updateAllDC();
    
    return 1;
}


// Return:
// 0: channel or dutyCycle is outside of limit
// 1: duty cycle was updated
int PWM_setDC( int channel, int dutyCycle )
{          
    // Check duty cycle and channel limits
    if( channel >= CNT_ePWM_channel )
    {
        return 0;
    }
    
    if( dutyCycle < 0 || dutyCycle > PWM_DC_MAX )
    {
        return 0;
    }
      
    // update buffer array of channel and update duty cycle register of all channel
    PWMDutyCycle[channel] = dutyCycle;
    PWM_updateAllDC();
 
    return 1;
}



//---------------------------------------
// Internal Functions
//---------------------------------------
static void PWM_updateAllDC( void )
{
    // update duty cycle of all channels    
    PWM0.pulsewidth_us( PWM_calcDutyCycle( PWMDutyCycle[PWM_CH0] ) );
    PWM1.pulsewidth_us( PWM_calcDutyCycle( PWMDutyCycle[PWM_CH1] ) );
    PWM2.pulsewidth_us( PWM_calcDutyCycle( PWMDutyCycle[PWM_CH2] ) );
    PWM3.pulsewidth_us( PWM_calcDutyCycle( PWMDutyCycle[PWM_CH3] ) );
    PWM4.pulsewidth_us( PWM_calcDutyCycle( PWMDutyCycle[PWM_CH4] ) );
    PWM5.pulsewidth_us( PWM_calcDutyCycle( PWMDutyCycle[PWM_CH5] ) ); 
}


// Calculate duty cycle register value with respect to period time
static int PWM_calcDutyCycle( unsigned char DutyCyclePerc )
{
    if( DutyCyclePerc == 0 )
    {
        return 0;
    }
    
    if( DutyCyclePerc >= 100 )
    {
        return PWMPeriode_us + 1;
    }
    
    return (int)( (float)PWMPeriode_us * ( (float)DutyCyclePerc / 100.0) );
}
