#include "mbed.h"
#include "uart.h"
#include "interpret.h"
#include "mbed_unit.h"
#include "rs232.h"
#include "can.h"
#include "pwm.h"
#include "ana.h"
#include "dido.h"
#include "RS485.h"


//---------------------------------------
// Hardware recources
//---------------------------------------


DigitalOut K1_Relay(P0_29);
DigitalOut K2_Relay(P0_30);
 

//---------------------------------------
// Prototypes
//---------------------------------------



//---------------------------------------
// Internal variables
//---------------------------------------


//---------------------------------------
// External variables
//---------------------------------------



//---------------------------------------
// Global Functions
//---------------------------------------
void MBED_init( void )
{
    PWM_init();
    RS232_init();
    CAN_init();
    ANA_init();
    DIDO_init();
    K1_Relay=0;
    K2_Relay=0;
}



// process commands
void MBED_deviceID_process( void )
{
    unsigned int temp32U;
    int temp32;
    char *ptrTempChar;
    int i;

    switch( uartBuffer[ INT_BUF_COMMAND ] ) {
            //---------------------------
        case MBED_SET_RPM_LEVEL_LOW:
            //---------------------------
        {
              if( uartBuffer[INT_BUF_1DATA] ) 
                K1_Relay=1;
                  else 
                K1_Relay=0;
        
            // Generate acknowledge
            INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            break;
        }


        //---------------------------
        case MBED_SET_RPM_LEVEL_HIGH:
            //---------------------------
        {
              if( uartBuffer[INT_BUF_1DATA] ) 
                K2_Relay=1;
                  else 
                K2_Relay=0;
      
            // Generate acknowledge
            INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            break;

        }


        //---------------------------
        case MBED_SET_PWM_PORT:
            //---------------------------
        {
            // Set output high or low
            if( uartBuffer[INT_BUF_1DATA] & (1 << PWM_CH0) ) PWM_setDC(PWM_CH0, 100);
            else PWM_setDC(PWM_CH0, 0);
            if( uartBuffer[INT_BUF_1DATA] & (1 << PWM_CH1) ) PWM_setDC(PWM_CH1, 100);
            else PWM_setDC(PWM_CH1, 0);
            if( uartBuffer[INT_BUF_1DATA] & (1 << PWM_CH2) ) PWM_setDC(PWM_CH2, 100);
            else PWM_setDC(PWM_CH2, 0);
            if( uartBuffer[INT_BUF_1DATA] & (1 << PWM_CH3) ) PWM_setDC(PWM_CH3, 100);
            else PWM_setDC(PWM_CH3, 0);
            if( uartBuffer[INT_BUF_1DATA] & (1 << PWM_CH4) ) PWM_setDC(PWM_CH4, 100);
            else PWM_setDC(PWM_CH4, 0);
            if( uartBuffer[INT_BUF_1DATA] & (1 << PWM_CH5) ) PWM_setDC(PWM_CH5, 100);
            else PWM_setDC(PWM_CH5, 0);

            // Generate acknowledge
            INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);

            break;
        }


        //---------------------------
        case MBED_SET_PWM_FREQ:
            //---------------------------
        {
            // Get frequency data in [Hz]
            temp32U = 0;

            temp32U = uartBuffer[INT_BUF_1DATA];
            temp32U <<= 8;
            temp32U |= (unsigned int)uartBuffer[INT_BUF_1DATA + 1];

            // check limit
            if( PWM_updateFreq(temp32U) ) {
                // Generate acknowledge
                INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            } else { // outside of supported limit
                // Generate acknowledge
                INT_generateACKFrame(INT_ID_MBED, INT_COM_VAL_NOTVALID);
            }

            break;
        }


        //---------------------------
        case MBED_SET_PWM_DC:
            //---------------------------
        {
            if( PWM_setDC((int)uartBuffer[INT_BUF_1DATA], (int)uartBuffer[INT_BUF_1DATA + 1]) ) {
                // Generate acknowledge
                INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            } else { // outside of supported limit
                // Generate acknowledge
                INT_generateACKFrame(INT_ID_MBED, INT_COM_VAL_NOTVALID);
            }
            break;

        }


        //---------------------------
        case MBED_WRITE_RS232:
            //---------------------------
        {
            if( RS232_sendData( (char *)&uartBuffer[INT_BUF_1DATA], uartBuffer[INT_BUF_NUM] ) ) {
                INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            } else {
                INT_generateACKFrame(INT_ID_MBED, INT_COM_VAL_NOTVALID);
            }

            break;
        }


        //---------------------------
        case MBED_READ_RS232:
            //---------------------------
        {
            RS232_receiveData( ptrTempChar, &temp32 );

            // Generate acknowledge
            uartBuffer[INT_BUF_DEVICEID] = INT_ID_MBED;
            uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
            uartBuffer[INT_BUF_NUM] = temp32;
            // copy data over to uart buffer
            for( i = 0; i < temp32; i++) {
                uartBuffer[INT_BUF_1DATA + i] = *ptrTempChar;
                ptrTempChar++;
            }
            UART_handler.bytesToWrite = 3 + temp32;

            break;
        }


        //---------------------------
        case MBED_SET_RTSRS232:
            //---------------------------
        {
            RS232_setRTS( (int)uartBuffer[INT_BUF_1DATA] );
            INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);

            break;
        }


        //---------------------------
        case MBED_GET_CTSRS232:
            //---------------------------
        {
            // Generate acknowledge
            uartBuffer[INT_BUF_DEVICEID] = INT_ID_MBED;
            uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
            uartBuffer[INT_BUF_NUM] = 1;
            uartBuffer[INT_BUF_1DATA] = RS232_getCTS();
            UART_handler.bytesToWrite = 4;

            break;
        }


        //---------------------------
        case MBED_WRITE_CAN:
            //---------------------------
        {
            // Get CAN-ID
            temp32 = 0;

            temp32 = uartBuffer[INT_BUF_1DATA];
            temp32 <<= 8;
            temp32 |= (int)uartBuffer[INT_BUF_1DATA + 1];

            if( CAN_send( temp32, uartBuffer[INT_BUF_1DATA + 2], &uartBuffer[INT_BUF_1DATA + 3] ) ) {
                INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            } else {
                INT_generateACKFrame(INT_ID_MBED, INT_COM_VAL_NOTVALID);
            }

            break;
        }


        //---------------------------
        case MBED_READ_CAN:
            //---------------------------
        {
            // Get CAN-ID
            temp32 = 0;

            temp32 = uartBuffer[INT_BUF_1DATA];
            temp32 <<= 8;
            temp32 |= (int)uartBuffer[INT_BUF_1DATA + 1];

            if( CAN_get( &temp32, &uartBuffer[INT_BUF_1DATA + 2], &uartBuffer[INT_BUF_1DATA + 3] ) ) {
                // Generate acknowledge
                uartBuffer[INT_BUF_DEVICEID] = INT_ID_MBED;
                uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
                uartBuffer[INT_BUF_NUM] = 3 + uartBuffer[INT_BUF_1DATA + 2];        //
                uartBuffer[INT_BUF_1DATA] = (char)(0x000000FF & (temp32 >> 8) );    // CAN-ID high byte
                uartBuffer[INT_BUF_1DATA + 1] = (char)(0x000000FF & temp32 );       // CAN-ID low byte

                UART_handler.bytesToWrite = 6 + uartBuffer[INT_BUF_1DATA + 2];      //
            } else {
                // Generate acknowledge, no new data frame
                uartBuffer[INT_BUF_DEVICEID] = INT_ID_MBED;
                uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
                uartBuffer[INT_BUF_NUM] = 3;        //
                uartBuffer[INT_BUF_1DATA] = 0;      // CAN-ID high byte
                uartBuffer[INT_BUF_1DATA + 1] = 0 ; // CAN-ID low byte
                uartBuffer[INT_BUF_1DATA + 2] = 0 ; // 0 Data frames
                UART_handler.bytesToWrite = 6;      //
            }

            break;
        }


        //---------------------------
        case MBED_READ_ADC:
            //---------------------------
        {
            // Get ADC value of channel indicated in uartBuffer[]
            temp32U = (unsigned int)ANA_getFiltered( (int)uartBuffer[INT_BUF_1DATA] );
            // Generate acknowledge
            uartBuffer[INT_BUF_DEVICEID] = INT_ID_MBED;
            uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
            uartBuffer[INT_BUF_NUM] = 3;
            // uartBuffer[INT_BUF_1DATA] = KEEP channel number
            uartBuffer[INT_BUF_1DATA + 1] = (char)(0x000000FF & (temp32U >> 8) );    // high byte
            uartBuffer[INT_BUF_1DATA + 2] = (char)(0x000000FF & temp32U );           // low byte

            UART_handler.bytesToWrite = 6 ;      //

            break;
        }


        //---------------------------
        case MBED_SET_RELAY:
            //---------------------------
        {
            if( uartBuffer[INT_BUF_1DATA] ) {
                DIDO_setRelayON();
            } else {
                DIDO_setRelayOFF();
            }

            INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            break;
        }


        //---------------------------
        case MBED_SET_DO1:
            //---------------------------
        {
            if( uartBuffer[INT_BUF_1DATA] ) {
                DIDO_setDO1ON();
            } else {
                DIDO_setDO1OFF();
            }

            INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            break;
        }


        //---------------------------
        case MBED_SET_DO2:
            //---------------------------
        {
            if( uartBuffer[INT_BUF_1DATA] ) {
                DIDO_setDO2ON();
            } else {
                DIDO_setDO2OFF();
            }

            INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            break;
        }

        //---------------------------
        case MBED_GET_DI:
            //---------------------------
        {
            temp32U = 0;

            if( DIDO_getDI1() ) {
                temp32U |= (1 << 0);
            }

            if( DIDO_getDI2() ) {
                temp32U |= (1 << 1);
            }

            if( DIDO_getDI3() ) {
                temp32U |= (1 << 2);
            }

            if( DIDO_getDI4() ) {
                temp32U |= (1 << 3);
            }

            if( DIDO_getDI5() ) {
                temp32U |= (1 << 4);
            }

            // Generate acknowledge
            uartBuffer[INT_BUF_DEVICEID] = INT_ID_MBED;
            uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
            uartBuffer[INT_BUF_NUM] = 1;
            uartBuffer[INT_BUF_1DATA] = (char)(0x000000FF & temp32U );
            UART_handler.bytesToWrite = 4 ;      //

            break;
        }


        //---------------------------
        case MBED_WRITE_RS485:
            //---------------------------
        {
            if( RS485_sendData( (char *)&uartBuffer[INT_BUF_1DATA], (int)uartBuffer[INT_BUF_NUM] ) ) {
                INT_generateACKFrame(INT_ID_MBED, INT_COM_ACK);
            } else {
                INT_generateACKFrame(INT_ID_MBED, INT_COM_VAL_NOTVALID);
            }

            break;
        }


        //---------------------------
        case MBED_READ_RS485:
            //---------------------------
        {
            RS485_receiveData( ptrTempChar, (int *)&uartBuffer[INT_BUF_NUM] );

            // Generate acknowledge
            uartBuffer[INT_BUF_DEVICEID] = INT_ID_MBED;
            uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
            // copy data over to uart buffer
            for( i = 0; i < uartBuffer[INT_BUF_NUM]; i++) {
                uartBuffer[INT_BUF_1DATA + i] = *ptrTempChar;
                ptrTempChar++;
            }
            UART_handler.bytesToWrite = 3 + uartBuffer[INT_BUF_NUM];

            break;
        }


        //-----------------
        default:
            //-----------------
        {
            // Command not supported
            INT_generateACKFrame(INT_ID_MBED, INT_COM_COM_NOTSUPP);
        }

    }//switch

}

//---------------------------------------
// Internal Functions
//---------------------------------------
