#include "interpret.h"
#include "uart.h"
#include "mbed_unit.h"
#include "pt100.h"
#include "di.h"
#include "do.h"
#include "RS485.h"


//---------------------------------------
// Hardware recources
//---------------------------------------
DigitalOut led_system_ready(LED1);


//---------------------------------------
// Prototypes
//---------------------------------------

void INT_selectDeviceID( void );
void INT_deviceID_process_Broadcast( void );


//---------------------------------------
// Internal variables
//---------------------------------------
static sINT_handler INT_handler;

//---------------------------------------
// External variables
//---------------------------------------


//---------------------------------------
// Global Functions
//---------------------------------------

void INT_init( void )
{
    INT_handler.state = INT_STATE_INIT; 
    INT_handler.receivedDeviceID = 0;
    INT_handler.receivedCommand = 0;
    led_system_ready = 0;
}



void INT_poll( void )
{
    switch( INT_handler.state )
    {
        //---------------------------
        case INT_STATE_INIT:
        //---------------------------
        {
            led_system_ready = 0;
            DI_init();
            MBED_init();  // PWM, CAN, RS232,...
            PT100_init();
            DO_init();
            RS485_init();            
            led_system_ready = 1;            
            INT_handler.state = INT_STATE_IDLE;
            
            break;
        }
        
    
        //---------------------------
        case INT_STATE_IDLE:
        //---------------------------
        {
            if( UART_newFrame() )
            {
                INT_handler.state = INT_STATE_CHECK;
            }
        
            break;
        }
        
        
        //---------------------------
        case INT_STATE_CHECK:
        //---------------------------
        {
            //---------------------------------
            if( UART_checkReceivedCRC() )
            //---------------------------------
            {
                // backup device ID and command for acknowledge
                INT_handler.receivedDeviceID = uartBuffer[INT_BUF_DEVICEID];
                INT_handler.receivedCommand = uartBuffer[INT_BUF_COMMAND];
                INT_handler.state = INT_STATE_INTERPRET;                    
            }
            //------------------------
            else // CRC check failed --> ignore frame
            //------------------------
            {
                UART_reset();
                INT_handler.state = INT_STATE_IDLE;
            }
       
            break;
        }
        
        
        //---------------------------
        case INT_STATE_INTERPRET:
        //---------------------------
        {
            INT_selectDeviceID();
            INT_handler.state = INT_STATE_ACK;
            break;
        }
        
        
        //---------------------------
        case INT_STATE_ACK:
        //---------------------------
        {     
            // INT_selectDeviceID() has already updated UART_buffer[] --> just send
            UART_sendData();
            INT_handler.state = INT_STATE_IDLE;
            break;
        }
        
        
        //---------------------------
        default:
        //---------------------------
        {
            INT_handler.state = INT_STATE_INIT; 
        }   
    
    }//switch

}


void INT_generateACKFrame(char deviceID, char command)
{
    uartBuffer[INT_BUF_DEVICEID] = deviceID;
    uartBuffer[INT_BUF_COMMAND] = command;
    uartBuffer[INT_BUF_NUM] = 0;
    UART_handler.bytesToWrite = 3;
}


//---------------------------------------
// Local Functions
//---------------------------------------


// Here, the DeviceID gets used to communicate with the requested HW-unit
// Modify this function, if new HW-units are integreted.
// The XXX_deviceID_process() function also generates the acknowledge frame and
// place it in the uartBuffer[].
// The next state, INT_STATE_ACK, of the Interprete FSM is than sending these data
void INT_selectDeviceID( void )
{
    switch( INT_handler.receivedDeviceID )
    {
        //----------------------
        case INT_ID_BROADCAST:
        //----------------------
        {
            INT_deviceID_process_Broadcast();
            break;
        }
        
     
        //----------------------
        case INT_ID_MBED:
        //----------------------
        {
            MBED_deviceID_process();
            break;
        }       
    
    
        //----------------------
        case INT_ID_PT100:
        //----------------------
        {
            PT100_deviceID_process();
            break;
        }
    
    
        //----------------------
        case INT_ID_DI:
        //----------------------
        {
            DI_deviceID_process();
            break;
        }
    
    
        //----------------------
        case INT_ID_DO:
        //----------------------
        {
            DO_deviceID_process();
            break;
        }
        
    
        //----------------------
        default:
        //----------------------
        {
            // Device ID is not supported, send ack.
            INT_generateACKFrame(INT_handler.receivedDeviceID, INT_COM_DEV_NOTSUPP);
        }
    }//switch
}




// Interprete Command for broadcast
void INT_deviceID_process_Broadcast( void )
{


}
