#include "mbed.h"
#include "uart.h"
#include "interpret.h"
#include "di.h"
#include "MCP23017.h"



//---------------------------------------
// Hardware recources
//---------------------------------------
MCP23017 *DI_PIN1_8_I2C;    // Pin: 1..8 and Low, High level reference
MCP23017 *DI_PIN9_24_I2C;   // Pin: 9..24


//---------------------------------------
// Prototypes
//---------------------------------------




//---------------------------------------
// Internal variables
//---------------------------------------
static int DI_PinMapping[ DI_NUMBER_PINS ];  // array, keeps MCP23017 pin number for easy access



//---------------------------------------
// External variables
//---------------------------------------



//---------------------------------------
// Global Functions
//---------------------------------------
void DI_init( void )
{
    DI_PinMapping[0] = DI_PIN1;
    DI_PinMapping[1] = DI_PIN2;
    DI_PinMapping[2] = DI_PIN3;
    DI_PinMapping[3] = DI_PIN4;
    DI_PinMapping[4] = DI_PIN5;
    DI_PinMapping[5] = DI_PIN6;
    DI_PinMapping[6] = DI_PIN7;
    DI_PinMapping[7] = DI_PIN8;
    DI_PinMapping[8] = DI_PIN9;
    DI_PinMapping[9] = DI_PIN10;
    DI_PinMapping[10] = DI_PIN11;
    DI_PinMapping[11] = DI_PIN12;
    DI_PinMapping[12] = DI_PIN13;
    DI_PinMapping[13] = DI_PIN14;
    DI_PinMapping[14] = DI_PIN15;
    DI_PinMapping[15] = DI_PIN16;
    DI_PinMapping[16] = DI_PIN17;
    DI_PinMapping[17] = DI_PIN18;
    DI_PinMapping[18] = DI_PIN19;
    DI_PinMapping[19] = DI_PIN20;
    DI_PinMapping[20] = DI_PIN21;
    DI_PinMapping[21] = DI_PIN22;
    DI_PinMapping[22] = DI_PIN23;
    DI_PinMapping[23] = DI_PIN24;

    DI_PIN1_8_I2C = new MCP23017( p9, p10, DI_PIN1_8_I2C_ADD);
    DI_PIN1_8_I2C->config(0, 0, 0x00FF);            // set to outputs, no pull-ups, not-inverted

  
    DI_PIN9_24_I2C = new MCP23017( p9, p10, DI_PIN9_24_I2C_ADD);
    DI_PIN9_24_I2C->config(0, 0, 0x00FF);           // set to outputs, no pull-ups, not-inverted

    
    DI_PIN1_8_I2C->write_bit(0, DI_LEVEL_LOW);
    DI_PIN1_8_I2C->write_bit(0, DI_LEVEL_HIGH); 
    DI_PIN1_8_I2C->write_mask(0x00FF, 0x00FF);      // set outputs high
    DI_PIN9_24_I2C->write_mask(0xFFFF, 0xFFFF);     // set outputs high  
}



// process commands
void DI_deviceID_process( void )
{
    unsigned int temp32U;
    unsigned short portTemp = 0;
    int i;
    int j;
    
    // Get data before processing and save to temp32U;
    // maximal 3 bytes
    temp32U = 0;
    for( i = 0; i < 3; i++ )
    {
        temp32U <<= 8;
        temp32U |= ( 0x000000FF & (unsigned int)(uartBuffer[INT_BUF_1DATA + i]) );
    }
    
    
    switch( uartBuffer[ INT_BUF_COMMAND ] )
    {
        //--------------------
        case DI_SET_LEVEL_LOW:
        //--------------------
        {
            // Write to DI HW
            if( uartBuffer[INT_BUF_1DATA] )
            {
                DI_PIN1_8_I2C->write_bit(1, DI_LEVEL_LOW);
            }
            else
            {
                DI_PIN1_8_I2C->write_bit(0, DI_LEVEL_LOW);
            }
            // Generate acknowledge
            INT_generateACKFrame(INT_ID_DI, INT_COM_ACK);

            break;
        }
        
        //--------------------
        case DI_GET_LEVEL_LOW:
        //--------------------
        {
            temp32U = DI_PIN1_8_I2C->read_bit(DI_LEVEL_LOW);
            
            // Generate acknowledge
            uartBuffer[INT_BUF_DEVICEID] = INT_ID_DI;
            uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
            uartBuffer[INT_BUF_NUM] = 1;
            uartBuffer[INT_BUF_1DATA] = (char)(0x000000FF & temp32U);
            UART_handler.bytesToWrite = 4;
        
            break;
        }  
        
        //--------------------
        case DI_SET_LEVEL_HIGH:
        //--------------------
        {            
            // Write to DI HW
            if( uartBuffer[INT_BUF_1DATA] )
            {
                DI_PIN1_8_I2C->write_bit(1, DI_LEVEL_HIGH);
            }
            else
            {
                DI_PIN1_8_I2C->write_bit(0, DI_LEVEL_HIGH);
            }
            // Generate acknowledge
            INT_generateACKFrame(INT_ID_DI, INT_COM_ACK);

            break;
        }
        
        //--------------------
        case DI_GET_LEVEL_HIGH:
        //--------------------
        {
            temp32U = DI_PIN1_8_I2C->read_bit(DI_LEVEL_HIGH);
            
            // Generate acknowledge
            uartBuffer[INT_BUF_DEVICEID] = INT_ID_DI;
            uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
            uartBuffer[INT_BUF_NUM] = 1;
            uartBuffer[INT_BUF_1DATA] = (char)(0x000000FF & temp32U);
            UART_handler.bytesToWrite = 4;
        
            break;
        }  
        
        
        //--------------------
        case DI_SET_SINGLE:
        //--------------------
        {
            // Limit check of pin number
            if( (uartBuffer[INT_BUF_1DATA] == 0) || (uartBuffer[INT_BUF_1DATA] > DI_NUMBER_PINS) )
            {
                INT_generateACKFrame(INT_ID_DI, INT_COM_VAL_NOTVALID);        
                break;
            }
            
            if( uartBuffer[INT_BUF_1DATA] > 8 ) // --> write to: DI_PIN9_24_I2C
            {
                DI_PIN9_24_I2C->write_bit(0, DI_PinMapping[uartBuffer[INT_BUF_1DATA] - 1]);
                // Generate acknowledge
                INT_generateACKFrame(INT_ID_DI, INT_COM_ACK);
                break;
            }
            
            // otherwise:--> write to: DI_PIN1_8_I2C
            DI_PIN1_8_I2C->write_bit(0, DI_PinMapping[uartBuffer[INT_BUF_1DATA] - 1]);
            // Generate acknowledge
            INT_generateACKFrame(INT_ID_DI, INT_COM_ACK);
            break; 
        }
        
        
        //--------------------
        case DI_CLEAR_SINGLE:
        //--------------------
        {
            // Limit check of pin number
            if( (uartBuffer[INT_BUF_1DATA] == 0) || (uartBuffer[INT_BUF_1DATA] > DI_NUMBER_PINS) )
            {
                INT_generateACKFrame(INT_ID_DI, INT_COM_VAL_NOTVALID);        
                break;
            }
            
            if( uartBuffer[INT_BUF_1DATA] > 8 ) // --> write to: DI_PIN9_24_I2C
            {
                DI_PIN9_24_I2C->write_bit(1, DI_PinMapping[uartBuffer[INT_BUF_1DATA] - 1]);
                // Generate acknowledge
                INT_generateACKFrame(INT_ID_DI, INT_COM_ACK);
                break;
            }
            
            // otherwise:--> write to: DI_PIN1_8_I2C
            DI_PIN1_8_I2C->write_bit(1, DI_PinMapping[uartBuffer[INT_BUF_1DATA] - 1]);
            // Generate acknowledge
            INT_generateACKFrame(INT_ID_DI, INT_COM_ACK);
            break; 
        }
        
        
        //--------------------
        case DI_GET_SINGLE:
        //--------------------
        {
            // Limit check of pin number
            if( (uartBuffer[INT_BUF_1DATA] == 0) || (uartBuffer[INT_BUF_1DATA] > DI_NUMBER_PINS) )
            {
                INT_generateACKFrame(INT_ID_DI, INT_COM_VAL_NOTVALID);        
                break;
            }
            
            if( uartBuffer[INT_BUF_1DATA] > 8 ) // --> read from: DI_PIN9_24_I2C
            {
                temp32U = DI_PIN9_24_I2C->read_bit( DI_PinMapping[uartBuffer[INT_BUF_1DATA] - 1] );
            }
            else // otherwise:--> read from: DI_PIN1_8_I2C
            {
                temp32U = DI_PIN1_8_I2C->read_bit( DI_PinMapping[uartBuffer[INT_BUF_1DATA] - 1] );
            }
            
            // Invert because DO function is inverted      
            temp32U = (~temp32U & 0x00000001);
                            
            // Generate acknowledge
            uartBuffer[INT_BUF_DEVICEID] = INT_ID_DI;
            uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
            uartBuffer[INT_BUF_NUM] = 1;
            uartBuffer[INT_BUF_1DATA] = (char)(0x000000FF & temp32U);
            UART_handler.bytesToWrite = 4;
            
            break;    
        }
          
          
        //--------------------
        case DI_SET_PORT:
        //--------------------
        { 
            // process Pin: 1..8 --> write to: DI_PIN1_8_I2C
            portTemp = 0;
            for( i = 0; i < 8; i++ )
            {
               if( temp32U & (1 << i) ) // Bit is set --> set coresponding pin
               {
                    portTemp |= ( 1 << DI_PinMapping[i]);
               }
            }
            DI_PIN1_8_I2C->write_mask((~portTemp & 0x00FF), 0x00FF);   // set outputs
            

            // process Pin: 9..24 --> write to: DI_PIN9_24_I2C
            portTemp = 0;
            for( i = 8; i < 24; i++ )
            {
               if( temp32U & (1 << i) ) // Bit is set --> set coresponding pin
               {
                    portTemp |= ( 1 << DI_PinMapping[i]);
               }
            }
            DI_PIN9_24_I2C->write_mask(~portTemp, 0xFFFF);   // set outputs
            
            // Generate acknowledge
            INT_generateACKFrame(INT_ID_DI, INT_COM_ACK);
            
            break;
        }
        
        
        //--------------------
        case DI_GET_PORT:
        //--------------------
        { 
            temp32U = 0;
            
            // process Pin: 1..8 --> read to: DI_PIN1_8_I2C
            portTemp = ~( DI_PIN1_8_I2C->read_mask(0x00FF) );
            
            for( i = 0; i < 8; i++ ) {
                if(  portTemp & (1 << i) ) {
                    for( j = 0; j < 16; j++ ) {
                        if( DI_PinMapping[j] == i ) {
                            temp32U |= (1 << j);
                        }
                    }//for j

                }// if
            }// for i
           
            
            // process Pin: 9..24 --> read to: DI_PIN9_24_I2C
            portTemp = ~( DI_PIN9_24_I2C->read_mask(0xFFFF) );
            
            for( i = 0; i < 16; i++ ) {
                if(  portTemp & (1 << i) ) {
                    for( j = 8; j < 24; j++ ) {
                        if( DI_PinMapping[j] == i ) {
                            temp32U |= (1 << j);
                        }
                    }//for j

                }// if
            }// for i

            // Generate acknowledge
            uartBuffer[INT_BUF_DEVICEID] = INT_ID_DI;
            uartBuffer[INT_BUF_COMMAND] = INT_COM_ACK;
            uartBuffer[INT_BUF_NUM] = 3;
            uartBuffer[INT_BUF_1DATA]       = (char)(0x000000FF & (temp32U >> 16));
            uartBuffer[INT_BUF_1DATA + 1]   = (char)(0x000000FF & (temp32U >> 8) );
            uartBuffer[INT_BUF_1DATA + 2]   = (char)(0x000000FF & (temp32U) );
            UART_handler.bytesToWrite = 6;
            break;
        
        }

        
        //-----------------
        default:
        //-----------------
        {
            // Command not supported
            INT_generateACKFrame(INT_ID_DI, INT_COM_COM_NOTSUPP);
        }     
    }//switch
    
}

//---------------------------------------
// Internal Functions
//---------------------------------------

