#include "mbed.h"
#include "ana.h"
#include "filter.h"


//---------------------------------------
// Hardware recources
//---------------------------------------
AnalogIn ana1(p15);
AnalogIn ana2(p16);
AnalogIn ana3(p17);
Ticker ana_sw_timer;


//---------------------------------------
// Prototypes
//---------------------------------------
static void ANA_periodical( void );



//---------------------------------------
// Internal variables
//---------------------------------------
static sFLT_LP1Order16BitHandler filterHandlerAna1;
static sFLT_LP1Order16BitHandler filterHandlerAna2;
static sFLT_LP1Order16BitHandler filterHandlerAna3;
static unsigned short ANA_rawSamples[CNT_eANA_Channels];
static unsigned short ANA_filteredSamples[CNT_eANA_Channels];


//---------------------------------------
// External variables
//---------------------------------------



//---------------------------------------
// Global Functions
//---------------------------------------
void ANA_init( void )
{   
    int i;
    
    for( i = 0; i < CNT_eANA_Channels; i++ )
    {
        ANA_rawSamples[i] = 0;
        ANA_filteredSamples[i] = 0;
    }

    filterHandlerAna1.filterShift = ANA_LP_FILTER_SHIFT_ANA1;
    filterHandlerAna2.filterShift = ANA_LP_FILTER_SHIFT_ANA2;
    filterHandlerAna3.filterShift = ANA_LP_FILTER_SHIFT_ANA3;
    
    FLT_LP_1Order_16Bit_Init( &filterHandlerAna1 );
    FLT_LP_1Order_16Bit_Init( &filterHandlerAna2 );
    FLT_LP_1Order_16Bit_Init( &filterHandlerAna3 );
    
    ana_sw_timer.attach_us(&ANA_periodical, ANA_SW_TIMER_PERIOD );
}


unsigned short ANA_getRaw(int channel)
{

    if( (channel < 0) || (channel >= CNT_eANA_Channels) )
    {
        return 0;
    }
    
    return ANA_rawSamples[channel];
}


unsigned short ANA_getFiltered( int channel )
{
    if( (channel < 0) || (channel >= CNT_eANA_Channels) )
    {
        return 0;
    }
    
    return ANA_filteredSamples[channel];
}


float ANA_scale(float offset, float gain, unsigned short input)
{
    return (float)input * gain - offset;
}



//---------------------------------------
// Internal Functions
//---------------------------------------

static void ANA_periodical( void )
{ 
    ANA_rawSamples[ANA_CH1] = ana1.read_u16();
    ANA_rawSamples[ANA_CH2] = ana2.read_u16();
    ANA_rawSamples[ANA_CH3] = ana3.read_u16();

    ANA_filteredSamples[ANA_CH1] = FLT_LP_1Order_16Bit( &filterHandlerAna1, ANA_rawSamples[ANA_CH1] );   
    ANA_filteredSamples[ANA_CH2] = FLT_LP_1Order_16Bit( &filterHandlerAna2, ANA_rawSamples[ANA_CH2] );   
    ANA_filteredSamples[ANA_CH3] = FLT_LP_1Order_16Bit( &filterHandlerAna3, ANA_rawSamples[ANA_CH3] );   
}
