#include "mbed.h"
#include "RS485.h"
#include "crc16.h"


//---------------------------------------
// Hardware recources
//---------------------------------------
Serial uartRS485(p28, p27);
DigitalOut RS485RTS( p20 );
Timeout RS485Timeout;
Timeout RS485TimeoutTx;



//---------------------------------------
// Prototypes
//---------------------------------------
void RS485_initStructure( void );
void RS485_rxCallback( void );
void RS485_rxTimeout( void );
void RS485_txTimeout( void );

// void testcomm(void);  // UpdateAll

//---------------------------------------
// Internal variables
//---------------------------------------
char RS485Buffer[RS485_BUFFER_LENGTH];



//---------------------------------------
// External variables
//---------------------------------------
sRS485_handler RS485_handler;



//---------------------------------------
// Global Functions
//---------------------------------------

void RS485_init( void )
{
    uartRS485.baud(RS485_BAUDRATE);
 //   uartRS485.format(RS485_BITS, Serial::None, RS485_STOPBIT );
    uartRS485.format(RS485_BITS, Serial::Even, RS485_STOPBIT );   // 21.01.2014 KED
    RS485RTS = 0;
    RS485_reset();
    uartRS485.attach( &RS485_rxCallback, Serial::RxIrq );      // call RS485_rx() on every RX interrupt
    
    /*
    #warning TEST
    testcomm();  // UpdateAll
    wait(1.0);
    testcomm();  // UpdateAll
    */
}


void RS485_reset( void )
{
    RS485_handler.ptrRXBuffer = &RS485Buffer[0];
    RS485_handler.bytesToRead = 0;
    RS485_handler.mode = RS485_MODE_LISTEN;
}


// Return 1: Sending was successfull
// Return 0: Could not send data zero bytes
int RS485_sendData( char *ptrDataToSend, int NumBytes )
{
    int i;

    if( NumBytes > 0 ) 
    {
        // Set RS485 driver to sending mode
        RS485RTS = 1;
        
        // Send data
        for( i = 0; i < NumBytes; i++ ) 
        {
            while( !uartRS485.writeable() );
            uartRS485.putc(*ptrDataToSend);
            ptrDataToSend++;
        }//for
        
        // Restart timeout timer
        RS485TimeoutTx.attach_us( &RS485_txTimeout, RS485_TX_TIMEOUT_US );
        
        return 1;
    }//if

    return 0;
}






// Return 1: New data available, get pointer to buffer and number of bytes to read
// Return 0: No new data available
int RS485_receiveData( char *& ptrData, int *ptrNumBytes )
{    
    // Dummy operation to avoid compiler warning
    ptrData++;

    if( RS485_handler.mode != RS485_MODE_RX_READY )
    {    
        *ptrNumBytes = 0;
        return 0;
    }

    *ptrNumBytes = RS485_handler.bytesToRead;
    ptrData = &RS485Buffer[0];
        
    RS485_reset();
    return 1;    
}



//---------------------------------------
// Internal Functions
//---------------------------------------

// Function is called on RX interrupt
void RS485_rxCallback( void )
{
    char tempChar;

    tempChar = uartRS485.getc();

    // Only save data, if not in blocked mode
    if( RS485_handler.mode == RS485_MODE_LISTEN ) {
        if( RS485_handler.bytesToRead < RS485_BUFFER_LENGTH ) { // no buffer overflow
            // Restart timeout timer
            RS485Timeout.attach_us( &RS485_rxTimeout, RS485_RX_TIMEOUT_US );
            // Save received byte
            *RS485_handler.ptrRXBuffer = tempChar;
            RS485_handler.ptrRXBuffer++;
            RS485_handler.bytesToRead++;
        }
    }// if
}


void RS485_txTimeout( void )
{
    // Set RS485 driver to receiving mode
    RS485RTS = 0;
}


void RS485_rxTimeout( void )
{
    // Disable timeout
    RS485Timeout.detach();
    RS485_handler.mode = RS485_MODE_RX_READY;   // Transmission is ready
}


/*
// TEST LCD communication
void testcomm(void)  // UpdateAll
{
        char i;
        unsigned short clacCRC16;
    
        RS485Buffer[0] = 0x00;  // Address 
        RS485Buffer[1] = 0x04;  // Command = UpdateAll 
          
        for (i=0; i < 80; i++)
        {
            RS485Buffer[i+2] = 0x30;
        }

        RS485Buffer[82] = 0x01; // linie 
        RS485Buffer[83] = 0x02; // pos
        RS485Buffer[84] = 0x03; // BlinkingCurserOn
   
     
        RS485Buffer[85] = 0x00; // LED Error
    
        RS485Buffer[86] = 0x00; // high Contrast  
        RS485Buffer[87] = 0x00; // low Contrast
    

        calcCRC16(&RS485Buffer[0], 88, &clacCRC16);
        RS485Buffer[88] = (unsigned char)(0x00FF & (clacCRC16>>8));     // set High Byte
        RS485Buffer[89] = (unsigned char)(0x00FF & clacCRC16);          // set Low Byte  
     
        RS485_sendData( &RS485Buffer[0], 90 );  
} 

*/