/*  MCP23008 library for Arduino
    Copyright (C) 2013 Stefan Jaensch, based on MCP23017 from David Pye    <davidmpye@gmail.com
    Modified for use on the MBED ARM platform

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef     MBED_MCP23008_H
#define     MBED_MCP23008_H

#include    "mbed.h"

//
// Register defines from data sheet - we set IOCON.BANK to 0
// as it is easier to manage the registers sequentially.
//
#define     IODIR       0x00
#define     IPOL        0x01
#define     GPINTEN     0x02
#define     DEFVAL      0x03
#define     INTCON      0x04
#define     IOCON       0x05
#define     GPPU        0x06
#define     INTF        0x07
#define     INTCAP      0x08
#define     GPIO        0x09
#define     OLAT        0x0A

#define     I2C_BASE_ADDRESS    0x40

#define     DIR_OUTPUT      0
#define     DIR_INPUT       1


#define MCP23008_GP0    ( 0x01 )
#define MCP23008_GP1    ( 0x02 )
#define MCP23008_GP2    ( 0x04 )
#define MCP23008_GP3    ( 0x08 )
#define MCP23008_GP4    ( 0x10 )
#define MCP23008_GP5    ( 0x20 )
#define MCP23008_GP6    ( 0x40 )
#define MCP23008_GP7    ( 0x80 )


/** MCP23008 class
 *
 * Allow access to an I2C connected MCP23008 8-bit I/O extender chip
 * Example:
 * @code
 *      MCP23008     *par_port; 
        ...
        par_port = new MCP23008( p9, p10, I2C_ADD);
        par_port->config(0xF0,0,0); 
 * @endcode
 *
 */
class MCP23008 {
public:
    /** Constructor for the MCP23008 connected to specified I2C pins at a specific address
     *
     * 8-bit I/O expander with I2C interface
     *
     * @param   sda         I2C data pin
     * @param   scl         I2C clock pin
     * @param   i2cAddress  I2C address
     */
    MCP23008(PinName sda, PinName scl, int i2cAddress);

    /** Reset MCP23008 device to its power-on state
     */    
    void reset(void);

    /** Configure an MCP23008 device
     *
     * @param   dir_config         data direction value (1 = input, 0 = output)
     * @param   pullup_config      100k pullup value (1 = enabled, 0 = disabled)
     * @param   polarity_config    polarity value (1 = flip, 0 = normal)
     */           
    void config(unsigned char dir_config, unsigned char pullup_config,  unsigned char polarity_config);
      
    /** Write a masked 16-bit value to the device
     *
     * @param   data    8-bit data value
     * @param   mask    8-bit mask value
     */       
    void write_mask(unsigned char data, unsigned char mask);
    
    /** Read a 8-bit value from the device and apply mask
     *
     * @param   mask    8-bit mask value
     * @return          8-bit data with mask applied
     */     
    int  read_mask(unsigned char mask);

    /** Write a 0/1 value to an output bit
     *
     * @param   value         0 or 1
     * @param   bit_number    bit number range 0 --> 7
     */   
    void write_bit(int value, int bit_number);

    /** Read a 0/1 value from an input bit
     *
     * @param   bit_number    bit number range 0 --> 7
     * @return                0/1 value read
     */       
    int  read_bit(int bit_number);
    

    

protected:
    void writeRegister(int regAddress, unsigned char data);
    unsigned char  readRegister(int regAddress);
    
    
    I2C     _i2c;
    int     MCP23008_i2cAddress;                        // physical I2C address
    unsigned char   shadow_GPIO, shadow_IODIR, shadow_GPPU, shadow_IPOL;     // Cached copies of the register values    
};

#endif