#include "mbed.h"
#include "ble/BLE.h"
#include "SEGGER_RTT.h"
#include "BME280_SPI.h"

//For float values make sure you create a wrapper class to handle the exponent and mantissa properly
struct ValueBytes {
    static const unsigned OFFSET_OF_FLAGS    = 0;
    static const unsigned OFFSET_OF_VALUE    = OFFSET_OF_FLAGS + sizeof(uint8_t);
    static const unsigned SIZEOF_VALUE_BYTES = sizeof(uint8_t) + sizeof(float);

    static const unsigned UNITS_FLAG_POS = 0;
    static const unsigned TIMESTAMP_FLAG_POS         = 1;
    static const unsigned TYPE_FLAG_POS  = 2;

    static const uint8_t  UNITS_UNIT1    = 0; //add as many possible units that the value byte can hold
    static const uint8_t  UNITS_UNIT2 = 1;

    ValueBytes(float initialValue) : bytes() {
        /* Assumption: values are expressed in some units of mesurement */
        bytes[OFFSET_OF_FLAGS] =  (UNITS_UNIT1 << UNITS_FLAG_POS) |
                                  (false << TIMESTAMP_FLAG_POS) |
                                  (false << TYPE_FLAG_POS);
        updateValue(initialValue);
    }

    void updateValue(float temp) {
        uint32_t temp_ieee11073 = quick_ieee11073_from_float(temp);
        memcpy(&bytes[OFFSET_OF_VALUE], &temp_ieee11073, sizeof(float));
    }

    uint8_t       *getPointer(void) {
        return bytes;
    }

    const uint8_t *getPointer(void) const {
        return bytes;
    }

private:
    /**
     * @brief A very quick conversion between a float vale and 11073-20601 FLOAT-Type.
     * @param value The sensor value as a float.
     * @return The value in 11073-20601 FLOAT-Type format.
     */
    uint32_t quick_ieee11073_from_float(float value) {
        uint8_t  exponent = 0xFE; //Exponent is -2
        uint32_t mantissa = (uint32_t)(value * 100);

        return (((uint32_t)exponent) << 24) | mantissa;
    }

private:
    /* First byte: 8-bit flags. Second field is a float holding the value. */
    /* See https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.temperature_measurement.xml */
    uint8_t bytes[SIZEOF_VALUE_BYTES];
};

ValueBytes valueBytes(0.0f);
ReadOnlyGattCharacteristic<ValueBytes>  tempMeasurement(GattCharacteristic::UUID_TEMPERATURE_MEASUREMENT_CHAR, (ValueBytes *)valueBytes.getPointer(), GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY); //characteristics that can be used to store a float value

/**Function to update the BLE characteristics value that holds a float value
* @param value the float value to update
*/
void updateCharacteristic(float value)
{
    if (BLE::Instance(BLE::DEFAULT_INSTANCE).getGapState().connected) {
        valueBytes.updateValue(value);
        BLE::Instance(BLE::DEFAULT_INSTANCE).gattServer().write(tempMeasurement.getValueHandle(), valueBytes.getPointer(), sizeof(ValueBytes));
    }
}




DigitalOut led(LED1, 1);
uint16_t customServiceUUID  = 0xA000;
uint16_t readCharUUID       = 0xA001;
uint16_t writeCharUUID      = 0xA002;

const static char     DEVICE_NAME[]        = "iTracker"; // change this
static const uint16_t uuid16_list[]        = {0xFFFF}; //Custom UUID, FFFF is reserved for development


/* Set Up custom Characteristics */
static uint8_t readValue[10] = {0};
ReadOnlyArrayGattCharacteristic<uint8_t, sizeof(readValue)> readChar(readCharUUID, readValue);

static uint8_t writeValue[10] = {0};
WriteOnlyArrayGattCharacteristic<uint8_t, sizeof(writeValue)> writeChar(writeCharUUID, writeValue);

/* Set up custom service */
GattCharacteristic *characteristics[] = {&readChar, &writeChar};
GattService        customService(customServiceUUID, characteristics, sizeof(characteristics) / sizeof(GattCharacteristic *));


/*
 *  Restart advertising when phone app disconnects
*/
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *)
{
    BLE::Instance(BLE::DEFAULT_INSTANCE).gap().startAdvertising();
}

/*
 *  Handle writes to writeCharacteristic
*/
void writeCharCallback(const GattWriteCallbackParams *params)
{
    /* Check to see what characteristic was written, by handle */
    if(params->handle == writeChar.getValueHandle()) {
        /* toggle LED if only 1 byte is written */
        if(params->len == 1) {
            //led = params->data[0];
            (params->data[0] == 0x00) ? SEGGER_RTT_printf(0, "led on\n\r") : SEGGER_RTT_printf(0, "led off\n\r"); // print led toggle
        }
        /* Print the data if more than 1 byte is written */
        else {
            SEGGER_RTT_printf(0, "Data received: length = %d, data = 0x",params->len);
            for(int x=0; x < params->len; x++) {
                SEGGER_RTT_printf(0, "%x", params->data[x]);
            }
            SEGGER_RTT_printf(0, "\n\r");
        }
        /* Update the readChar with the value of writeChar */
        BLE::Instance(BLE::DEFAULT_INSTANCE).gattServer().write(readChar.getValueHandle(), params->data, params->len);
    }
}
/*
 * Initialization callback
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE &ble          = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);
    ble.gattServer().onDataWritten(writeCharCallback);

    /* Setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE); // BLE only, no classic BT
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED); // advertising type
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME)); // add name
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list)); // UUID's broadcast in advertising packet
    ble.gap().setAdvertisingInterval(100); // 100ms.

    /* Add our custom service */
    ble.addService(customService);

    /* Start advertising */
    ble.gap().startAdvertising();
}

/*
 *  Main loop
*/
int main(void)
{
    /* initialize stuff */
    SEGGER_RTT_printf(0, "\n\r********* Starting Main Loop *********\n\r");

    BLE& ble = BLE::Instance(BLE::DEFAULT_INSTANCE);
    ble.init(bleInitComplete);

    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) {
        /* spin loop */
    }

    BME280_SPI sensor(p3, p5, p4, p2); // mosi, miso, sclk, cs

    /*while (true) {
       SEGGER_RTT_printf(0, "%2.2f degC, %04.2f hPa, %2.2f %%\n", sensor.getTemperature(), sensor.getPressure(), sensor.getHumidity());
       wait(5.0);
    }*/

    /* Infinite loop waiting for BLE interrupt events */
    while (true) {
        ble.waitForEvent(); /* Save power */
    }
}


