/* 
    Created by Naresh Krishnamoorthy
  
  The LIS2MDL is a low power magnetometer, here used as 3 DoF solution.
  Library may be used freely and without limit with attribution.
*/

#ifndef LIS2MDL_h
#define LIS2MDL_h

#include "mbed.h"

//*********************************
//Register map for LIS2MDL'
// http://www.st.com/content/ccc/resource/technical/document/datasheet/group3/29/13/d1/e0/9a/4d/4f/30/DM00395193/files/DM00395193.pdf/jcr:content/translations/en.DM00395193.pdf
//*********************************
#define LIS2MDL_OFFSET_X_REG_L        0x45
#define LIS2MDL_OFFSET_X_REG_L        0x46
#define LIS2MDL_OFFSET_X_REG_L        0x47
#define LIS2MDL_OFFSET_X_REG_L        0x48
#define LIS2MDL_OFFSET_X_REG_L        0x49
#define LIS2MDL_OFFSET_X_REG_L        0x4A
#define LIS2MDL_WHO_AM_I              0x4F
#define LIS2MDL_CFG_REG_A             0x60
#define LIS2MDL_CFG_REG_B             0x61
#define LIS2MDL_CFG_REG_C             0x62
#define LIS2MDL_INT_CTRL_REG          0x63
#define LIS2MDL_INT_SOURCE_REG        0x64
#define LIS2MDL_INT_THS_L_REG         0x65
#define LIS2MDL_INT_THS_H_REG         0x66
#define LIS2MDL_STATUS_REG            0x67
#define LIS2MDL_OUTX_L_REG            0x68
#define LIS2MDL_OUTX_H_REG            0x69
#define LIS2MDL_OUTY_L_REG            0x6A
#define LIS2MDL_OUTY_H_REG            0x6B
#define LIS2MDL_OUTZ_L_REG            0x6C
#define LIS2MDL_OUTZ_H_REG            0x6D
#define LIS2MDL_TEMP_OUT_L_REG        0x6E
#define LIS2MDL_TEMP_OUT_H_REG        0x6F

#define LIS2MDL_ADDRESS               (0x1E << 1)

//******************************
// MODR legal values
//******************************
#define MODR_10Hz   0x00
#define MODR_20Hz   0x01
#define MODR_50Hz   0x02
#define MODR_100Hz  0x03

/** LIS2MDL class.
 *  Used for interfacing with the LIS2MDL sensor on board the itracker
 */
class LIS2MDL
{
public:
    /**Public constructor
     * @param p_i2c Mbed I2C class object
     * @param addr Address of the I2C object  
     *  
    */
    LIS2MDL(I2C& p_i2c, uint8_t addr);
    
    /** init function to set the sensors initialisation parameters
    *  @param  MODR See the MODR legal values in the defines in LIS2MDL.h
    *
    */
    void init(uint8_t MODR);
    
    /** Function to get the CHIP ID
    *
    *  @return uin8_t returns the chip id. In this can 64
    *  see  http://www.st.com/content/ccc/resource/technical/document/datasheet/group3/29/13/d1/e0/9a/4d/4f/30/DM00395193/files/DM00395193.pdf/jcr:content/translations/en.DM00395193.pdf
    */
    uint8_t getChipID();
    
    /** Read the raw sensor data
    *   @params destination pointer to the array that will store the results
    *   see http://www.st.com/content/ccc/resource/technical/document/datasheet/group3/29/13/d1/e0/9a/4d/4f/30/DM00395193/files/DM00395193.pdf/jcr:content/translations/en.DM00395193.pdf
    */
    void readData(int16_t * destination);
    
    /** Function to get the status register value
    *   
    *   @return uint8_t value of the status register
    *   see http://www.st.com/content/ccc/resource/technical/document/datasheet/group3/29/13/d1/e0/9a/4d/4f/30/DM00395193/files/DM00395193.pdf/jcr:content/translations/en.DM00395193.pdf
    */
    uint8_t status();
    
    /** Function to reset the LIS2MDL sensor
    */
    void reset();
    
    /** Function to generate the offset bias stored in the chip as part of the calib 
    *   @param dest1 Magnetic Bias offset of the sensor
    *   @param dest2 Magnetic Scale offset of the sensor    
    */
    void offsetBias(float * dest1, float * dest2);
    
    /** Function to read the temperature of the internal tempo sensor
    *
    *   @return uint8_t temperature reading of the internal temp sensor
    */
    int16_t readTemperature();
    
    /** Self check function for the sensor
    *
    */
    void lis2mdlSelfCheck();

    /** I2C function for writing a Byte to the LIS2MDL sensor
    *   @param address address of the sensor
    *   @param subaddress register location to which to write data
    *   @param data data to be written 
    */
    void writeByte(uint8_t address, uint8_t subAddress, uint8_t data);
    
    /** I2C function for reading a Byte from the LIS2MDL sensor
    *   @param address address of the sensor
    *   @param subaddress register location to which to write data
    */
    uint8_t readByte(uint8_t address, char subAddress);
    
    /** I2C function for reading many Bytes from the LIS2MDL sensor
    *   @param address address of the sensor
    *   @param subaddress register location to which to write data
    *   @param count number of bytes to read
    *   @param dest pointer to the array which will store the read values    
    */
    void readBytes(uint8_t address, uint8_t subAddress, uint8_t count, char * dest);

protected:
    I2C *_i2c_p;
    I2C &_i2c;

};

#endif