/**
 * Copyright (c) 2015 Digi International Inc.,
 * All rights not expressly granted are reserved.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Digi International Inc. 11001 Bren Road East, Minnetonka, MN 55343
 * =======================================================================
 */

#if !defined(__AT_CMD_FRAME_H_)
#define __AT_CMD_FRAME_H_

#include "ApiFrame.h"

#define REM_AT_CMD_RESP_FRM_ID_OFFSET   0
#define REM_AT_CMD_RESP_STATUS_OFFSET   13
#define REM_AT_CMD_RESP_CMD_DATA_OFFSET 14
#define REM_AT_CMD_RESP_OVERHEAD        14 /* ID + ADDR64 + ADDR16 + CMD + status */

#define ATCMD_RESP_FRAME_ID_OFFSET      0
#define ATCMD_RESP_CMD_LOW_OFFSET       1
#define ATCMD_RESP_CMD_HIGH_OFFSET      2
#define ATCMD_RESP_STATUS_OFFSET        3
#define ATCMD_RESP_DATA_OFFSET          4
#define ATCMD_RESP_NW_ADDR_H_OFFSET     4
#define ATCMD_RESP_NW_ADDR_L_OFFSET     5
#define ATCMD_RESP_SH_ADDR_L_OFFSET     6
#define ATCMD_RESP_SH_ADDR_H_OFFSET     10
#define ATCMD_RESP_NI_OFFSET            14

#define ATCMD_RESP_OVERHEAD             4 /* ID + CMD + status */

#define ATCMD_802_RESP_SIGN_STR_OFFSET  14
#define ATCMD_802_RESP_NI_OFFSET        15

#define MAX_NI_PARAM_LEN                20

/** Class for the AT command api frames. Derived from ApiFrame */
class AtCmdFrame : public ApiFrame
{
    public:

        /**
         * AtCmdResp
         */
        enum AtCmdResp {
            AtCmdRespOk             = 0,     /**< Ok */
            AtCmdRespError          = 1,     /**< Error */
            AtCmdRespInvalidCmd     = 2,     /**< Invalid Command */
            AtCmdRespInvalidParam   = 3,     /**< Invalid Parameter */
            AtCmdRespTxFailure      = 4,     /**< Tx Failure */
            AtCmdRespLenMismatch    = 0xfd,  /**< Length Mismatch (Error generated by the library) */
            AtCmdRespInvalidAddr    = 0xfe,  /**< Invalid Address (Error generated by the library) */
            AtCmdRespTimeout        = 0xff,   /**< Timeout (Error generated by the library) */
        };

        /**
         * ModemStatus
         */
        enum ModemStatus {
            HwReset         = 0,     /**< Hardware reset */
            WdReset         = 1,     /**< Watchdog timer reset */
            JoinedNW        = 2,     /**< Joined network (routers and end devices) */
            Disassociated   = 3,     /**< Disassociated */
            SyncLost        = 4,     /**< Synchronization Lost */
            CoordRealign    = 5,     /**< Coordinator realignment */
            CoordStarted    = 6,     /**< Coordinator started */
            NwSecKeyUpdated = 7,     /**< Network security key was updated */
            NwWokeUp        = 0x0B,  /**< NwWokeUp */
            NwToSleep       = 0x0C,  /**< NwToSleep */
            VccExceeded     = 0x0D,  /**< VccExceeded: PRO S2B only? */
            ModConfChangeJoinInProg = 0x11, /**< Modem configuration changed while join in progress */
        };

        /** Class constructor
         *
         * @param cmd at command of the frame
         * @param cmd_param command parameter
         * @param param_len length of the command param
         */
        AtCmdFrame(const char * const cmd, const uint32_t cmd_param);

        /** Class constructor
         *
         * @param cmd at command of the frame
         * @param cmd_param pointer to command parameter
         * @param param_len length of the command param
         */
        AtCmdFrame(const char * const cmd, const uint8_t * cmd_param = NULL, uint16_t param_len = 0);

        /** Class constructor
         *
         * @param remote 64 bit address of the remote device where we want to run the command
         * @param cmd at command of the frame
         * @param cmd_param command parameter
         */
        AtCmdFrame(uint64_t remote, const char * const cmd, uint32_t cmd_param);

        /** Class constructor
         *
         * @param remote 64 bit address of the remote device where we want to run the command
         * @param cmd at command of the frame
         * @param cmd_param pointer to command parameter
         * @param param_len length of the command param
         */
        AtCmdFrame(uint64_t remote, const char * const cmd, const uint8_t * cmd_param = NULL, uint16_t param_len = 0);

        /** Class constructor
         *
         * @param remote 16 bit address of the remote device where we want to run the command
         * @param cmd at command of the frame
         * @param cmd_param command parameter
         */
        AtCmdFrame(uint16_t remote, const char * const cmd, uint32_t cmd_param);

        /** Class constructor
         *
         * @param remote 16 bit address of the remote device where we want to run the command
         * @param cmd at command of the frame
         * @param cmd_param pointer to command parameter
         * @param param_len length of the command param
         */
        AtCmdFrame(uint16_t remote, const char * const cmd, const uint8_t * cmd_param = NULL, uint16_t param_len = 0);

        /** Class constructor
         *
         * @param remote 64 bit address of the remote device where we want to run the command
         * @param remote 16 bit address of the remote device where we want to run the command
         * @param cmd at command of the frame
         * @param cmd_param command parameter
         */
        AtCmdFrame(uint64_t remote64, uint16_t remote16, const char * const cmd, uint32_t cmd_param);

        /** Class constructor
         *
         * @param remote 64 bit address of the remote device where we want to run the command
         * @param remote 16 bit address of the remote device where we want to run the command
         * @param cmd at command of the frame
         * @param cmd_param pointer to command parameter
         * @param param_len length of the command param
         */
        AtCmdFrame(uint64_t remote64, uint16_t remote16, const char * const cmd, const uint8_t * cmd_param = NULL, uint16_t param_len = 0);

    protected:
        /** build_at_cmd_frame method used by the constructors to create the at command frame
         *
         * @param cmd at command of the frame
         * @param cmd_params pointer to command parameter
         * @param param_len length of the command param
         */
        void build_at_cmd_frame(const char *cmd, const uint8_t *cmd_params, uint8_t payload_len, bool reverse = true);

        /** build_at_cmd_remote_frame method used by the constructors to create the at command frame
         *
         * @param remote64 64 bit address of the remote device where we want to run the command
         * @param remote16 16 bit address of the remote device where we want to run the command
         * @param cmd at command of the frame
         * @param cmd_params pointer to command parameter
         * @param param_len length of the command param
         */
        void build_at_cmd_remote_frame(uint64_t remote64, uint16_t remote16,
                const char *const cmd, const uint8_t *const cmd_params, uint8_t payload_len, bool reverse = true);
};

#endif /* __AT_CMD_FRAME_H_ */
